﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_SdkThread.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

#include "testNfp_MenuRender.h"
#include "testNfp_TagAccessor.h"

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    #define PRINT_VARIABLE(a,f) NNT_PRINT_LOG(#a " == " f "\n",(a))

    // タグ検知スレッドと関連パラメータです。
    nn::os::SystemEventType g_ActivateEvent;
    nn::os::SystemEventType g_DeactivateEvent;
    nn::os::SystemEventType g_AvailabilityChangeEvent;
    nn::os::ThreadType      g_TagDetectionThread;
    nn::os::ThreadType      g_AvailabilityCheckThread;
    bool                    g_IsRunning = false;
    bool                    g_IsRunningForAvailabilityCheckThread = false;
    bool                    g_IsWriteOrderAscending  = false;
    bool                    g_IsInitialized = false;
    const int               TagDetectionThreadStackSize = 4 * 1024;
    NN_ALIGNAS( 4096 ) char g_StackBuffer[ TagDetectionThreadStackSize ];
    const int               AvailabilityCheckThreadStackSize = 4 * 1024;
    NN_ALIGNAS( 4096 ) char g_StackBufferForAvailabilityCheckThread[ AvailabilityCheckThreadStackSize ];

    // 検出したタグの情報です。
    TagData                 g_TagData;

#if !defined(NNT_NFP_PLATFORM_NX) // NX には別の amiibo 設定起動 I/F が用意されました。
    nn::nfp::Parameter      g_Parameter;
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // 上記のタグ情報を保護するための同期オブジェクトです。
    nn::os::Mutex g_Mutex(false);

    //==================================================
    //! @brief      バイトオーダを変換します。
    //! @param[in]  value  対象の値です。
    //! @result     バイトオーダを変換した値です。
    //==================================================
    inline uint32_t ReverseEndian32(uint32_t value) NN_NOEXCEPT
    {
        return (value << 24) | ((value << 8) & 0x00FF0000) |
              ((value >> 8) & 0x0000FF00) | (value >> 24);
    }

    //==================================================
    //! @brief      タグ情報をログ出力します。
    //! @param[in]  tagData  タグ情報です。
    //==================================================
    inline void PrintTagData(TagData tagData) NN_NOEXCEPT
    {
        NNT_PRINT_LOG("Print tagData \n");
        const int NicknameCharacterSize = sizeof(tagData.registerInfo.nickname[0]);
        const int NicknameDataMaxLength =
                sizeof(tagData.registerInfo.nickname) / NicknameCharacterSize;

        // (1文字表示するのに必要な文字長 ＋カンマ) * 文字列長(終端込み)
        static char nicknameLogBuffer[((NicknameCharacterSize * 2) + 1) * NicknameDataMaxLength];

        PRINT_VARIABLE(tagData.adminInfo.moveCounter,"%d");
        PRINT_VARIABLE(tagData.registerInfo.fontRegion,"%d");
#if !defined(NNT_NFP_PLATFORM_NX) //NX には国コードがない
        PRINT_VARIABLE(tagData.registerInfo.country,"%d");
#endif //!defined(NNT_NFP_PLATFORM_NX)
        std::memset(nicknameLogBuffer,0x00,sizeof(nicknameLogBuffer));
        for(int i = 0;i < NicknameDataMaxLength;i++)
        {
            // 1文字表示するのに必要な文字長 + カンマ + 終端文字
            char tmp[(NicknameCharacterSize * 2) + 1 + 1] = {};
            std::sprintf(tmp,"%02x,",tagData.registerInfo.nickname[i]);
            std::strncat(nicknameLogBuffer,tmp,std::strlen(tmp));
        }
        NNT_PRINT_LOG("tagData.registerInfo.nickName == {%s}\n",nicknameLogBuffer);

        const int DumpLengthPerLine = 32;
        // (1byte表示するのに必要な文字長 ＋カンマ) * 1行に表示するデータ数 + 終端文字
        static char applicationAreaLogBuffer[(2 + 1) * DumpLengthPerLine + 1];

        std::memset(applicationAreaLogBuffer,0x00,sizeof(applicationAreaLogBuffer));
        NNT_PRINT_LOG("tagData.applicationArea == {\n");
        int len = sizeof(tagData.applicationArea) / sizeof(tagData.applicationArea[0]);
        for(int i = 0;i < len;i++)
        {
            // 1byte表示するのに必要な文字長 ＋カンマ + 終端文字
            char tmp[2 + 1 + 1] = {};
            std::sprintf(tmp,"%02x,",tagData.applicationArea[i]);
            std::strncat(applicationAreaLogBuffer,tmp,std::strlen(tmp));

            if(((i + 1) % DumpLengthPerLine) == 0 || i == (len - 1))
            {
                NNT_PRINT_LOG("%s\n",applicationAreaLogBuffer);
                std::memset(applicationAreaLogBuffer,0x00,sizeof(applicationAreaLogBuffer));
            }
        }
        NNT_PRINT_LOG("}\n");
    }

    //==================================================
    //! @brief      タグをマウントします。
    //! @result     実行結果です。
    //==================================================
    nn::Result NfpMount() NN_NOEXCEPT
    {
        nn::Result result = nnt::nfp::wrapper::Mount();
        g_Mutex.Lock();
        if (result.IsSuccess())
        {
            g_TagData.state = TagState_Good;
        }
        else if (nn::nfp::ResultNotSupported().Includes(result))
        {
            g_TagData.state = TagState_NotSupported;
            NNT_PRINT_LOG("===== NOT_SUPPORTED_TAG_INFO =====\n");
            nn::nfp::TagInfo tagInfo;
            if(nnt::nfp::wrapper::GetTagInfo(&tagInfo).IsFailure())
            {
                NNT_PRINT_LOG(" Faild to Read TagInfo.\n");
                NNT_PRINT_LOG("  Module=%d Description=%d \n",result.GetModule(),
                       result.GetDescription());
                g_Mutex.Unlock();
                return result;
            }
            NNT_PRINT_LOG(" UID:");
            for ( uint32_t i = 0; i < tagInfo.tagId.length; i++ )
            {
                NNT_PRINT_LOG_NO_PREFIX( " %02X", tagInfo.tagId.uid[i] );
            }
            NNT_PRINT_LOG_NO_PREFIX("\n");
            NNT_PRINT_LOG(" UID Length: %d\n", tagInfo.tagId.length);
            NNT_PRINT_LOG(" Protocol: %d\n", tagInfo.protocol);
            NNT_PRINT_LOG(" Type: %d\n", tagInfo.type);
        }
        else if (nn::nfp::ResultInvalidFormatVersion().Includes(result))
        {
            g_TagData.state = TagState_InvalidFormatVersion;
        }
        else if (nn::nfp::ResultNeedRestore().Includes(result))
        {
            g_TagData.state = TagState_NeedRestore;
        }
        else if (nn::nfp::ResultNeedFormat().Includes(result))
        {
            g_TagData.state = TagState_NeedFormat;
        }
        else
        {
            g_TagData.state = TagState_None;
        }
        g_Mutex.Unlock();
        return result;
    }

    //==================================================
    //! @brief      タグ情報を取得します。
    //! @param[out] pOutTagData タグ情報の出力先です。
    //! @result     実行結果です。
    //==================================================
    nn::Result NfpGetTagInfo(TagData *pOutTagData) NN_NOEXCEPT
    {
        // 本来、この API はマウント前であっても実行できます。
        nn::Result result = nnt::nfp::wrapper::GetTagInfo( &pOutTagData->tagInfo );
        pOutTagData->hasTagInfo = result.IsSuccess();
        if(result.IsSuccess() == false)
        {
            NNT_PRINT_LOG("GetTagInfo() Result Failure:  Module=%d Description=%d\n",
                          result.GetModule(),result.GetDescription());
        }
        return result;
    }

    //==================================================
    //! @brief      タグの共用領域の情報を取得します。
    //! @param[out] pOutTagData タグ情報の出力先です。
    //! @result     実行結果です。
    //==================================================
    nn::Result NfpGetCommonInfo(TagData *pOutTagData) NN_NOEXCEPT
    {
        nn::Result result = nnt::nfp::wrapper::GetCommonInfo( &pOutTagData->commonInfo );
        pOutTagData->hasCommonInfo = result.IsSuccess();
        if(result.IsSuccess() == false)
        {
            NNT_PRINT_LOG("GetCommonInfo() Result Failure:  Module=%d Description=%d\n",
                          result.GetModule(),result.GetDescription());
        }
        return result;
    }

    //==================================================
    //! @brief      タグの共用領域の初期登録情報を取得します。
    //! @param[out] pOutTagData タグ情報の出力先です。
    //! @result     実行結果です。
    //==================================================
    nn::Result NfpGetRegisterInfo(TagData *pOutTagData) NN_NOEXCEPT
    {
        nn::Result result = nnt::nfp::wrapper::GetRegisterInfo( &pOutTagData->registerInfo );
        pOutTagData->hasRegisterInfo = result.IsSuccess();
        if(result.IsSuccess() == false)
        {
            NNT_PRINT_LOG("GetRegisterInfo() Result Failure:  Module=%d Description=%d\n",
                          result.GetModule(),result.GetDescription());
        }

        return result;
    }

    //==================================================
    //! @brief      タグの共用領域の管理情報を取得します。
    //! @param[out] pOutTagData タグ情報の出力先です。
    //! @result     実行結果です。
    //==================================================
    nn::Result NfpGetAdminInfo(TagData *pOutTagData) NN_NOEXCEPT
    {
        nn::Result result = nnt::nfp::wrapper::GetAdminInfo( &pOutTagData->adminInfo );
        pOutTagData->hasAdminInfo = result.IsSuccess();
        if(result.IsSuccess() == false)
        {
            NNT_PRINT_LOG("GetAdminInfo() Result Failure:  Module=%d Description=%d\n",
                          result.GetModule(),result.GetDescription());
        }
        return result;
    }

    //==================================================
    //! @brief      タグのアプリケーション専用領域を取得します。
    //! @param[out] pOutTagData タグ情報の出力先です。
    //! @result     実行結果です。
    //==================================================
    nn::Result NfpGetApplicationArea(TagData *pOutTagData) NN_NOEXCEPT
    {
        nn::Result result;

        // AccessID が異なる場合は開けません。
        // 既にオープンされている場合にも ResultSuccess を返します。
        result = nnt::nfp::wrapper::OpenApplicationArea(AccessId);
        if (result.IsFailure())
        {
            NNT_PRINT_LOG("OpenApplicationArea() Result Failure:  Module=%d Description=%d\n",
                              result.GetModule(),result.GetDescription());
            pOutTagData->hasApplicationArea = false;
            return result;
        }

        result = nnt::nfp::wrapper::GetApplicationArea(pOutTagData->applicationArea,
                                              sizeof(pOutTagData->applicationArea));
        if (result.IsFailure())
        {
            pOutTagData->hasApplicationArea = false;
            NNT_PRINT_LOG("GetApplicationArea() Result Failure:  Module=%d Description=%d\n",
                              result.GetModule(),result.GetDescription());
            return result;
        }

        pOutTagData->hasApplicationArea = true;
        return nn::ResultSuccess();
    }

    //==================================================
    //! @brief      指定したタグ情報のSystemWriteCounter値を取得します。
    //! @param[out] pInTagData タグ情報の入力元です。
    //! @result     実行結果です。
    //==================================================
    nn::Result GetSystemWriteCounter(TagData *pInTagData) NN_NOEXCEPT
    {
        nn::nfp::NfpData nfpData;
        nnt::nfp::wrapper::GetAll(&nfpData);
        pInTagData->systemWriteCounter = nfpData.systemInfo.systemWriteCounter;
        return nn::ResultSuccess();
    }

    //==================================================
    //! @brief      タグデータのキャッシュを最新の状態にします。
    //==================================================
    void NfpCacheUpdate() NN_NOEXCEPT
    {
        // タグデータを初期化します。
        TagData tagData;
        std::memset(&tagData, 0, sizeof(TagData));

        // 各種の情報を取得します。この情報取得のソースは NFP ライブラリ内のキャッシュです。
        // Flush を実行しなければタグに書き込まれない情報も反映されています。
        NfpGetTagInfo(&tagData);
        NfpGetCommonInfo(&tagData);
        NfpGetRegisterInfo(&tagData);
        NfpGetAdminInfo(&tagData);
        NfpGetApplicationArea(&tagData);
        GetSystemWriteCounter(&tagData);
        if (tagData.hasCommonInfo)
        {
            tagData.state = TagState_Good;
        }

        PrintTagData(tagData);

        // 結果を反映します。
        {
            g_Mutex.Lock();
            std::memcpy(&g_TagData, &tagData, sizeof(TagData));

            // AmiiboSetting 用の情報をセットします。
#if !defined(NNT_NFP_PLATFORM_NX) // NX には別の amiibo 設定起動 I/F が用意されました。
            std::memcpy(&g_Parameter.input.tagInfo, &g_TagData.tagInfo, sizeof(nn::nfp::TagInfo));
            g_Parameter.input.isRegistered = g_TagData.hasTagInfo;
            std::memcpy(&g_Parameter.input.registerInfo, &g_TagData.registerInfo, sizeof(nn::nfp::RegisterInfo));
            std::memcpy(&g_Parameter.input.commonInfo, &g_TagData.commonInfo, sizeof(nn::nfp::CommonInfo));
#endif // !defined(NNT_NFP_PLATFORM_NX)
            g_Mutex.Unlock();
        }
    }

    //==================================================
    //! @brief      タグデータのキャッシュをクリアします。
    //==================================================
    void NfpCacheClear() NN_NOEXCEPT
    {
        g_Mutex.Lock();
        std::memset(&g_TagData, 0, sizeof(TagData));
        g_Mutex.Unlock();
    }

    //==================================================
    //! @brief      タグが検出されたときには実行する処理です。
    //! @details    自動的にタグをマウントしてデータを読み込みます。
    //==================================================
    void NfpActivate() NN_NOEXCEPT
    {
        TagData tagData = {};
        NfpGetTagInfo(&tagData);

        {
            g_Mutex.Lock();
            std::memcpy(&g_TagData, &tagData, sizeof(TagData));
            g_Mutex.Unlock();
        }
    }

    void NfpDeactivate() NN_NOEXCEPT
    {
        NfpCacheClear();
    }

    void NfpTagDetectionThread(void *pPtr) NN_NOEXCEPT
    {
        NN_UNUSED(pPtr);
        nn::os::MultiWaitType multiWait;
        nn::os::MultiWaitHolderType activeHolder;
        nn::os::MultiWaitHolderType deactiveHolder;

        NNT_PRINT_LOG("*** Start %s \n",NN_CURRENT_FUNCTION_NAME);

        // StopNfp() が実行されるまでタグの発見・喪失を待ち受け続けます。
        const int WaitMilliSeconds = 500;

        while(g_IsRunning)
        {
            if(g_ActivateEvent._state != nn::os::SystemEventType::State_NotInitialized &&
               g_DeactivateEvent._state != nn::os::SystemEventType::State_NotInitialized)
            {
                break;
            }
            // アタッチ・デタッチイベントが初期化されるまで待つ
            nnt::nfp::Sleep(WaitMilliSeconds);
        }

        if(g_IsRunning == false)
        {
            return;
        }

        // 多重待ちオブジェクトリストを構築
        nn::os::InitializeMultiWait( &multiWait );
        nn::os::InitializeMultiWaitHolder(&activeHolder, &g_ActivateEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &activeHolder);
        nn::os::InitializeMultiWaitHolder(&deactiveHolder, &g_DeactivateEvent);
        nn::os::LinkMultiWaitHolder(&multiWait, &deactiveHolder);

        while(g_IsRunning)
        {
            nn::os::MultiWaitHolderType* pSignaledHolder;
            pSignaledHolder = nn::os::TimedWaitAny(&multiWait,
                                                   nn::TimeSpan::FromMilliSeconds(WaitMilliSeconds));

            if (pSignaledHolder == &deactiveHolder)
            {
                NNT_PRINT_LOG("*** Deactivate Event \n");
                NfpDeactivate();
                nn::os::ClearSystemEvent(&g_DeactivateEvent);
            }
            else if (pSignaledHolder == &activeHolder)
            {
                NNT_PRINT_LOG("*** Activate Event \n");
                NfpActivate();
                nn::os::ClearSystemEvent(&g_ActivateEvent);
            }
            else
            {
                g_Mutex.Lock();
                if( g_IsInitialized == true )
                {
                    //イベントが発生しなかった場合はデバイスハンドルが取得できるかチェックする
                    nn::nfp::DeviceHandle deviceHandle;
                    int outCount;
                    nn::Result result = nnt::nfp::wrapper::ListDevices(&deviceHandle,&outCount,1);
                    static nn::Result resultOld = nn::ResultSuccess();

                    if(result.IsFailure())
                    {
                        if(resultOld.IsSuccess())
                        {
                            NNT_PRINT_LOG("*** ListDevices() Failed ... \n");
                        }
                    }
                    else if(outCount == 0)
                    {
                        NNT_PRINT_LOG("*** Device dose not exist ... \n");
                    }

                    resultOld = result;
                }
                g_Mutex.Unlock();
            }
        }

        // リストから外す
        nn::os::UnlinkMultiWaitHolder( &activeHolder );
        nn::os::UnlinkMultiWaitHolder( &deactiveHolder );

        // ファイナライズ
        nn::os::FinalizeMultiWaitHolder( &activeHolder );
        nn::os::FinalizeMultiWaitHolder( &deactiveHolder );
        nn::os::FinalizeMultiWait( &multiWait );
        nn::os::DestroySystemEvent(&g_DeactivateEvent);
        nn::os::DestroySystemEvent(&g_ActivateEvent);

        NNT_PRINT_LOG("*** End %s \n",NN_CURRENT_FUNCTION_NAME);

    }

    void NfpAvailabilityCheckThread(void *pPtr) NN_NOEXCEPT
    {
        NN_UNUSED(pPtr);

        NNT_PRINT_LOG("*** Start %s \n",NN_CURRENT_FUNCTION_NAME);

        const int WaitMilliSeconds = 500;

        while(g_IsRunningForAvailabilityCheckThread)
        {
            bool isReceived = nn::os::TimedWaitSystemEvent(&g_AvailabilityChangeEvent,
                                                           nn::TimeSpan::FromMilliSeconds(WaitMilliSeconds));

            if (isReceived)
            {
                NNT_PRINT_LOG("*** Availability Change Event \n");
                NfpDeactivate();
                nn::os::ClearSystemEvent(&g_AvailabilityChangeEvent);
            }
        }

        nn::os::DestroySystemEvent(&g_AvailabilityChangeEvent);

        NNT_PRINT_LOG("*** End %s \n",NN_CURRENT_FUNCTION_NAME);

    }
}

//==================================================
//! @brief      タグをアンマウントします。
//! @result     実行結果です。
//==================================================
nn::Result  NfpUnmount() NN_NOEXCEPT
{
    return  nnt::nfp::wrapper::Unmount();
}

#if defined(NNT_NFP_LIB_VERSION_BETA)
nn::Result  NfpWifiSwitch() NN_NOEXCEPT
{
    #if !defined(NNT_NFP_PLATFORM_NX)
    return   nn::srv::PublishToSubscriber(nn::srv::NOTIFICATION_WIFI_BUTTON);
    #else // !defined(NNT_NFP_PLATFORM_NX)
    bool nfcEnable = nnt::nfp::IsNfcEnable();
    if (nfcEnable == false)
    {
        // Nfc Off だったら On に変更
        nnt::nfp::WifiOn();
    }
    else
    {
        // Nfc On だったら Off に変更
        nnt::nfp::WifiOff();
    }
    NNT_PRINT_LOG("%s IsNfcEnable = %d \n", NN_CURRENT_FUNCTION_NAME, nnt::nfp::IsNfcEnable());
    return nn::ResultSuccess();
    #endif // !defined(NNT_NFP_PLATFORM_NX)
}
#endif // defined(NNT_NFP_LIB_VERSION_BETA)

#if defined(NNT_NFP_LIB_VERSION_BETA)
nn::Result  NfpStartNicknameAndOwnerSettings() NN_NOEXCEPT
{
    nn::Result result;
    nn::nfp::DeviceHandle deviceHandle;
    bool isRegistered;
    nn::nfp::RegisterInfo registerInfo = {};
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::TagInfo tagInfo = {};
    nn::nfp::RegisterInfo registerInfoSet = {};

    deviceHandle = nnt::nfp::GetCurrentDeviceHandle();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::GetRegisterInfo(&registerInfoSet));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    startParam.optionFlags = nn::nfp::AmiiboSettingsOptionFlags_Default;
    startParam.deviceHandle = deviceHandle;

    result = nnt::nfp::wrapper::StartNicknameAndOwnerSettings(&deviceHandle,
                                                              &isRegistered,
                                                              &registerInfo,
                                                              startParam,
                                                              tagInfo,
                                                              registerInfoSet);

    return result;
}

nn::Result  NfpStartGameDataEraser() NN_NOEXCEPT
{
    nn::Result result;
    nn::nfp::DeviceHandle deviceHandle;
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::TagInfo tagInfo = {};

    deviceHandle = nnt::nfp::GetCurrentDeviceHandle();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    startParam.optionFlags = nn::nfp::AmiiboSettingsOptionFlags_Default;
    startParam.deviceHandle = deviceHandle;

    result = nnt::nfp::wrapper::StartGameDataEraser(&deviceHandle,
                                                    startParam,
                                                    tagInfo);

    return result;
}

nn::Result  NfpStartRestorer() NN_NOEXCEPT
{
    nn::Result result;
    nn::nfp::DeviceHandle deviceHandle;
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::TagInfo tagInfo = {};

    deviceHandle = nnt::nfp::GetCurrentDeviceHandle();
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    startParam.optionFlags = nn::nfp::AmiiboSettingsOptionFlags_Default;
    startParam.deviceHandle = deviceHandle;

    result = nnt::nfp::wrapper::StartRestorer(&deviceHandle,
                                              startParam,
                                              tagInfo);

    return result;
}

nn::Result  NfpStartNicknameAndOwnerSettingsPrivate() NN_NOEXCEPT
{
    nn::Result result;
    nn::nfp::DeviceHandle deviceHandle;
    bool isRegistered;
    nn::nfp::RegisterInfo registerInfo = {};
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::TagInfo tagInfo = {};

    deviceHandle = nnt::nfp::GetCurrentDeviceHandle();
    startParam.optionFlags = nn::nfp::AmiiboSettingsOptionFlags_Default;
    startParam.deviceHandle = deviceHandle;

    result = nnt::nfp::wrapper::StartNicknameAndOwnerSettings(&tagInfo,
                                                              &deviceHandle,
                                                              &isRegistered,
                                                              &registerInfo,
                                                              startParam);

    return result;
}

nn::Result  NfpStartGameDataEraserPrivate() NN_NOEXCEPT
{
    nn::Result result;
    nn::nfp::DeviceHandle deviceHandle;
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::TagInfo tagInfo = {};

    deviceHandle = nnt::nfp::GetCurrentDeviceHandle();
    startParam.optionFlags = nn::nfp::AmiiboSettingsOptionFlags_Default;
    startParam.deviceHandle = deviceHandle;

    result = nnt::nfp::wrapper::StartGameDataEraser(&tagInfo,
                                                    &deviceHandle,
                                                    startParam);

    return result;
}

nn::Result  NfpStartRestorerPrivate() NN_NOEXCEPT
{
    nn::Result result;
    nn::nfp::DeviceHandle deviceHandle;
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::TagInfo tagInfo = {};

    deviceHandle = nnt::nfp::GetCurrentDeviceHandle();
    startParam.optionFlags = nn::nfp::AmiiboSettingsOptionFlags_Default;
    startParam.deviceHandle = deviceHandle;

    result = nnt::nfp::wrapper::StartRestorer(&tagInfo,
                                              &deviceHandle,
                                              startParam);

    return result;
}

nn::Result  NfpStartFormatterPrivate() NN_NOEXCEPT
{
    nn::Result result;
    nn::nfp::DeviceHandle deviceHandle;
    nn::nfp::AmiiboSettingsStartParam startParam = {};
    nn::nfp::TagInfo tagInfo = {};

    deviceHandle = nnt::nfp::GetCurrentDeviceHandle();
    startParam.optionFlags = nn::nfp::AmiiboSettingsOptionFlags_Default;
    startParam.deviceHandle = deviceHandle;

    result = nnt::nfp::wrapper::StartFormatter(&tagInfo,
                                               &deviceHandle,
                                               startParam);

    return result;
}
#endif // defined(NNT_NFP_LIB_VERSION_BETA)

nn::Result CheckBackup() NN_NOEXCEPT
{
    // バックアップデータを格納するために必要なバッファサイズを取得します。
    // このバッファサイズは BACKUP_BUFFER_ALIGNMENT に調整されています。
#if !defined(NNT_NFP_PLATFORM_NX)
    uint32_t requiredSize = nnt::nfp::wrapper::GetBackupSaveDataSize();
#else
    const uint32_t requiredSize = sizeof(nn::nfp::BackupData);
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // バックアップバッファを確保します。 BACKUP_BUFFER_ALIGNMENT の境界に合わせます。
#if !defined(NNT_NFP_PLATFORM_NX)
    uint32_t alignment = nn::nfp::BACKUP_BUFFER_ALIGNMENT;
    uptr nfpHeap = reinterpret_cast<uptr>(new bit8[requiredSize + alignment - 1]);
    void* alignedNfpHeap = reinterpret_cast<void*>((nfpHeap + alignment - 1) & ~(alignment - 1));
#else
    static NN_ALIGNAS(4) nn::Bit8 alignedNfpHeap[requiredSize];
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // バックアップデータを一括して読み込みます。完了まで 1 秒程度かかります。
#if !defined(NNT_NFP_PLATFORM_NX)
    nnt::nfp::wrapper::GetAllBackupSaveData(alignedNfpHeap, requiredSize);
#else
    size_t readSize;
    nnt::nfp::wrapper::ReadBackupData(alignedNfpHeap, &readSize, requiredSize);
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // バックアップデータのヘッダ情報を取得します。
#if !defined(NNT_NFP_PLATFORM_NX)
    nn::nfp::BackupHeaderInfo header;
    nn::nfp::BackupEntryInfo entry;
    nnt::nfp::wrapper::GetBackupHeaderFromMemory(&header, alignedNfpHeap, requiredSize);
#else
    nn::nfp::BackupDataHeader& header = reinterpret_cast<nn::nfp::BackupData*>(alignedNfpHeap)->header;
#endif //!defined(NNT_NFP_PLATFORM_NX)

    for(int i = 0; i < header.entryNum; i++)
    {
#if !defined(NNT_NFP_PLATFORM_NX)
        nnt::nfp::wrapper::GetBackupEntryFromMemory(&entry, 0, alignedNfpHeap, requiredSize);
        NNT_PRINT_LOG("Backup DATE:%04d %02d %02d\n",entry.entryRegisterDate.year, entry.entryRegisterDate.month, entry.entryRegisterDate.day);
#else
        nn::nfp::BackupDataToc& toc = reinterpret_cast<nn::nfp::BackupData*>(alignedNfpHeap)->toc[i];
        NNT_PRINT_LOG("Backup DATE:%04d %02d %02d\n", toc.entryRegisterDate.GetYear(), toc.entryRegisterDate.GetMonth(), toc.entryRegisterDate.GetDay());
#endif
    }

    return nn::ResultSuccess();

}

void StartTagDetectionThread() NN_NOEXCEPT
{
    if( g_IsRunning == false )
    {
        NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(&g_TagDetectionThread,
                                     NfpTagDetectionThread,
                                     nullptr,
                                     g_StackBuffer,
                                     sizeof(g_StackBuffer),
                                     nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_TagDetectionThread);
        g_IsRunning = true;
    }
}

void EndTagDetectionThread() NN_NOEXCEPT
{
    if( g_IsRunning == true )
    {
        //タグ検出スレッドの実行フラグを落としてスレッドの終了を待つ
        g_IsRunning = false;
        nn::os::WaitThread(&g_TagDetectionThread);
        nn::os::DestroyThread(&g_TagDetectionThread);
     }
}

void StartAvailabilityCheckThread() NN_NOEXCEPT
{
    if( g_IsRunningForAvailabilityCheckThread == false )
    {
        NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(&g_AvailabilityCheckThread,
                                     NfpAvailabilityCheckThread,
                                     nullptr,
                                     g_StackBufferForAvailabilityCheckThread,
                                     sizeof(g_StackBufferForAvailabilityCheckThread),
                                     nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_AvailabilityCheckThread);
        g_IsRunningForAvailabilityCheckThread = true;
    }
}

void EndAvailabilityCheckThread() NN_NOEXCEPT
{
    if( g_IsRunningForAvailabilityCheckThread == true )
    {
        //利用可不可チェックスレッドの実行フラグを落としてスレッドの終了を待つ
        g_IsRunningForAvailabilityCheckThread = false;
        nn::os::WaitThread(&g_AvailabilityCheckThread);
        nn::os::DestroyThread(&g_AvailabilityCheckThread);
     }
}

nn::Result NfpInitialize() NN_NOEXCEPT
{
    if (g_IsRunning == true)
    {
        EndTagDetectionThread();
    }
    if (g_IsRunningForAvailabilityCheckThread == true)
    {
        EndAvailabilityCheckThread();
    }
    nn::Result result;

    result = nnt::nfp::wrapper::InitializeDebug();

    if( result.IsSuccess() )
    {
        g_IsInitialized = true;
        StartTagDetectionThread();
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachAvailabilityChangeEvent(&g_AvailabilityChangeEvent));
        StartAvailabilityCheckThread();
        result = nn::ResultSuccess();
    }
    else
    {
        g_IsInitialized = false;
    }

    return result;
}

nn::Result NfpFinalize() NN_NOEXCEPT
{
    g_Mutex.Lock();
    nnt::nfp::wrapper::FinalizeDebug();
    g_IsInitialized = false;
    g_Mutex.Unlock();

    return nn::ResultSuccess();
}

nn::Result NfpListDevices() NN_NOEXCEPT
{
    nn::nfp::DeviceHandle deviceHandle;
    int outCount = -1;

    nn::Result result = nnt::nfp::wrapper::ListDevices(&deviceHandle,&outCount,1);
    NNT_PRINT_LOG("detect %d devices result = %s\n",outCount, nnt::nfp::GetNfpResultTypeString(result));
    nnt::nfp::SetCurrentDeviceHandle(deviceHandle);

    // タグの発見や喪失の通知を受け取るイベントを取得します。イベントの初期化や解放は不要です。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&g_ActivateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&g_DeactivateEvent));

    return result;
}

nn::Result NfpStartDetection() NN_NOEXCEPT
{
    nn::Result result = nnt::nfp::wrapper::StartDetection();
    return result;
}

nn::Result NfpStopDetection() NN_NOEXCEPT
{
    NfpCacheClear();
    nn::Result result = nnt::nfp::wrapper::StopDetection();
    return result;
}

#if defined(NNT_NFP_LIB_VERSION_ALPHA)
nn::Result NfpRestore() NN_NOEXCEPT
{
    // タグをバックアップデータから復元します。
    // この操作は即座にタグに反映され、自動的にタグがマウントされます。
    nn::Result result = nnt::nfp::wrapper::Restore();
    if (result.IsFailure())
    {
        return result;
    }
    NfpCacheUpdate();
    return nn::ResultSuccess();
}
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)

nn::Result NfpFormat() NN_NOEXCEPT
{
    // タグをフォーマットします。この操作は即座にタグに反映されます。
    const nn::Bit8 InitialData[] = { 0 };
    nn::Result result = nnt::nfp::wrapper::Format(InitialData, 0);
    if (result.IsFailure())
    {
        return result;
    }

    // フォーマットに成功していればマウントできるはずです。
    NfpActivate();
    return nn::ResultSuccess();
}

nn::Result NfpCreateApplicationArea() NN_NOEXCEPT
{
    nn::Result result;

    nn::nfp::ApplicationAreaCreateInfo info;

    // アプリケーション専用領域を作成します。このデモではビッグエンディアンでデータを書き込みます。
    // CTR はリトルエンディアンなのでバイトオーダを変換します。
    uint32_t counter     = ReverseEndian32(0);
    info.accessId        = AccessId;
    info.pInitialData    = &counter;
    info.initialDataSize = sizeof(counter);
    result = nnt::nfp::wrapper::CreateApplicationArea(info);
    if (result.IsFailure())
    {
        return result;
    }

    NfpCacheUpdate();
    return nn::ResultSuccess();
}

nn::Result NfpSetApplicationArea() NN_NOEXCEPT
{

    for(nn::Bit8 cnt = 0; cnt < sizeof(g_TagData.applicationArea); cnt++ )
    {
        if(false == g_IsWriteOrderAscending)
        {
            g_TagData.applicationArea[cnt] = (0xFF - cnt);
        }
        else
        {
            g_TagData.applicationArea[cnt] = cnt;
        }
    }
    g_IsWriteOrderAscending  = !g_IsWriteOrderAscending ;


    // NFP ライブラリのキャッシュに書き込みます。
    nn::Result result;
    result = nnt::nfp::wrapper::SetApplicationArea(
        static_cast<void *>(g_TagData.applicationArea),
        sizeof(g_TagData.applicationArea),
        g_TagData.tagInfo.tagId
    );

    if (result.IsFailure())
    {
        return result;
    }

    // NFP ライブラリから取得できるデータはこの時点で書き換えられていますが、
    // 実際にタグに書き込まれるのは Flush の時点です。
    NfpCacheUpdate();
    return nn::ResultSuccess();
}

#if defined(NNT_NFP_PLATFORM_NX)
nn::Result NfpRecreateApplicationArea() NN_NOEXCEPT
{
    static bool flg = false;
    nn::Result result;

    nn::nfp::ApplicationAreaCreateInfo info;

    // アプリケーション専用領域を作成します。このデモではビッグエンディアンでデータを書き込みます。
    // CTR はリトルエンディアンなのでバイトオーダを変換します。
    uint32_t counter     = ReverseEndian32(0);
    info.accessId        = (flg ? AccessId + 1 : AccessId);
    info.pInitialData    = &counter;
    info.initialDataSize = sizeof(counter);
    result = nnt::nfp::wrapper::RecreateApplicationArea(info);
    if (result.IsFailure())
    {
        return result;
    }

    flg = (!flg);

    NfpCacheUpdate();
    return nn::ResultSuccess();
}
#endif // defined(NNT_NFP_PLATFORM_NX)

nn::Result NfpFlush() NN_NOEXCEPT
{
    nn::Result result = nnt::nfp::wrapper::Flush();
    if (result.IsFailure())
    {
        return result;
    }
    NfpCacheUpdate();
    return nn::ResultSuccess();
}

nn::Result NfpDeleteApplicationArea() NN_NOEXCEPT
{
    nn::Result result = nnt::nfp::wrapper::DeleteApplicationArea();
    if (result.IsFailure())
    {
        return result;
    }
    NfpCacheUpdate();
    return nn::ResultSuccess();
}

nn::Result NfpSetRegisterInfo() NN_NOEXCEPT
{
    nn::nfp::RegisterInfoPrivate info;
    nn::Result result;

#if !defined(NNT_NFP_PLATFORM_NX) //NXではテスト用 Mii を使う
    // このデモでは自分の Mii を取得して書き込みます。
    // 自分の Mii が設定されていることを確認してください。
    result = nn::friends::GetMyMii(&info.miiData);
    if (result.IsFailure())
    {
        return result;
    }
#else
    #if defined(NNT_NFP_LIB_MII_ENABLE)
    nnt::nfp::BuildMiiData(info.miiData);
    #endif // defined(NNT_NFP_LIB_MII_ENABLE)
#endif // !defined(NNT_NFP_PLATFORM_NX)

    std::memcpy(info.nickname, nnt::nfp::NormalTagNickName, sizeof(nnt::nfp::NormalTagNickName));
    info.fontRegion  = 0x01;

    // NFP ライブラリのキャッシュに書き込みます。
    result = nnt::nfp::wrapper::SetRegisterInfo(info);
    if (result.IsFailure())
    {
        return result;
    }

    // NFP ライブラリから取得できるデータはこの時点で書き換えられていますが、
    // 実際にタグに書き込まれるのは Flush の時点です。
    NfpCacheUpdate();
    return nn::ResultSuccess();
}

nn::Result NfpSetAnotherRegisterInfo() NN_NOEXCEPT
{
    nn::nfp::RegisterInfoPrivate info;
    nn::Result result;

#if defined(NNT_NFP_LIB_MII_ENABLE)
    nnt::nfp::BuildAnotherMiiData(info.miiData);
#endif // defined(NNT_NFP_LIB_MII_ENABLE)

    std::memcpy(info.nickname, nnt::nfp::NormalTagNickName2, sizeof(nnt::nfp::NormalTagNickName2));
    info.fontRegion  = 0x01;

    // NFP ライブラリのキャッシュに書き込みます。
    result = nnt::nfp::wrapper::SetRegisterInfo(info);
    if (result.IsFailure())
    {
        return result;
    }

    // NFP ライブラリから取得できるデータはこの時点で書き換えられていますが、
    // 実際にタグに書き込まれるのは Flush の時点です。
    NfpCacheUpdate();
    return nn::ResultSuccess();
}

nn::Result NfpDeleteRegisterInfo() NN_NOEXCEPT
{
    nn::Result result = nnt::nfp::wrapper::DeleteRegisterInfo();
    if (result.IsFailure())
    {
        return result;
    }
    NfpCacheUpdate();
    return nn::ResultSuccess();
}

void NfpGetTagData(TagData* pOutTagData) NN_NOEXCEPT
{
    NNT_NFP_ASSERT(pOutTagData != nullptr);
    g_Mutex.Lock();
    std::memcpy(pOutTagData, &g_TagData, sizeof(TagData));
    g_Mutex.Unlock();
}

nn::Result  NfpMountDirect() NN_NOEXCEPT
{
    nn::Result result = NfpMount();
    if (result.IsSuccess())
    {
        NfpCacheUpdate();
    }

    return result;
}

nn::Result  NfpDeleteSystemSaveData() NN_NOEXCEPT
{
#if !defined(NNT_NFP_PLATFORM_NX)
    return  nn::nfp::CTR::DeleteSystemSaveData();
#else
    //NX の NFPライブラリにはシステムセーブデータを削除する機能がない別の方法で削除
    nnt::nfp::DeleteSystemSaveData();
    return nn::ResultSuccess();
#endif //!defined(NNT_NFP_PLATFORM_NX)
}

nn::Result NfpBreakTag() NN_NOEXCEPT
{
    nn::Result result = nnt::nfp::wrapper::BreakTag(nn::nfp::BreakType_Hmac);
    return result;
}
