﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nfp.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testNfp_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    // バックアップデータを読み込むためのバッファです。
    nn::Bit8  g_BackupBuffer[4 * 1024 * 1024];
    nn::Bit8* g_pBackupData;
    const size_t BackupBufferSize = sizeof(g_BackupBuffer) - nnt::nfp::IpcAlignment;

} // end of anonymous namespace

//================================================================================
// 全プラットフォームで共通のテストスイートです。
// 必ずタグを設置してからテストを開始してください。
//================================================================================

class NfpAutoWithoutTag : public nnt::nfp::TestFramework
{
protected:

    NfpAutoWithoutTag() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::nfp::InitializeHidController();
        // アライメントを調整します。
        g_pBackupData = static_cast<nn::Bit8*>(nnt::nfp::AlignBuffer(g_BackupBuffer,
                                                                     nnt::nfp::IpcAlignment));
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        // 次のテストに影響が出ないようにライブラリを一旦終了しておきます。
        if(nnt::nfp::wrapper::GetState() == nn::nfp::State_Init)
        {
            nnt::nfp::FinalizeSystem();
        }
    }
};


//================================================================================
// テストケースの実装です。
//================================================================================

#if !defined(NNT_NFP_PLATFORM_NX)
TEST_F(NfpAutoWithoutTag, TestCaseInvalidOperationNone)
#else
TEST_F(NfpAutoWithoutTag, TestCaseInvalidOperationNoneDeathTest) //NXではアボートを期待
#endif // !defined(NNT_NFP_PLATFORM_NX)
{
    nn::Bit8                           tagData[nn::nfp::ApplicationAreaSizeV2] = {};
#if !defined(NNT_NFP_PLATFORM_NX) || (!defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32))
    //NXでDEATHテストをしない場合に使用しない変数
    nn::Bit8                           buffer[nn::nfp::ApplicationAreaSizeV2]  = {};
    nn::nfp::TagInfo                   tagInfo          = {};
    nn::nfp::RegisterInfo              regInfo          = {};
    nn::nfp::RegisterInfoPrivate       regInfoPrivate   = {};
    nn::nfp::CommonInfo                commonInfo       = {};
    nn::nfp::ModelInfo                 modelInfo        = {};
    nn::nfp::AdminInfo                 adminInfo        = {};
    nn::nfp::RegisterInfoPrivate       regInfoSet       = {};
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo      = { nnt::nfp::ZeroTagId,
                                                            nn::nfp::ApplicationAreaSizeV2,
                                                            tagData };
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
    nn::hid::NpadIdType                npadId;
#endif // !defined(NNT_NFP_PLATFORM_NX) || (!defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32))

    std::memcpy(tagData,nnt::nfp::ZeroTagData,nn::nfp::ApplicationAreaSizeV2);

    // NONE 状態のテストです。
#if !defined(NNT_NFP_PLATFORM_NX) //NXではFinalizeは返り値をもたない
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Finalize());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::FinalizeSystem());
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if defined(NNT_NFP_PLATFORM_NX) //NXで追加されたAPI
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    // デバイスハンドル取得のテストです。
    nn::nfp::DeviceHandle deviceHandle;
    int outCount;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::StopDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::MountRom());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetApplicationArea(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetApplicationArea(buffer,
                                                                    sizeof(buffer),
                                                                    tagInfo.tagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Flush());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Restore());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetModelInfo(&modelInfo));
#else
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartDetection());
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StopDetection());
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::Mount());
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::MountRom());
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::Unmount());
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetApplicationArea(buffer, sizeof(buffer)));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::SetApplicationArea(buffer,
                                                               sizeof(buffer),
                                                               tagInfo.tagId));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::Flush());
        #if defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::Restore());
        #endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetRegisterInfo(&regInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetRegisterInfo(&regInfoPrivate));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetModelInfo(&modelInfo));

    bool outValue = false;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ExistsApplicationArea(&outValue));
    EXPECT_EQ(false, outValue);
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されているので除外
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::InitializeCreateInfo(&appAreaInfo));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#else
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetConnectionStatus(&connectionStatus));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::Format(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
#else
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::Format(buffer, sizeof(buffer)));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではAPIが削除されているので除外
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::DeleteApplicationArea());
    nn::nfp::BackupHeaderInfo          backupHeader     = {};
    nn::nfp::BackupEntryInfo           backupEntry      = {};
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetBackupHeaderFromMemory(&backupHeader, g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetBackupEntryFromMemory(&backupEntry, 0, g_pBackupData, BackupBufferSize));
#else
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::DeleteApplicationArea());
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::GetNpadId(&npadId));
    size_t readBackupDataSize;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &readBackupDataSize, BackupBufferSize));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
    #else
    NN_UNUSED(BackupBufferSize);
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは高速タグ検出は実装されないので除外
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(false));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(true));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_TypeA));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_All));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではアボートを期待 //NX には ReplaceOtherRegionName がない
    nn::Bit16 nickName[11];
    nn::Bit16 dst[nn::nfp::NicknameLengthMax + 1];
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::ReplaceOtherRegionName(dst, nickName, sizeof(dst), nn::nfp::FontRegion_JpUsEu));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) && defined(NNT_NFP_LIB_VERSION_BETA)//NXではアボートを期待
    #if !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)//Releaseの場合は不定
    nn::nfp::AmiiboSettingsStartParam startParam;
    bool isRegistered;
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, &regInfo, startParam, tagInfo, regInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &deviceHandle, &isRegistered, &regInfo, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartGameDataEraser(
            &deviceHandle, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartRestorer(&deviceHandle, startParam, tagInfo));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartNicknameAndOwnerSettings(
            &tagInfo, &deviceHandle, &isRegistered, &regInfo, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartGameDataEraser(
            &tagInfo, &deviceHandle, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartRestorer(&tagInfo, &deviceHandle, startParam));
    NNT_NFP_ASSERT_DEATH(nnt::nfp::wrapper::StartFormatter(&tagInfo, &deviceHandle, startParam));
    #endif // !defined(NN_SDK_BUILD_RELEASE) && defined(NN_BUILD_CONFIG_OS_WIN32)
#endif //defined(NNT_NFP_PLATFORM_NX) && defined(NNT_NFP_LIB_VERSION_BETA)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
    nn::nfp::ConnectionStatus          connectionStatus = {};
    nn::nfp::TargetConnectionStatus    targetStatus;
    nn::os::EventType                  connectEvent;
    nn::os::EventType                  disconnectEvent;
    uint32_t                           sentSize;
    uint32_t                           totalSize;
    nn::Result                         result;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetConnectResult(&result));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Disconnect());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateResult(&result));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
#endif // !defined(NNT_NFP_PLATFORM_NX)
} // NOLINT(impl/function_size)

TEST_F(NfpAutoWithoutTag, TestCaseInvalidOperationInit)
{
    nn::Bit8                           buffer[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::TagInfo                   tagInfo          = {};
    nn::nfp::RegisterInfo              regInfo          = {};
    nn::nfp::CommonInfo                commonInfo       = {};
    nn::nfp::ModelInfo                 modelInfo        = {};
    nn::nfp::AdminInfo                 adminInfo        = {};
    nn::nfp::RegisterInfoPrivate       regInfoSet       = {};
    nn::Bit8                           tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo      = { nnt::nfp::ZeroTagId,
                                                            nn::nfp::ApplicationAreaSizeV2,
                                                            tagData };

    std::memcpy(tagData,nnt::nfp::ZeroTagData,nn::nfp::ApplicationAreaSizeV2);

    // INIT 状態に遷移させます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());

    // INIT 状態のテストです。
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは返り値を持たない
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Initialize());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::InitializeSystem());
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound, nnt::nfp::wrapper::StopDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound, nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound, nnt::nfp::wrapper::MountRom());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound, nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::GetApplicationArea(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::SetApplicationArea(buffer,
                                                                    sizeof(buffer),
                                                                    tagInfo.tagId));
#if defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
    bool outValue = false;
    NNT_EXPECT_RESULT_FAILURE(
        nn::nfp::ResultNfcDeviceNotFound,
        nnt::nfp::wrapper::ExistsApplicationArea(&outValue));
    EXPECT_EQ(false, outValue);
#endif // (NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound, nnt::nfp::wrapper::Flush());
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound, nnt::nfp::wrapper::Restore());
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::GetModelInfo(&modelInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::Format(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::ResultNfcDeviceNotFound,
                              nnt::nfp::wrapper::DeleteApplicationArea());

#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnect・FWアップデートは実装されないので除外
    // 以降は CTR 専用のテストなので SNAKE のテストはここで終了です。
    if (nn::os::IsRunOnSnake())
    {
        return;
    }

    nn::os::EventType                      connectEvent;
    nn::os::EventType                      disconnectEvent;
    uint32_t                               sentSize;
    uint32_t                               totalSize;
    // INIT 状態 (接続前) のテストです。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Disconnect());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));

    // INIT 状態 (接続試行中) に遷移させます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());

    // INIT 状態 (接続試行中) のテストです。接続が一瞬では終わらないことを期待しています。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));

    // INIT 状態 (接続済) に遷移させます。
    EXPECT_EQ(false, nnt::nfp::WaitForConnect(0));
    EXPECT_EQ(true,  nnt::nfp::WaitForConnect());

    // INIT 状態 (接続済) のテストです。
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));

    // INIT 状態 (更新用接続試行中) に遷移させます。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    // NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());

    // INIT 状態 (更新用接続試行中) のテストです。
    // NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
    // NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
    // NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
    // NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));

    // INIT 状態 (更新用接続済) に遷移させます。
    // EXPECT_EQ(false, nnt::nfp::WaitForConnect(0));
    // EXPECT_EQ(true,  nnt::nfp::WaitForConnect());

    // INIT 状態 (更新用接続済) のテストです。
    // NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
    // NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
    // NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
#endif // !defined(NNT_NFP_PLATFORM_NX)
}

TEST_F(NfpAutoWithoutTag, TestCaseInvalidOperationSearch)
{
    nn::Bit8                           buffer[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::TagInfo                   tagInfo          = {};
    nn::nfp::RegisterInfo              regInfo          = {};
    nn::nfp::CommonInfo                commonInfo       = {};
    nn::nfp::ModelInfo                 modelInfo        = {};
    nn::nfp::AdminInfo                 adminInfo        = {};
    nn::nfp::RegisterInfoPrivate       regInfoSet       = {};
    nn::Bit8                           tagData[nn::nfp::ApplicationAreaSizeV2] = {};
    nn::nfp::ApplicationAreaCreateInfo appAreaInfo      = { nnt::nfp::ZeroTagId,
                                                            nn::nfp::ApplicationAreaSizeV2,
                                                            tagData };

    std::memcpy(tagData,nnt::nfp::ZeroTagData,nn::nfp::ApplicationAreaSizeV2);

    // RW_SEARCH 状態まで遷移させます。
    nnt::nfp::DoSearch();

    // RW_SEARCH 状態のテストです
#if !defined(NNT_NFP_PLATFORM_NX) //NXでは返り値を持たない
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Initialize());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::InitializeSystem());
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::StartDetection());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Mount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::MountRom());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Unmount());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::OpenApplicationArea(nnt::nfp::ZeroTagId));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::CreateApplicationArea(appAreaInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetApplicationArea(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetApplicationArea(buffer,
                                                                    sizeof(buffer),
                                                                    tagInfo.tagId));
#if defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::RecreateApplicationArea(appAreaInfo));
#endif // (NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Flush());
#if defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Restore());
#endif // defined(NNT_NFP_LIB_VERSION_ALPHA)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetTagInfo(&tagInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::GetRegisterInfo(&regInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetCommonInfo(&commonInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetModelInfo(&modelInfo));
#if !defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventは成功する
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#endif // !defined(NNT_NFP_PLATFORM_NX)
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::Format(buffer, sizeof(buffer)));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::GetAdminInfo(&adminInfo));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::SetRegisterInfo(regInfoSet));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::DeleteRegisterInfo());
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
                              nnt::nfp::wrapper::DeleteApplicationArea());

#if defined(NNT_NFP_PLATFORM_NX)
    bool outValue = false;
    NNT_EXPECT_RESULT_FAILURE(
        nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE,
        nnt::nfp::wrapper::ExistsApplicationArea(&outValue));
    EXPECT_EQ(false, outValue);
#endif // (NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXでは高速タグ検出は実装されないので除外
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(false));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetPowerSavingModeEnabled(true));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_TypeA));
    NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nn::nfp::SetProtocolFilter(nn::nfp::NfcProtocol_All));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NXではConnectは実装されないので除外
    // CTR 専用のテストです。
    if (!nn::os::IsRunOnSnake())
    {
        nn::os::EventType                      connectEvent;
        nn::os::EventType                      disconnectEvent;
        uint32_t                                sentSize;
        uint32_t                                totalSize;
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::UpdateTargetFirmware(&activateEvent));
        NNT_EXPECT_RESULT_FAILURE(nn::nfp::RESULT_ERROR_INVALID_DEVICE_STATE, nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)
}

TEST_F(NfpAutoWithoutTag, TestCaseGoodOperationNone)
{
    // 状態取得系の API のテストです。
#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::NONE, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(false, nnt::nfp::DoExistsApplicationArea());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Unexpected, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(nn::nfp::State_None, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(0, nnt::nfp::wrapper::GetBackupSaveDataSize());
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // 初期化 API のテストです。
#if !defined(NNT_NFP_PLATFORM_NX) // NXではInitializeは返り値を持たないので実行できることのみ確認
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeWithRetry());
#else
    nnt::nfp::wrapper::Initialize();
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) // NXではFinalizeは返り値を持たない
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Finalize());
#else
    nnt::nfp::wrapper::Finalize();
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) // NXではInitializeSystemは返り値を持たないので実行できることのみ確認
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());
#else
    nnt::nfp::wrapper::InitializeSystem();
#endif // !defined(NNT_NFP_PLATFORM_NX)
#if !defined(NNT_NFP_PLATFORM_NX) // NXではFinalizeSystemは返り値を持たない
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::FinalizeSystem());
#else
    nnt::nfp::wrapper::FinalizeSystem();
#endif // !defined(NNT_NFP_PLATFORM_NX)
}

TEST_F(NfpAutoWithoutTag, TestCaseGoodOperationInit)
{
    // INIT 状態に移行します。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::InitializeSystemWithRetry());

#if defined(NNT_NFP_PLATFORM_NX)
    // 後続のテストの為にデバイスハンドルを取得しておく
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::SearchDeviceHandle());
#endif // defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
    // バックアップ関係の API のテストです。この時点で必ず 1 つ以上バックアップが存在します。
    nn::nfp::BackupHeaderInfo  backupHeader     = {};
    nn::nfp::BackupEntryInfo   backupEntry      = {};
    EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
    EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&backupHeader, g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&backupEntry, 0, g_pBackupData, BackupBufferSize));
#else
    // バックアップ関係の API のテストです。(この時点で 1 つ以上バックアップが存在するかは不定。)
    nn::nfp::BackupDataHeader headerInfo;
    size_t dataSize;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
    EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
    std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
    EXPECT_TRUE(0 <= headerInfo.entryNum);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // イベントセット系の API のテストです。
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));

#if !defined(NNT_NFP_PLATFORM_NX) // NXではAPIが削除されている
    // reserved が 0 初期化されていることを確認します。
    nn::nfp::RegisterInfoPrivate       regInfoSet;
    nn::nfp::ApplicationAreaCreateInfo createInfo;
    std::memset(&regInfoSet, 0xFF, sizeof(regInfoSet));
    std::memset(&createInfo, 0xFF, sizeof(createInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(&regInfoSet, sizeof(regInfoSet)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(&createInfo, sizeof(createInfo)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // 状態取得系の API です。
#if defined(NNT_NFP_PLATFORM_NX) //NXで追加されたAPI
    // デバイスハンドル取得のテストです。
    nn::nfp::DeviceHandle deviceHandle;
    int outCount;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
    // デバイスハンドルからNpad IDを取得するテストです。
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId));
    EXPECT_TRUE(nnt::nfp::CheckNpadStyle(npadId));
#endif // defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnect機能は実装されない
    nn::nfp::ConnectionStatus connectionStatus;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectionStatus(&connectionStatus));
    EXPECT_EQ(nn::nfp::BEFORE_ACTIVATION, connectionStatus.deactivateReason);
    EXPECT_TRUE(nnt::nfp::IsAllZero(connectionStatus.reserved, sizeof(connectionStatus.reserved)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::INIT, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(false, nnt::nfp::DoExistsApplicationArea());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Init, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)

    // タグの検出開始に関するテストです。
#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnect機能は実装されない
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
    EXPECT_TRUE(nnt::nfp::WaitForConnect());
#endif // !defined(NNT_NFP_PLATFORM_NX)

    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnect機能は実装されないので除外
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
    // ニックネーム変換のテストです。
    nn::Bit16 nickName[] = {0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
                       0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000};
    NNT_EXPECT_RESULT_SUCCESS(nn::nfp::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu));
#endif //!defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnect機能は実装されないので除外
    // SNAKE では FANGATE 関連の API が全て成功します。
    nn::Result                      connectResult;
    nn::Result                      updateResult;
    nn::nfp::TargetConnectionStatus targetStatus;
    nn::os::EventType                   updateEvent;
    uint32_t                             sentSize;
    uint32_t                             totalSize;
    if (nn::os::IsRunOnSnake())
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(connectResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(updateResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    }
    else
    {
        // CTR の FANGATE 接続前の API テストです。 SNAKE では以降のテストを実施しません。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_DISCONNECTED, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());

        // CTR の FANGATE 接続試行中の API テストです。接続が一瞬で終わらないことを前提としています。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_CONNECTING, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
        EXPECT_TRUE(nnt::nfp::WaitForConnect());

        // CTR の FANGATE 接続後の API テストです。
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(connectResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    }
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) // NXではFinalizeSystemは返り値を持たない
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
    nnt::nfp::FinalizeSystem();
#endif // !defined(NNT_NFP_PLATFORM_NX)
}

TEST_F(NfpAutoWithoutTag, TestCaseGoodOperationSearch)
{
    // SEARCH 状態に移行します。
    nnt::nfp::DoSearch();

#if !defined(NNT_NFP_PLATFORM_NX)
    // バックアップ関係の API のテストです。この時点で必ず 1 つ以上バックアップが存在します。
    nn::nfp::BackupHeaderInfo  backupHeader     = {};
    nn::nfp::BackupEntryInfo   backupEntry      = {};
    EXPECT_TRUE(0 < nnt::nfp::wrapper::GetBackupSaveDataSize());
    EXPECT_TRUE(nnt::nfp::wrapper::GetBackupSaveDataSize() < BackupBufferSize);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetAllBackupSaveData(g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupHeaderFromMemory(&backupHeader, g_pBackupData, BackupBufferSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetBackupEntryFromMemory(&backupEntry, 0, g_pBackupData, BackupBufferSize));
#else
    // バックアップ関係の API のテストです。(この時点で 1 つ以上バックアップが存在するかは不定。)
    nn::nfp::BackupDataHeader headerInfo;
    size_t dataSize;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ReadBackupData(g_pBackupData, &dataSize, BackupBufferSize));
    EXPECT_EQ(sizeof(nn::nfp::BackupData), dataSize);
    std::memcpy(&headerInfo, &(reinterpret_cast<nn::nfp::BackupData*>(g_pBackupData)->header), sizeof(headerInfo));
    EXPECT_TRUE(0 <= headerInfo.entryNum);
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::WriteBackupData(g_pBackupData, sizeof(nn::nfp::BackupData)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if defined(NNT_NFP_PLATFORM_NX) //NXではState_None以外であればAttachActivateEventが成功する
    // アタッチ・デタッチイベント登録のテストです。
    nn::os::SystemEventType            activateEvent;
    nn::os::SystemEventType            deactivateEvent;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachActivateEvent(&activateEvent));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::AttachDeactivateEvent(&deactivateEvent));
#endif // defined(NNT_NFP_PLATFORM_NX)

    // タグの検出停止に関するテストです。
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StopDetection());
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::StartDetection());

    // 状態取得系の API です。
#if defined(NNT_NFP_PLATFORM_NX) //NXで追加されたAPI
    // デバイスハンドル取得のテストです。
    nn::nfp::DeviceHandle deviceHandle;
    int outCount;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ListDevices(&deviceHandle, &outCount, 1));
    nn::hid::NpadIdType npadId;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetNpadId(&npadId));
#endif // defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnect機能は実装されない
    nn::nfp::ConnectionStatus connectionStatus;
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectionStatus(&connectionStatus));
    EXPECT_EQ(nn::nfp::BEFORE_ACTIVATION, connectionStatus.deactivateReason);
    EXPECT_TRUE(nnt::nfp::IsAllZero(connectionStatus.reserved, sizeof(connectionStatus.reserved)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX)
    EXPECT_EQ(nn::nfp::RW_SEARCH, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(false, nnt::nfp::DoExistsApplicationArea());
#else
    EXPECT_EQ(nn::nfp::DeviceState_Search, nnt::nfp::wrapper::GetDeviceState());
    EXPECT_EQ(nn::nfp::State_Init, nnt::nfp::wrapper::GetState());
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) // NXではAPIが削除されているため除外
    // reserved が 0 初期化されていることを確認します。
    nn::nfp::RegisterInfoPrivate       regInfoSet;
    nn::nfp::ApplicationAreaCreateInfo createInfo;
    std::memset(&regInfoSet, 0xFF, sizeof(regInfoSet));
    std::memset(&createInfo, 0xFF, sizeof(createInfo));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeRegisterInfoSet(&regInfoSet));
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::InitializeCreateInfo(&createInfo));
    EXPECT_TRUE(nnt::nfp::IsAllZero(&regInfoSet, sizeof(regInfoSet)));
    EXPECT_TRUE(nnt::nfp::IsAllZero(&createInfo, sizeof(createInfo)));
#endif // !defined(NNT_NFP_PLATFORM_NX)

#if !defined(NNT_NFP_PLATFORM_NX) //NX には ReplaceOtherRegionName がない
    // ニックネーム変換のテストです。
    nn::Bit16 nickName[] = {0x0100, 0x2000, 0x3000, 0x4000, 0x7F00,
                       0x8000, 0xFF00, 0x0001, 0x0010, 0xFFFF, 0x0000};
    NNT_EXPECT_RESULT_SUCCESS(nn::nfp::ReplaceOtherRegionName(nickName, nickName, sizeof(nickName), nn::nfp::FontRegion_JpUsEu));
#endif //!defined(NNT_NFP_PLATFORM_NX)

    // ライブラリの解放に関するテストです。
#if !defined(NNT_NFP_PLATFORM_NX) // NXではFinalizeSystemは返り値を持たない
    NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::FinalizeSystem());
#else
    nnt::nfp::FinalizeSystem();
#endif // !defined(NNT_NFP_PLATFORM_NX)
    nnt::nfp::DoSearch();

#if !defined(NNT_NFP_PLATFORM_NX) // NXではConnect機能が実装されないため除外
    // SNAKE では FANGATE 関連の API が全て成功します。
    nn::Result                      connectResult;
    nn::Result                      updateResult;
    nn::nfp::TargetConnectionStatus targetStatus;
    nn::os::EventType                   updateEvent;
    uint32_t                             sentSize;
    uint32_t                             totalSize;

    if (nn::os::IsRunOnSnake())
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Connect());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(connectResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::ConnectForFwUpdate());
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::UpdateTargetFirmware(&updateEvent));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(updateResult);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateProgress(&sentSize, &totalSize));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    }
    else
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetTargetConnectionStatus(&targetStatus));
        EXPECT_EQ(nn::nfp::TARGET_CONNECTED, targetStatus);
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetConnectResult(&connectResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::GetUpdateResult(&updateResult));
        NNT_EXPECT_RESULT_SUCCESS(nnt::nfp::wrapper::Disconnect());
    }
#endif
}
