﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/oe.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Event.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/nfp/testMifare_Common.h>

//================================================================================
// このテストで使用する定義です。
//================================================================================

namespace
{
    const uint32_t   AgingRepeat = 1000000;
    const uint32_t   RandomSleepTimeMax = 300;
    const char* MifareDeviceState[] =
    {
        "Init",          //!< タグを探していない状態です。
        "Search",        //!< タグを探している状態です。
        "Active",        //!< DeviceState_Search でタグを検知するとこの状態になります。
        "Deactive",      //!< タグを検知した後、タグが離れるとこの状態になります。
        "Unexpected"     //!< 想定していない状態です。
    };
    const char* MifareLibraryState[] =
    {
        "None",          //!< MIFARE ライブラリが初期化されていない状態です。起動時はこの状態です。
        "Init",          //!< MIFARE ライブラリが初期化されている状態です。
        "Unexpected"     //!< 想定していない状態です。
    };

    // ブロックカウント最小
    const int MinBlockCount = 1;
    // ブロックカウント最大
    const int MaxBlockCount = 16;
    // 読み込みデータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockData) g_readBlockData[MaxBlockCount] = {};
    // 読み込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(ReadBlockParameter) g_readBlockParameter[MaxBlockCount] = {};
    // 書き込みブロックパラメータ
    NNT_NFC_COMMON_IF_TYPE_MIFARE(WriteBlockParameter) g_writeBlockParameter[MaxBlockCount] = {};
} // end of anonymous namespace

//------------------------------------------------------------------
// Macro Defnition
//------------------------------------------------------------------

#define MIFARE_LOG( ... ) \
    do { \
        NN_LOG("[MIFARE_AGING] LIB_STATE = %s DEV_STATE = %s LINE = %04d ",\
               MifareLibraryState[nnt::mifare::wrapper::GetState()],          \
               MifareDeviceState[nnt::mifare::wrapper::GetDeviceState()],     \
               __LINE__);                                               \
        NN_LOG(__VA_ARGS__);                                            \
    } while(NN_STATIC_CONDITION(false))

#define NNT_MIFARE_SWITCH_NFC_ENABLE // NfcのON／OFF切り替え機能スイッチ

//================================================================================
// テストで使用する共通関数です。
//================================================================================
namespace {
    int RandomSleep(int max)
    {
        nn::Bit16 randNum;
        nnt::mifare::CreateRandomNumberSequence(&randNum, 1);
        int ms = randNum % max;
        nnt::mifare::Sleep(ms);
        return ms;
    }

    bool IsAllowResultRead(nn::Result targetResult)
    {
        if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultAccessError())
        {
            return false;
        }
        else if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultWrite(nn::Result targetResult)
    {
        if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDisabled())
        {
            return false;
        }
        else if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultAccessError())
        {
            return false;
        }
        else if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNeedRestart())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool IsAllowResultStopDetect(nn::Result targetResult)
    {
        if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDeviceNotFound())
        {
            return false;
        }
        else if(targetResult <= NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::ResultNfcDisabled())
        {
            return false;
        }
        NNT_EXPECT_RESULT_SUCCESS(targetResult);
        return true;
    }

    bool CheckForActivateEvent(bool isActivate)
    {
        EXPECT_TRUE(isActivate);
        return isActivate;
    }

    bool ReadWriteTag(int blockCount) NN_NOEXCEPT
    {
        nn::Result result;

        nnt::mifare::PrepareWriteParam(g_writeBlockParameter, blockCount);
        nnt::mifare::PrintWriteBlockParameter(g_writeBlockParameter, blockCount);
        if(false == IsAllowResultWrite(
                nnt::mifare::wrapper::Write(g_writeBlockParameter, blockCount)))
        {
            return false;
        }
        nnt::mifare::PrepareReadParam(g_readBlockParameter, blockCount);
        if(false == IsAllowResultRead(nnt::mifare::wrapper::Read(
                g_readBlockData, g_readBlockParameter, blockCount)))
        {
            return false;
        }
        nnt::mifare::PrintReadBlockData(g_readBlockData, blockCount);
        // 書き込んだデータを読み込めたか確認
        nnt::mifare::CheckReadData(g_readBlockData, g_writeBlockParameter, blockCount);

        return true;
    }

#if defined(NNT_MIFARE_SWITCH_NFC_ENABLE)
    // スレッドスタックサイズ
    const size_t ThreadStackSize = 4 * 1024;
    // スレッドタイプ
    nn::os::ThreadType  g_Thread;
    // スレッド継続停止のスイッチ
    bool g_ThreadSwitch = true;
    // スレッドのスタック
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];

    // ランダムでSetNfcEnabled()を切り替えるスレッド
    void SwitchNfcEnabledThread(void* pArg) NN_NOEXCEPT
    {
        NN_UNUSED(pArg);
        MIFARE_LOG("%s start \n", NN_CURRENT_FUNCTION_NAME);
        // Nfc On 時間の最大値と最小値(ms)
        const int NfcOnTimeMax = 30000;
        const int NfcOnTimeMin = 10000;
        // Nfc Off 時間の最大値と最小値(ms)
        const int NfcOffTimeMax = 5000;
        const int NfcOffTimeMin = 1000;
        // 試験が終了するまで繰り返す
        g_ThreadSwitch = true;
        while(g_ThreadSwitch)
        {
            std::srand(nn::os::GetSystemTick().GetInt64Value() & 0xFFFFFFFF);
            // スリープ終了でNfc有効無効を切り替える
            if(nnt::mifare::IsNfcEnable())
            {
                // スリープ時間を設定
                int intervalTime = (std::rand() % (NfcOnTimeMax - NfcOnTimeMin)) + NfcOnTimeMin;
                MIFARE_LOG("%s sleep IsNfcEnable = %d intervalTime = %d\n",
                        NN_CURRENT_FUNCTION_NAME, nnt::mifare::IsNfcEnable(), intervalTime);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(intervalTime));
                nnt::mifare::NfcOff();
                MIFARE_LOG("%s <<Nfc Off>> IsNfcEnable = %d \n",
                        NN_CURRENT_FUNCTION_NAME, nnt::mifare::IsNfcEnable());
            }
            else
            {
                // スリープ時間を設定
                int intervalTime = (std::rand() % (NfcOffTimeMax - NfcOffTimeMin)) + NfcOffTimeMin;
                MIFARE_LOG("%s sleep IsNfcEnable = %d intervalTime = %d\n",
                        NN_CURRENT_FUNCTION_NAME, nnt::mifare::IsNfcEnable(), intervalTime);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(intervalTime));
                nnt::mifare::NfcOn();
                MIFARE_LOG("%s <<Nfc On>> IsNfcEnable = %d \n",
                        NN_CURRENT_FUNCTION_NAME, nnt::mifare::IsNfcEnable());
            }
        }
        MIFARE_LOG("%s end \n", NN_CURRENT_FUNCTION_NAME);
    }
#endif // defined(NNT_MIFARE_SWITCH_NFC_ENABLE)
}

//================================================================================
// Mifareライブラリの状態に応じて書込みとフォーマットを繰り返すエイジングです。
//================================================================================

class MifareAgingRemoveTag : public nnt::mifare::TestFramework
{
protected:

    MifareAgingRemoveTag() NN_NOEXCEPT
    {
        // oeライブラリを初期化します。実機のみ
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        nn::oe::Initialize();
#endif //defined(NN_BUILD_CONFIG_OS_HORIZON)
        // コントローラの初期化
        nnt::mifare::InitializeHidController();
        nnt::mifare::wrapper::SetApiCallLoggingMode(nnt::mifare::wrapper::LogMode_Aging);
    }
};

//================================================================================
// エイジングの実装です。
//================================================================================

TEST_F(MifareAgingRemoveTag,TestCaseAgingRemoveTag)
{

    bool doNextStep = true;
    int blockCount = 1;

    for(int agingCount = 0; agingCount < AgingRepeat; agingCount++)
    {
        if(true == doNextStep)
        {
            MIFARE_LOG("RepeatCount : %0d \n", agingCount);
        }
        doNextStep = true;

        int ms = RandomSleep(RandomSleepTimeMax);

        nn::Bit16 command;
        nnt::mifare::CreateRandomNumberSequence(&command, 1);

        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState deviceState = nnt::mifare::wrapper::GetDeviceState();
        NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State libState = nnt::mifare::wrapper::GetState();
        MIFARE_LOG("SleepTime : %04d \n", ms);
        MIFARE_LOG("DeviceState  : %02d \n", deviceState);
        MIFARE_LOG("LibraryState : %02d \n", libState);

        if(libState != NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::State_Init)
        {
            NNT_MIFARE_ASSERT_EQUAL(NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected,deviceState);
            MIFARE_LOG("nnt::mifare::Initialize\n");
            NNT_MIFARE_ASSERT_RESULT_SUCCESS_FATAL(nnt::mifare::InitializeSystemWithRetry());
#if defined(NNT_MIFARE_SWITCH_NFC_ENABLE)
            // NfcのOn／Offを切り替えるスレッドを起動する
            MIFARE_LOG("\nCreate SwitchNfcEnabledThread\n");
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(
                    &g_Thread, SwitchNfcEnabledThread, nullptr, g_ThreadStack,
                    sizeof(g_ThreadStack), nn::os::DefaultThreadPriority));
            nn::os::StartThread(&g_Thread);
#endif // defined(NNT_MIFARE_SWITCH_NFC_ENABLE)
            continue;
        }

        switch(deviceState)
        {
        case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Init:
            {
                MIFARE_LOG("nnt::mifare::DoSearch\n");
                nnt::mifare::DoSearch();
            }
            break;
        case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Search:
            {
                MIFARE_LOG("======= Waiting Tag Attached =======\n");

                switch (command % 2)
                {
                case 0:
                    {
                        doNextStep = CheckForActivateEvent(nnt::mifare::WaitForActivate());
                        if(false == doNextStep)
                        {
                            break;
                        }
                        MIFARE_LOG("Tag is detected\n");
                    }
                    break;
                case 1:
                    {
                        MIFARE_LOG(NNT_NFC_COMMON_IF_NAMESPACE_STRING_MIFARE "::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::mifare::wrapper::StopDetection());
                    }
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;
        case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Active:
            {
                switch(command % 2)
                {
                case 0:
                    {
                        MIFARE_LOG("nn::::nfc::mifare::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::mifare::wrapper::StopDetection());
                    }
                    break;
                case 1:
                    {
                        doNextStep = ReadWriteTag(blockCount);
                        // ブロックは最大16ブロック
                        blockCount++;
                        if (blockCount > MaxBlockCount)
                        {
                            blockCount = MinBlockCount;
                        }
                    }
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;
        case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Deactive:
            {
                switch(command % 3)
                {
                case 0:
                    {
#if defined(NNT_MIFARE_SWITCH_NFC_ENABLE)
                        g_ThreadSwitch = false;
                        nn::os::WaitThread(&g_Thread);
                        nn::os::DestroyThread(&g_Thread);
#endif // defined(NNT_MIFARE_SWITCH_NFC_ENABLE)
                        MIFARE_LOG("nnt::mifare::FinalizeSystem\n");
                        NNT_MIFARE_ASSERT_RESULT_SUCCESS_FATAL(nnt::mifare::FinalizeSystem());
                    }
                    break;
                case 1:
                    {
                        MIFARE_LOG(NNT_NFC_COMMON_IF_NAMESPACE_STRING_MIFARE "::StopDetection\n");
                        doNextStep = IsAllowResultStopDetect(nnt::mifare::wrapper::StopDetection());
                    }
                    break;
                case 2:
                    {
                        MIFARE_LOG("nnt::mifare::DoSearch\n");
                        nnt::mifare::DoSearch();
                    }
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            break;
        case NNT_NFC_COMMON_IF_NAMESPACE_MIFARE::DeviceState_Unexpected:
            {
                MIFARE_LOG("nnt::mifare::DoSearch\n");
                nnt::mifare::DoSearch();
            }
            break;
        default: NN_UNEXPECTED_DEFAULT;
            break;
        }
        if(false == doNextStep)
        {
            //処理失敗時のリトライなのでループカウントがインクリメントされないようにする
            agingCount--;
        }
    }
#if defined(NNT_MIFARE_SWITCH_NFC_ENABLE)
    //NfcのOn／Offを切り替えるスレッドを停止する
    g_ThreadSwitch = false;
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
#endif // defined(NNT_MIFARE_SWITCH_NFC_ENABLE)

    NNT_MIFARE_ASSERT_RESULT_SUCCESS_FATAL(nnt::mifare::wrapper::StopDetection());
    NNT_MIFARE_ASSERT_RESULT_SUCCESS_FATAL(nnt::mifare::FinalizeSystem());
} // NOLINT(impl/function_size)
