﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"
#include <nn/news/detail/service/core/news_ReceivedHistoryManager.h>

// ストレステストを有効にするかどうか
#define ENABLE_STRESS_TEST 1 // NOLINT(preprocessor/const)

using namespace nn::news::detail::service::core;

TEST(ReceivedHistoryManager, Initialize)
{
    nn::time::Initialize();

    std::srand(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));

    ASSERT_RESULT_SUCCESS(nnt::news::MountSystemStorage(0x8000000000000099));
}

TEST(ReceivedHistoryManager, Clear)
{
    // 複数回呼出確認。
    ReceivedHistoryManager::GetInstance().Clear();
    ReceivedHistoryManager::GetInstance().Clear();
}

TEST(ReceivedHistoryManager, Language)
{
    ReceivedHistoryManager::Record records[3] = {};

    for (int8_t i = 0; i < 3; i++)
    {
        records[i].domain = 'N';
        records[i].newsId = 1;
        records[i].language = i + 1;
        records[i].dataId = 1;

        // 言語違いはすべて追加可能
        EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().AddOrUpdate(records[i]));
    }

    ReceivedHistoryManager::SearchKey key = {false, 0, 'N', 1, 0, 9};
    bool isOverwritable;

    for (int8_t i = 0; i < 3; i++)
    {
        key.language = i + 1;
        EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable, key));
        EXPECT_TRUE(isOverwritable);
    }

    ReceivedHistoryManager::GetInstance().Clear();
}

#if ENABLE_STRESS_TEST == 1

TEST(ReceivedHistoryManager, Stress)
{
    int count = ReceivedHistoryManager::HistoryCountMax * 2;
    int percent = -1;

    for (int i = 0; i < count; i++)
    {
        ReceivedHistoryManager::Record record = {};

        record.domain = 'N';
        record.newsId = i;
        record.language = 0;
        record.dataId = 0;

        EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().AddOrUpdate(record));

        ReceivedHistoryManager::SearchKey key[4] =
        {
            {false, 0, record.domain, record.newsId, 0, 0}, // 同じ言語・同じデータ ID
            {false, 0, record.domain, record.newsId, 0, 1}, // 同じ言語・違うデータ ID
            {false, 0, record.domain, record.newsId, 1, 0}, // 違う言語・同じデータ ID
            {false, 0, record.domain, record.newsId, 1, 1}  // 違う言語・違うデータ ID
        };

        bool isImportable[4];
        bool isOverwritable[4];

        // 追加したてのニュース ID
        for (int n = 0; n < 4; n++)
        {
            EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable[n], key[n]));
            EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable[n], key[n]));
            EXPECT_FALSE(isImportable[n]);
        }

        EXPECT_FALSE(isOverwritable[0]);
        EXPECT_TRUE(isOverwritable[1]); // 同じ言語・違うデータ ID なら上書き可能
        EXPECT_FALSE(isOverwritable[2]);
        EXPECT_FALSE(isOverwritable[3]);

        // 受信履歴に存在しないニュース ID
        for (int n = 0; n < 4; n++)
        {
            key[n].newsId = 0xFFFFFFFFFFFFFFFF;
            EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable[n], key[n]));
            EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable[n], key[n]));
            EXPECT_TRUE(isImportable[n]);
            EXPECT_FALSE(isOverwritable[n]);
        }

        // 境界値（全件）
        if (i == ReceivedHistoryManager::HistoryCountMax - 1)
        {
            // ぎりぎり受信履歴に残っているニュース ID
            for (int n = 0; n < 4; n++)
            {
                key[n].newsId = 0;
                EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable[n], key[n]));
                EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable[n], key[n]));
                EXPECT_FALSE(isImportable[n]);
            }

            EXPECT_FALSE(isOverwritable[0]);
            EXPECT_TRUE(isOverwritable[1]); // 同じ言語・違うデータ ID なら上書き可能
            EXPECT_FALSE(isOverwritable[2]);
            EXPECT_FALSE(isOverwritable[3]);
        }
        // 境界値（全件 + 1）
        if (i == ReceivedHistoryManager::HistoryCountMax)
        {
            // 受信履歴から消えてしまったニュース ID
            for (int n = 0; n < 4; n++)
            {
                key[n].newsId = 0;
                EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable[n], key[n]));
                EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable[n], key[n]));
                EXPECT_TRUE(isImportable[n]);
                EXPECT_FALSE(isOverwritable[n]);
            }
            // 受信履歴から消えてしまったニュース ID
            for (int n = 0; n < 4; n++)
            {
                key[n].newsId = ReceivedHistoryManager::CountPerBlock - 1;
                EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable[n], key[n]));
                EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable[n], key[n]));
                EXPECT_TRUE(isImportable[n]);
                EXPECT_FALSE(isOverwritable[n]);
            }

            // ぎりぎり受信履歴に残っているニュース ID
            for (int n = 0; n < 4; n++)
            {
                key[n].newsId = ReceivedHistoryManager::CountPerBlock;
                EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable[n], key[n]));
                EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable[n], key[n]));
                EXPECT_FALSE(isImportable[n]);
            }

            EXPECT_FALSE(isOverwritable[0]);
            EXPECT_TRUE(isOverwritable[1]); // 同じ言語・違うデータ ID なら上書き可能
            EXPECT_FALSE(isOverwritable[2]);
            EXPECT_FALSE(isOverwritable[3]);
        }

        int random = rand() % count;

        // 途中に存在しうるキー
        for (int n = 0; n < 4; n++)
        {
            key[n].newsId = random;
            EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable[n], key[n]));
            EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable[n], key[n]));
        }

        // 追加回数が HistoryCountMax までは全件入る。
        if (i < ReceivedHistoryManager::HistoryCountMax)
        {
            if (random <= i)
            {
                for (int n = 0; n < 4; n++)
                {
                    EXPECT_FALSE(isImportable[n]);
                }
                EXPECT_FALSE(isOverwritable[0]);
                EXPECT_TRUE(isOverwritable[1]); // 同じ言語・違うデータ ID なら上書き可能
                EXPECT_FALSE(isOverwritable[2]);
                EXPECT_FALSE(isOverwritable[3]);
            }
            else
            {
                for (int n = 0; n < 4; n++)
                {
                    EXPECT_TRUE(isImportable[n]);
                    EXPECT_FALSE(isOverwritable[n]);
                }
            }
        }
        else
        {
            // HistoryCountMax 以降は、CountPerBlock 分だけ削除してから追加される。

            // 最新のブロック位置
            int last = (i / ReceivedHistoryManager::CountPerBlock) * ReceivedHistoryManager::CountPerBlock;
            // 最新のブロック位置 - HistoryCountMax + CountPerBlock が受信履歴ファイルの先頭に書かれているはず。
            int base = last - ReceivedHistoryManager::HistoryCountMax + ReceivedHistoryManager::CountPerBlock;

            if (random >= base && random <= i)
            {
                for (int n = 0; n < 4; n++)
                {
                    EXPECT_FALSE(isImportable[n]);
                }
                EXPECT_FALSE(isOverwritable[0]);
                EXPECT_TRUE(isOverwritable[1]); // 同じ言語・違うデータ ID なら上書き可能
                EXPECT_FALSE(isOverwritable[2]);
                EXPECT_FALSE(isOverwritable[3]);
            }
            else
            {
                for (int n = 0; n < 4; n++)
                {
                    EXPECT_TRUE(isImportable[n]);
                    EXPECT_FALSE(isOverwritable[n]);
                }
            }
        }

        int p = i * 100 / count;

        if (p != percent)
        {
            percent = p;
            NN_LOG("%3d %%...\n", percent);
        }
    }

    NN_LOG("100 %%\n");
} // NOLINT(impl/function_size)

TEST(ReceivedHistoryManager, Update)
{
    ReceivedHistoryManager::GetInstance().Clear();

    int count = ReceivedHistoryManager::HistoryCountMax;
    int percent = -1;

    for (int i = 0; i < count; i++)
    {
        ReceivedHistoryManager::Record record = {};

        record.domain = 'N';
        record.newsId = i;
        record.dataId = 0;

        EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().AddOrUpdate(record));

        int p = i * 100 / count;

        if (p != percent)
        {
            percent = p;
            NN_LOG("Add: %3d %%...\n", percent);
        }
    }

    NN_LOG("Add: 100 %%\n");

    count = ReceivedHistoryManager::HistoryCountMax * 3;
    percent = -1;

    for (int i = 0; i < count; i++)
    {
        ReceivedHistoryManager::Record record = {};

        record.domain = 'N';
        record.newsId = rand() % ReceivedHistoryManager::HistoryCountMax;
        record.dataId = 1 + count;

        // Add した範囲内でデータ ID を更新していく。
        EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().AddOrUpdate(record));

        int p = i * 100 / count;

        if (p != percent)
        {
            percent = p;
            NN_LOG("Update: %3d %%...\n", percent);
        }
    }

    NN_LOG("Update: 100 %%\n");

    ReceivedHistoryManager::SearchKey key = {};

    // ReceivedHistoryManager::HistoryCountMax 以上登録されたら、先頭の newsId = 0 は消えているはず。
    key.domain = 'N';
    key.newsId = 0;

    bool isImportable;
    bool isOverwritable;

    EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsImportable(&isImportable, key));

    key.dataId = 0xFFFFFFFFFFFFFFFF;
    EXPECT_RESULT_SUCCESS(ReceivedHistoryManager::GetInstance().IsOverwritable(&isOverwritable, key));

    EXPECT_FALSE(isImportable);
    EXPECT_TRUE(isOverwritable);
}

#endif

TEST(ReceivedHistoryManager, Finalize)
{
    nn::time::Finalize();

    nnt::news::UnmountSystemStorage();
}
