﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"

namespace
{
    nn::ApplicationId g_ApplicationId = {0x0100000000001000};
    const char* g_Passphrase = "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx";

    const char* g_TopicId = "nx_news_sp_siglo03";
}

namespace
{
    const int ThreadCount = 6;

    nn::os::ThreadType g_Threads[ThreadCount];
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStacks[ThreadCount][32 * 1024];

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
}

static uint32_t GetRandom() NN_NOEXCEPT
{
    uint32_t value;
    nn::os::GenerateRandomBytes(&value, sizeof (value));

    return value;
}

static void ClearThread(void*) NN_NOEXCEPT
{
    // 0 ～ 30 秒
    while (!g_StopEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(GetRandom() % 30000)))
    {
        nn::Result result = nn::news::ClearStorage();

        ASSERT_TRUE(result.IsSuccess() || nn::news::ResultLocked::Includes(result));

        if (result.IsSuccess())
        {
            ASSERT_RESULT_SUCCESS(nn::news::ClearSubscriptionStatusAll());
            ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicId, nn::news::SubscriptionStatus_Subscribed));
        }
    }
}

static void PostNewsThread(void*) NN_NOEXCEPT
{
    static nn::Bit8 s_MountRomCacheBuffer[1 * 1024];

    size_t mountRomCacheUseSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize));

    NN_ABORT_UNLESS(mountRomCacheUseSize <= sizeof (s_MountRomCacheBuffer));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("rom", s_MountRomCacheBuffer, sizeof (s_MountRomCacheBuffer)));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    nn::fs::FileHandle handle = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, "rom:/news.msgpack", nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    static nn::Bit8 s_Buffer[1 * 1024];
    size_t read = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&read, handle, 0, s_Buffer, sizeof (s_Buffer)));

    // 0 ～ 3 秒
    while (!g_StopEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(GetRandom() % 3000)))
    {
        nn::Result result = nn::news::PostLocalNews(s_Buffer, read);

        ASSERT_TRUE(result.IsSuccess() || nn::news::ResultLocked::Includes(result));
    }
}

static void DatabaseAccessThread(void*) NN_NOEXCEPT
{
    // 0 ～ 5 秒
    while (!g_StopEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(GetRandom() % 5000)))
    {
        nn::news::Database db;
        nn::news::NewsRecord records[10];
        int count;

        ASSERT_RESULT_SUCCESS(db.Open());
        ASSERT_RESULT_SUCCESS(db.GetList(&count, records, "", "", 0, NN_ARRAY_SIZE(records)));

        // 0 ～ 5 秒
        g_StopEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(GetRandom() % 5000));
    }
}

static void DataAccessThread(void*) NN_NOEXCEPT
{
    // 0 ～ 5 秒
    while (!g_StopEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(GetRandom() % 5000)))
    {
        nn::news::Database db;
        nn::news::NewsRecord record;
        int count;

        ASSERT_RESULT_SUCCESS(db.Open());
        ASSERT_RESULT_SUCCESS(db.GetList(&count, &record, "", "", 0, 1));

        if (count > 0)
        {
            nn::news::Data data;
            ASSERT_RESULT_SUCCESS(data.Open(record));

            int64_t size;
            ASSERT_RESULT_SUCCESS(data.GetSize(&size));

            // 0 ～ 5 秒
            g_StopEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(GetRandom() % 5000));
        }
    }
}

TEST(RandomAccess, Do)
{
    ASSERT_RESULT_SUCCESS(nn::news::ClearStorage());
    ASSERT_RESULT_SUCCESS(nn::news::ClearSubscriptionStatusAll());

    ASSERT_RESULT_SUCCESS(nn::news::SetPassphrase(g_ApplicationId, g_Passphrase));
    ASSERT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicId, nn::news::SubscriptionStatus_Subscribed));

    nn::os::CreateThread(&g_Threads[0], ClearThread, nullptr,
        g_ThreadStacks[0], sizeof (g_ThreadStacks[0]), nn::os::DefaultThreadPriority);

    nn::os::CreateThread(&g_Threads[1], PostNewsThread, nullptr,
        g_ThreadStacks[1], sizeof (g_ThreadStacks[1]), nn::os::DefaultThreadPriority);

    nn::os::CreateThread(&g_Threads[2], DatabaseAccessThread, nullptr,
        g_ThreadStacks[2], sizeof (g_ThreadStacks[2]), nn::os::DefaultThreadPriority);
    nn::os::CreateThread(&g_Threads[3], DatabaseAccessThread, nullptr,
        g_ThreadStacks[3], sizeof (g_ThreadStacks[3]), nn::os::DefaultThreadPriority);

    nn::os::CreateThread(&g_Threads[4], DataAccessThread, nullptr,
        g_ThreadStacks[4], sizeof (g_ThreadStacks[4]), nn::os::DefaultThreadPriority);
    nn::os::CreateThread(&g_Threads[5], DataAccessThread, nullptr,
        g_ThreadStacks[5], sizeof (g_ThreadStacks[5]), nn::os::DefaultThreadPriority);

    for (int i = 0; i < NN_ARRAY_SIZE(g_Threads); i++)
    {
        nn::os::StartThread(&g_Threads[i]);
    }

    // 10 分間のエージング
    nn::os::SleepThread(nn::TimeSpan::FromMinutes(10));

    g_StopEvent.Signal();

    for (int i = 0; i < NN_ARRAY_SIZE(g_Threads); i++)
    {
        nn::os::DestroyThread(&g_Threads[i]);
    }
}
