﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"
#include <nn/hid.h>

namespace
{
    nn::ApplicationId g_ApplicationId = {0x0100000000001000};
    const char* g_Passphrase = "xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx";

    const char* g_TopicList[] =
    {
        "nx_bcat_test_0",
        "nx_bcat_test_1",
        "nx_bcat_test_2",
        "nx_bcat_test_3",
        "nx_bcat_test_4",
        "nx_bcat_test_5",
        "nx_bcat_test_6",
        "nx_bcat_test_7",
        "nx_bcat_test_8",
        "nx_bcat_test_9",
        ""
    };

    int g_Cursor = 0;
}

static void ShowUsage() NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("A:      RequestImmediateReception\n");
    NN_LOG("B:      ShowList\n");
    NN_LOG("X:      ClearStorage\n");
    NN_LOG("Y:      ClearSubscribeAll\n");
    NN_LOG("L:      Subscribe\n");
    NN_LOG("R:      Unsubscribe\n");
    NN_LOG("Select: ShowUsage\n");
    NN_LOG("Start:  Exit\n");
    NN_LOG("--------------------------------------------------\n");
}

static const char* GetSubscriptionStatus(const char* topicId) NN_NOEXCEPT
{
    nn::news::SubscriptionStatus status;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::news::GetSubscriptionStatus(&status, topicId));

    switch (status)
    {
    case nn::news::SubscriptionStatus_Unconfigured:
        return "Unconfigured";
    case nn::news::SubscriptionStatus_Unsubscribed:
        return "Unsubscribed";
    case nn::news::SubscriptionStatus_Subscribed:
        return "Subscribed";
    case nn::news::SubscriptionStatus_AutoSubscribed:
        return "AutoSubscribed";
    default:
        return "";
    }
}

static void ShowTopcic(const char* title, int cursor) NN_NOEXCEPT
{
    NN_LOG("--------------------------------------------------\n");
    NN_LOG("%s\n", title);
    NN_LOG("--------------------------------------------------\n");

    for (int i = 0; i < g_TopicList[i][0] != '\0'; i++)
    {
        NN_LOG("%c %s(%s)\n", i == cursor ? '>' : ' ', g_TopicList[i], GetSubscriptionStatus(g_TopicList[i]));
    }

    NN_LOG("--------------------------------------------------\n");
    NN_LOG("A: Decide / B: Cancel / Up/Down: Select\n");
    NN_LOG("--------------------------------------------------\n");
}

static bool SelectCursor(const char* title, const nn::hid::DebugPadState& padStatus) NN_NOEXCEPT
{
    ShowTopcic(title, g_Cursor);

    nn::hid::DebugPadState prevPadState = padStatus;

    int cursor = g_Cursor;

    bool isDecide = false;

    while (NN_STATIC_CONDITION(1))
    {
        nn::hid::DebugPadState padState = {};
        nn::hid::GetDebugPadState(&padState);

        nn::hid::DebugPadButtonSet trigger = (~prevPadState.buttons) & padState.buttons;
        prevPadState = padState;

        if (trigger.Test<nn::hid::DebugPadButton::Up>())
        {
            if (cursor > 0)
            {
                cursor--;
            }
            ShowTopcic(title, cursor);
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Down>())
        {
            if (g_TopicList[cursor + 1][0] != '\0')
            {
                cursor++;
            }
            ShowTopcic(title, cursor);
        }
        else if (trigger.Test<nn::hid::DebugPadButton::A>())
        {
            g_Cursor = cursor;
            isDecide = true;
            break;
        }
        else if (trigger.Test<nn::hid::DebugPadButton::B>())
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // ボタンのトリガー状態を解除するまで待つ。
    while (NN_STATIC_CONDITION(1))
    {
        nn::hid::DebugPadState padState = {};
        nn::hid::GetDebugPadState(&padState);

        if (isDecide)
        {
            if (!padState.buttons.Test<nn::hid::DebugPadButton::A>())
            {
                NN_LOG("Decide. (%s)\n", g_TopicList[g_Cursor]);
                break;
            }
        }
        else
        {
            if (!padState.buttons.Test<nn::hid::DebugPadButton::B>())
            {
                NN_LOG("Cancel.\n");
                break;
            }
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));
    }

    return isDecide;
}

TEST(Npns, Initialize)
{
    nn::nifm::Initialize();
    nn::hid::InitializeDebugPad();

    ASSERT_RESULT_SUCCESS(nn::news::SetPassphrase(g_ApplicationId, g_Passphrase));
}

TEST(Npns, MainLoop)
{
    ShowUsage();

    nn::hid::DebugPadState prevPadState = {};

    while (NN_STATIC_CONDITION(1))
    {
        nn::hid::DebugPadState padState = {};
        nn::hid::GetDebugPadState(&padState);

        nn::hid::DebugPadButtonSet trigger = (~prevPadState.buttons) & padState.buttons;
        prevPadState = padState;

        if (trigger.Test<nn::hid::DebugPadButton::A>())
        {
            if (SelectCursor("RequestImmediateReception:Topic", padState))
            {
                NN_LOG("RequestImmediateReception.\n");
                EXPECT_RESULT_SUCCESS(nn::news::RequestImmediateReception(g_TopicList[g_Cursor]));
            }
        }
        else if (trigger.Test<nn::hid::DebugPadButton::B>())
        {
            nn::news::Database db;
            nn::news::NewsRecord records[10];
            int count;

            NN_LOG("ShowList.\n");
            ASSERT_RESULT_SUCCESS(db.Open());
            ASSERT_RESULT_SUCCESS(db.GetList(&count, records, "", "received_at DESC", 0, 10));

            NN_LOG("--------------------------------------------------\n");
            NN_LOG("news.db\n");
            NN_LOG("--------------------------------------------------\n");
            NN_LOG("Count = %d\n", count);

            for (int i = 0; i < count; i++)
            {
                NN_LOG("--------------------------------------------------\n");
                NN_LOG("NewsRecord[%d]\n", i);
                NN_LOG("    - NewsId: %s\n", records[i].newsId.value);
                NN_LOG("    - UserId: %s\n", records[i].userId.value);
                NN_LOG("    - ReceivedTime: %lld\n", records[i].receivedTime.value);
                NN_LOG("    - Read: %d\n", records[i].read);
                NN_LOG("    - Newly: %d\n", records[i].newly);
                NN_LOG("    - Displayed: %d\n", records[i].displayed);

                char wherePhrase[64] = {};
                nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "news_id = '%s'", records[i].newsId.value);

                EXPECT_RESULT_SUCCESS(db.UpdateWithAddition("displayed", 1, wherePhrase));
            }

            NN_LOG("--------------------------------------------------\n");
        }
        else if (trigger.Test<nn::hid::DebugPadButton::X>())
        {
            NN_LOG("ClearStorage.\n");
            EXPECT_RESULT_SUCCESS(nn::news::ClearStorage());
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Y>())
        {
            NN_LOG("ClearSubscriptionStatusAll.\n");
            EXPECT_RESULT_SUCCESS(nn::news::ClearSubscriptionStatusAll());
        }
        else if (trigger.Test<nn::hid::DebugPadButton::L>())
        {
            if (SelectCursor("Subscribe:Topic", padState))
            {
                NN_LOG("Subscribe.\n");
                EXPECT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicList[g_Cursor], nn::news::SubscriptionStatus_Subscribed));
            }
        }
        else if (trigger.Test<nn::hid::DebugPadButton::R>())
        {
            if (SelectCursor("Unsubscribe:Topic", padState))
            {
                NN_LOG("Unsubscribe.\n");
                EXPECT_RESULT_SUCCESS(nn::news::SetSubscriptionStatus(g_TopicList[g_Cursor], nn::news::SubscriptionStatus_Unsubscribed));
            }
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Select>())
        {
            ShowUsage();
        }
        else if (trigger.Test<nn::hid::DebugPadButton::Start>())
        {
            NN_LOG("Exit.\n");
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

TEST(Npns, Finalize)
{
#if defined (NN_BUILD_CONFIG_OS_WIN)
    nn::news::service::StopServer();
#endif
}
