﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testNews_Common.h"
#include <nn/news/detail/service/util/news_Sqlite.h>

namespace
{
    const char* Path = "news-dump:/news.dump.db";
}

TEST(Database, UpdateKey)
{
    ASSERT_RESULT_SUCCESS(nn::news::ClearStorage());
    ASSERT_RESULT_SUCCESS(nnt::news::PostLocalNews("rom:/news1.msgpack"));

    nn::news::Database db;
    nn::news::NewsRecord record;
    int count;

    ASSERT_RESULT_SUCCESS(db.Open());
    ASSERT_RESULT_SUCCESS(db.GetList(&count, &record, "", "", 0, 1));

    ASSERT_EQ(1, count);
    ASSERT_EQ(0, record.read);
    ASSERT_EQ(1, record.newly);
    ASSERT_EQ(0, record.displayed);
    ASSERT_EQ(0, record.extra1);

    char wherePhrase[64] = {};
    nn::util::SNPrintf(wherePhrase, sizeof (wherePhrase), "news_id = '%s'", record.newsId.value);

    EXPECT_RESULT(db.Update("news_id", "1", wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("user_id", "1", wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("topic_id", "1", wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("application_ids", "1", wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("received_at", 1, wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("published_at", 1, wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("expire_at", 1, wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("pickup_limit", 1, wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("deletion_priority", 1, wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("age_limit", 1, wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("surprise", 1, wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("bashotorya", 1, wherePhrase), nn::news::ResultNotUpdatable);
    EXPECT_RESULT(db.Update("point", 1, wherePhrase), nn::news::ResultNotUpdatable);

    EXPECT_RESULT_SUCCESS(db.Update("priority", 1, wherePhrase));
    EXPECT_RESULT_SUCCESS(db.Update("read", 1, wherePhrase));
    EXPECT_RESULT_SUCCESS(db.Update("newly", 0, wherePhrase));
    EXPECT_RESULT_SUCCESS(db.Update("displayed", 1, wherePhrase));
    EXPECT_RESULT_SUCCESS(db.Update("opted_in", 1, wherePhrase));
    EXPECT_RESULT_SUCCESS(db.Update("point_status", 1, wherePhrase));
    EXPECT_RESULT_SUCCESS(db.Update("extra_1", 1, wherePhrase));
    EXPECT_RESULT_SUCCESS(db.Update("extra_2", 1, wherePhrase));

    EXPECT_RESULT(db.Update("hoge", 1, wherePhrase), nn::news::ResultUnknownKey);
    EXPECT_RESULT(db.Update("piyo", 1, wherePhrase), nn::news::ResultUnknownKey);

    EXPECT_RESULT_SUCCESS(db.GetList(&count, &record, "", "", 0, 1));

    EXPECT_EQ(1, count);

    EXPECT_EQ(1, record.read);
    EXPECT_EQ(0, record.newly);
    EXPECT_EQ(1, record.displayed);
    EXPECT_EQ(1, record.extra1);
}

void DumpFile(const void* buffer, size_t size) NN_NOEXCEPT
{
    nn::fs::DeleteFile(Path);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateFile(Path, size));

    nn::fs::FileHandle handle = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, Path, nn::fs::OpenMode_Write));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, buffer, size,
        nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
}

void DumpDatabase() NN_NOEXCEPT
{
    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    ASSERT_TRUE(sqlite3_open(Path, &pHandle) == SQLITE_OK);

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        sqlite3_close(pHandle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::news::detail::service::util::Sqlite::SetPerformanceImprovementParameters(pHandle));

    const char* query =
        "SELECT "
            "news_id,"
            "user_id,"
            "topic_id,"
            "application_ids,"
            "received_at,"
            "published_at,"
            "expire_at,"
            "pickup_limit,"
            "priority,"
            "deletion_priority,"
            "age_limit,"
            "surprise,"
            "bashotorya,"
            "point,"
            "read,"
            "newly,"
            "displayed,"
            "opted_in,"
            "point_status,"
            "extra_1,"
            "extra_2"
        " FROM news";

    ASSERT_TRUE(sqlite3_prepare_v2(pHandle, query, -1, &pStatement, nullptr) == SQLITE_OK);

    int result = SQLITE_OK;

    int actualCount = 0;

    NN_LOG("======================================================================\n");
    NN_LOG("news.db\n");

    do
    {
        result = sqlite3_step(pStatement);

        if (result == SQLITE_ROW)
        {
            int index = 0;

            NN_LOG("======================================================================\n");
            NN_LOG("index: %d\n", actualCount);
            NN_LOG("--------------------------------------------------\n");
            NN_LOG("news_id:\n");
            NN_LOG("    - %s\n", reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)));
            NN_LOG("user_id:\n");
            NN_LOG("    - %s\n", reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)));
            NN_LOG("topic_id:\n");
            NN_LOG("    - %s\n", reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)));
            NN_LOG("application_ids:\n");
            NN_LOG("    - %s\n", reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)));
            NN_LOG("received_at:\n");
            NN_LOG("    - %lld\n", sqlite3_column_int64(pStatement, index++));
            NN_LOG("published_at:\n");
            NN_LOG("    - %lld\n", sqlite3_column_int64(pStatement, index++));
            NN_LOG("expire_at:\n");
            NN_LOG("    - %lld\n", sqlite3_column_int64(pStatement, index++));
            NN_LOG("pickup_limit:\n");
            NN_LOG("    - %lld\n", sqlite3_column_int64(pStatement, index++));
            NN_LOG("priority:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("deletion_priority:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("age_limit:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("surprise:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("bashotorya:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("point:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("read:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("newly:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("displayed:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("opted_in:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("point_status:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("extra_1:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("extra_2:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));

            actualCount++;
        }
    }
    while (result == SQLITE_ROW);

    NN_LOG("======================================================================\n");
} // NOLINT(impl/function_size)

TEST(Database, Dump)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHost("news-dump", "C:\\siglo\\news-dump"));

    ASSERT_RESULT_SUCCESS(nn::news::ClearStorage());

    static nn::Bit8 s_Buffer[2 * 1024 * 1024] = {};
    size_t size;

    EXPECT_RESULT(nn::news::GetNewsDatabaseDump(&size, s_Buffer, sizeof (s_Buffer)), nn::news::ResultNotFound);

    ASSERT_RESULT_SUCCESS(nnt::news::PostLocalNews("rom:/news1.msgpack"));
    ASSERT_RESULT_SUCCESS(nnt::news::PostLocalNews("rom:/news2.msgpack"));
    ASSERT_RESULT_SUCCESS(nnt::news::PostLocalNews("rom:/news3.msgpack"));

    EXPECT_RESULT_SUCCESS(nn::news::GetNewsDatabaseDump(&size, s_Buffer, sizeof (s_Buffer)));

    NN_LOG("NewsDatabaseDump: size = %zu\n", size);

    DumpFile(s_Buffer, size);
}

TEST(Database, AddColumn)
{
    sqlite3* pHandle = nullptr;
    sqlite3_stmt* pStatement = nullptr;

    ASSERT_TRUE(sqlite3_open(Path, &pHandle) == SQLITE_OK);

    NN_UTIL_SCOPE_EXIT
    {
        if (pStatement)
        {
            sqlite3_finalize(pStatement);
        }
        sqlite3_close(pHandle);
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::news::detail::service::util::Sqlite::SetPerformanceImprovementParameters(pHandle));

    EXPECT_RESULT_SUCCESS(nn::news::detail::service::util::Sqlite::AddColumn(pHandle, "news", "hoge1", 12345));
    EXPECT_RESULT_SUCCESS(nn::news::detail::service::util::Sqlite::AddColumn(pHandle, "news", "hoge2", "ABC"));

    EXPECT_RESULT(nn::news::detail::service::util::Sqlite::AddColumn(pHandle, "news", "hoge1", 67890),
        nn::news::ResultSqliteErrorError);
    EXPECT_RESULT(nn::news::detail::service::util::Sqlite::AddColumn(pHandle, "news", "hoge2", "DEF"),
        nn::news::ResultSqliteErrorError);

    const char BadTableName[] = {'\xE3', '\x81', '\0'};
    const char BadColumnName[] = {'\x2F', '\xC0', '\xAF', '\0'};
    const char BadDefaultValue[] = {'\xE3', '\x81', '\x82', '\xE3', '\0'};

    EXPECT_RESULT(nn::news::detail::service::util::Sqlite::AddColumn(pHandle, BadTableName, "hoge3", "XYZ"),
        nn::news::ResultInvalidArgument);
    EXPECT_RESULT(nn::news::detail::service::util::Sqlite::AddColumn(pHandle, "news", BadColumnName, "XYZ"),
        nn::news::ResultInvalidArgument);
    EXPECT_RESULT(nn::news::detail::service::util::Sqlite::AddColumn(pHandle, "news", "hoge3", BadDefaultValue),
        nn::news::ResultInvalidArgument);

    EXPECT_EQ(sqlite3_prepare_v2(pHandle, "SELECT news_id,hoge1,hoge2 FROM news", -1, &pStatement, nullptr), SQLITE_OK);

    int result = SQLITE_OK;

    int actualCount = 0;

    NN_LOG("======================================================================\n");
    NN_LOG("news.db\n");

    do
    {
        result = sqlite3_step(pStatement);

        if (result == SQLITE_ROW)
        {
            int index = 0;

            NN_LOG("======================================================================\n");
            NN_LOG("index: %d\n", actualCount);
            NN_LOG("--------------------------------------------------\n");
            NN_LOG("news_id:\n");
            NN_LOG("    - %s\n", reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)));
            NN_LOG("hoge1:\n");
            NN_LOG("    - %d\n", sqlite3_column_int(pStatement, index++));
            NN_LOG("hoge2:\n");
            NN_LOG("    - %s\n", reinterpret_cast<const char*>(sqlite3_column_text(pStatement, index++)));

            EXPECT_EQ(sqlite3_column_int(pStatement, 1), 12345);
            EXPECT_TRUE(nn::util::Strncmp(reinterpret_cast<const char*>(sqlite3_column_text(pStatement, 2)), "ABC", sizeof ("ABC")) == 0);

            actualCount++;
        }
    }
    while (result == SQLITE_ROW);

    NN_LOG("======================================================================\n");
}

TEST(Database, EscapeString)
{
    char escaped[8] = {};

    EXPECT_TRUE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "''", true));
    EXPECT_TRUE(nn::util::Strncmp(escaped, "''''", sizeof (escaped)) == 0);

    EXPECT_TRUE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "''", false));
    EXPECT_TRUE(nn::util::Strncmp(escaped, "''", sizeof (escaped)) == 0);

    EXPECT_TRUE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "\"\"", true));
    EXPECT_TRUE(nn::util::Strncmp(escaped, "\"\"", sizeof (escaped)) == 0);

    EXPECT_TRUE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "\"\"", false));
    EXPECT_TRUE(nn::util::Strncmp(escaped, "\"\"\"\"", sizeof (escaped)) == 0);

    const char BadValue[] = {'\xE3', '\x81', '\0'};
    EXPECT_FALSE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), BadValue));

    EXPECT_TRUE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), ""));
    EXPECT_TRUE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "1234567"));
    EXPECT_TRUE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "'''4"));

    EXPECT_FALSE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "12345678"));
    EXPECT_FALSE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "123456'"));
    EXPECT_FALSE(nn::news::detail::service::util::Sqlite::EscapeString(escaped, sizeof (escaped), "''''"));
}
