﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/socket.h>
#include <curl/curl.h>

#include <nn/netdiag/netdiag_BandWidthApi.h>
#include <nn/netdiag/netdiag_GlobalIpAddressApi.h>
#include <nn/netdiag/netdiag_NatApi.h>
#include <nn/nn_Log.h>

class NetdiagTestSuite : public ::testing::Test
{
protected:

    static void SetUpTestCase()
    {
        NNT_ASSERT_RESULT_SUCCESS( nn::nifm::Initialize() );
        NNT_ASSERT_RESULT_SUCCESS( nn::socket::Initialize(m_SocketConfigWithMemory) );
        curl_global_init(CURL_GLOBAL_DEFAULT);

        m_pNetworkConnection = new nn::nifm::NetworkConnection();
        m_pNetworkConnection->SubmitRequestAndWait();

        NNT_ASSERT_RESULT_SUCCESS(m_pNetworkConnection->GetResult());
        ASSERT_TRUE(m_pNetworkConnection->IsAvailable());
    }

    static void TearDownTestCase()
    {
        delete m_pNetworkConnection;
        m_pNetworkConnection = nullptr;

        curl_global_cleanup();
        NNT_ASSERT_RESULT_SUCCESS( nn::socket::Finalize() );
    }

    /*
    virtual void SetUp(){}
    virtual void TearDown() {}
    */

public:
    static nn::nifm::NetworkConnection* m_pNetworkConnection;
    static nn::socket::ConfigDefaultWithMemory m_SocketConfigWithMemory;

};
nn::nifm::NetworkConnection* NetdiagTestSuite::m_pNetworkConnection = nullptr;
nn::socket::ConfigDefaultWithMemory NetdiagTestSuite::m_SocketConfigWithMemory;

TEST_F(NetdiagTestSuite, NatType)
{
    nn::netdiag::NatType type;
#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS( nn::netdiag::DetectNatType(&type) );
#else
    // Windows は socket がサービスディスカバリ対応していないので
    // NATタイプ判定ができない。実行するだけで結果は確認しない
    nn::Result result = nn::netdiag::DetectNatType(&type);
    NN_UNUSED( result );
#endif
}

TEST_F(NetdiagTestSuite, DownloadBandWidth_DefaultTimeOut)
{
    nn::netdiag::BandWidth bandWidth;
    bool isTimeout = false;
#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS( nn::netdiag::MeasureBandWidthForDownload(&bandWidth, nn::netdiag::TimeoutForDownloadMeasuring, &isTimeout) );
#else
    // Windows では実行するだけで結果は確認しない
    nn::Result result = nn::netdiag::MeasureBandWidthForDownload(&bandWidth, nn::netdiag::TimeoutForDownloadMeasuring, &isTimeout);
    NN_UNUSED( result );
#endif
}

TEST_F(NetdiagTestSuite, DownloadBandWidth_2SecTimeOut)
{
    nn::netdiag::BandWidth bandWidth;
    bool isTimeout = false;
#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS( nn::netdiag::MeasureBandWidthForDownload(&bandWidth, 2 * 1000 /*msec*/, &isTimeout) );
#else
    // Windows では実行するだけで結果は確認しない
    nn::Result result = nn::netdiag::MeasureBandWidthForDownload(&bandWidth, 2 * 1000 /*msec*/, &isTimeout);
    NN_UNUSED( result );
#endif
}

TEST_F(NetdiagTestSuite, DownloadBandWidth_60SecTimeOut)
{
    nn::netdiag::BandWidth bandWidth;
    bool isTimeout = false;
#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS( nn::netdiag::MeasureBandWidthForDownload(&bandWidth, 60 * 1000 /*msec*/, &isTimeout) );
#else
    // Windows では実行するだけで結果は確認しない
    nn::Result result = nn::netdiag::MeasureBandWidthForDownload(&bandWidth, 60 * 1000 /*msec*/, &isTimeout);
    NN_UNUSED( result );

#endif
}

TEST_F(NetdiagTestSuite, UploadBandWidth_DefaultTimeOut)
{
    nn::netdiag::BandWidth bandWidth;
    bool isTimeout = false;
#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS( nn::netdiag::MeasureBandWidthForUpload(&bandWidth, nn::netdiag::TimeoutForUploadMeasuring, &isTimeout) );
    NN_LOG("isTimeout:%s\n", isTimeout ? "true" : "false");
#else
    // Windows では実行するだけで結果は確認しない
    nn::Result result = nn::netdiag::MeasureBandWidthForUpload(&bandWidth, nn::netdiag::TimeoutForUploadMeasuring, &isTimeout);
    NN_UNUSED( result );
#endif
}

TEST_F(NetdiagTestSuite, UploadBandWidth_2SecTimeOut)
{
    nn::netdiag::BandWidth bandWidth;
    bool isTimeout = false;
#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_EXPECT_RESULT_SUCCESS( nn::netdiag::MeasureBandWidthForUpload(&bandWidth, 2 * 1000 /*msec*/, &isTimeout) );
    NN_LOG("isTimeout:%s\n", isTimeout ? "true" : "false");
#else
    // Windows では実行するだけで結果は確認しない
    nn::Result result = nn::netdiag::MeasureBandWidthForUpload(&bandWidth, 2 * 1000 /*msec*/, &isTimeout);
    NN_UNUSED( result );
#endif
}

TEST_F(NetdiagTestSuite, UploadBandWidth_60SecTimeOut)
{
    nn::netdiag::BandWidth bandWidth;
    bool isTimeout = false;
#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_EXPECT_RESULT_SUCCESS( nn::netdiag::MeasureBandWidthForUpload(&bandWidth, 60 * 1000 /*msec*/, &isTimeout) );
    NN_LOG("isTimeout:%s\n", isTimeout ? "true" : "false");
#else
    // Windows では実行するだけで結果は確認しない
    nn::Result result = nn::netdiag::MeasureBandWidthForUpload(&bandWidth, 60 * 1000 /*msec*/, &isTimeout);
    NN_UNUSED( result );
#endif
}

TEST_F(NetdiagTestSuite, GlobalIpAddr)
{
    nn::netdiag::GlobalIpAddress addr;
    nn::Result result;
#ifndef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS( (result = nn::netdiag::GetGlobalIpAddress(&addr)) );

    if ( result.IsSuccess() )
    {
        NN_LOG( "IP Address: %s\n", addr.value );
    }
#else
    // Windows では実行するだけで結果は確認しない
    result = nn::netdiag::GetGlobalIpAddress(&addr);
    NN_UNUSED( result );
#endif
}
