﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include "NetTest_Port.h"

#include "Tests/CurlHttpUploadDownloadTest.h"
#include "Utils/CommandLineParser.h"
#include "../Common/ParseHelpers.h"

#include <nnt/nntest.h>
#include <curl/curl.h>

namespace
{
    const uint32_t WaitTypeBufLen  = 16;
    const uint32_t ProxyAuthBufLen = 32;
    const uint32_t CacheModeBufLen = 8;
}

// CurlHttpUploadDownload
TEST(natf, CurlHttpUploadDownload)
{
    bool isSuccess;
    NATF::Utils::ParserGroup parser;
    NATF::Modules::LibCurl::Params params;
    int argc;
    uint32_t timeoutMs = 0;
    uint32_t uploadSizeMb = 0;
    uint32_t defaultRepeatCount = 1;
    uint32_t repeatCount = defaultRepeatCount;
    bool defaultDoInit = true;
    bool doInitEvery = defaultDoInit;
    bool defaultDoVerifyPeer = true;
    bool defaultDoVerifyHostName = true;
    bool defaultDoVerifyDate = true;
    bool defaultTeamCityDown = false;
    bool defaultTeamCityUp = false;
    uint16_t defaultProxyPort = 0;
    const char * const * pArgv;

    nn::util::Uuid netProfile;
    char pUploadUrl[NATF::Modules::LibCurl::UrlBufferLen];
    char pDownloadUrl[NATF::Modules::LibCurl::UrlBufferLen];
    char pTestName[NATF::BaseTest::NameBufferLen];
    char pWaitType[WaitTypeBufLen];
    char pProxyAuth[ProxyAuthBufLen];
    char pCacheModeStr[CacheModeBufLen];
    const char* pDefaultWaitType = "Select";
    const char* pDefaultProxyServer = "";
    const char* pDefaultProxyCredentials = "";
    const char* pDefaultProxyAuth = "NEGOTIATE";
    const char* pDefaultCacheMode = "NONE";

    NN_NETTEST_LOG("\nEnter NATF Process\n\n");

    NETTEST_GET_ARGS(argc, pArgv);

    parser.AddParser(NATF::Utils::StringParser ("--Name", nullptr, pTestName, sizeof(pTestName)));
    parser.AddParser(NATF::Utils::UuidParser   ("--NetProfile", &nn::util::InvalidUuid, netProfile));
    parser.AddParser(NATF::Utils::StringParser ("--UploadUrl", nullptr, pUploadUrl, sizeof(pUploadUrl)));
    parser.AddParser(NATF::Utils::StringParser ("--DownloadUrl", nullptr, pDownloadUrl, sizeof(pDownloadUrl)));
    parser.AddParser(NATF::Utils::UInt32Parser ("--UploadSizeMb", nullptr, uploadSizeMb));
    parser.AddParser(NATF::Utils::UInt32Parser ("--TimeoutMs", nullptr, timeoutMs));
    parser.AddParser(NATF::Utils::Md5HashParser("--Md5Hash", nullptr, params.expectedHash));
    parser.AddParser(NATF::Utils::StringParser ("--WaitType", pDefaultWaitType, pWaitType, sizeof(pWaitType)));
    parser.AddParser(NATF::Utils::UInt32Parser ("--RepeatCount", &defaultRepeatCount, repeatCount));
    parser.AddParser(NATF::Utils::BoolParser   ("--DoInit", &defaultDoInit, doInitEvery));
    parser.AddParser(NATF::Utils::BoolParser   ("--VerifyPeer", &defaultDoVerifyPeer, params.doVerifyPeer));
    parser.AddParser(NATF::Utils::BoolParser   ("--VerifyHostName", &defaultDoVerifyHostName, params.doVerifyHostname));
    parser.AddParser(NATF::Utils::BoolParser   ("--VerifyDate", &defaultDoVerifyDate, params.doVerifyDate));
    parser.AddParser(NATF::Utils::StringParser ("--SessionCacheMode", pDefaultCacheMode, pCacheModeStr, sizeof(pCacheModeStr)));
    parser.AddParser(NATF::Utils::BoolParser   ("--LogTeamCityDownSpeed", &defaultTeamCityDown, params.publishDownSpeedToTeamCity));
    parser.AddParser(NATF::Utils::BoolParser   ("--LogTeamCityUpSpeed", &defaultTeamCityUp, params.publishUpSpeedToTeamCity));
    parser.AddParser(NATF::Utils::StringParser ("--ProxyServer", pDefaultProxyServer, params.pProxyServer, sizeof(params.pProxyServer)));
    parser.AddParser(NATF::Utils::StringParser ("--ProxyCredentials", pDefaultProxyCredentials, params.pProxyUserPwd, sizeof(params.pProxyUserPwd)));
    parser.AddParser(NATF::Utils::StringParser ("--ClientCert", "", params.pClientCertPath, sizeof(params.pClientCertPath)));
    parser.AddParser(NATF::Utils::StringParser ("--ServerCert", "", params.pServerCertPath, sizeof(params.pServerCertPath)));
    parser.AddParser(NATF::Utils::UInt16Parser ("--ProxyPort", &defaultProxyPort, params.proxyPort));
    parser.AddParser(NATF::Utils::StringParser ("--ProxyAuth", pDefaultProxyAuth, pProxyAuth, sizeof(pProxyAuth)));
    parser.AddParser(NATF::Utils::StringParser ("--HttpHeaderOption", "", params.pHttpHeaderOption, sizeof(params.pHttpHeaderOption)));

    if( !parser.Parse(argc, pArgv) )
    {
        NN_NETTEST_LOG(" * Failed to parse command line arguements!\n\n");
        EXPECT_EQ(false, true);
        return;
    }

    NetTest::StrUpr(pWaitType);
    if(nnt::ParserHelpers::GetWaitTypeFromString(pWaitType, WaitTypeBufLen, params.useSelect) == false)
    {
        FAIL();
        return;
    }

    if( strncmp(pProxyAuth, "", sizeof(pProxyAuth)) != 0 )
    {
        NetTest::StrUpr(pProxyAuth);
        if(nnt::ParserHelpers::GetProxyAuthFromString(pProxyAuth, sizeof(pProxyAuth), params.proxyAuthMethod) == false)
        {
            FAIL();
            return;
        }
    }

    NetTest::StrUpr(pCacheModeStr);
    if(nnt::ParserHelpers::GetCacheModeFromString(pCacheModeStr, sizeof(pCacheModeStr), params.sessionCacheMode) == false)
    {
        FAIL();
        return;
    }

    params.uploadSize = uploadSizeMb * 1024 * 1024;

    NATF::Utils::InitApi initApi( !doInitEvery * NATF::Tests::CurlHttpUploadDownload::InitFlags );

    if( !initApi.Init(netProfile) )
    {
        NN_NETTEST_LOG(" Failed to inititalize test!\n\n");
        EXPECT_EQ(false, true);
        return;
    }

    for(uint32_t i = 0; i < repeatCount; ++i)
    {
        NATF::Tests::CurlHttpUploadDownload curlHttpUploadDownload(pTestName, netProfile, timeoutMs, doInitEvery, pDownloadUrl, pUploadUrl, params);

        isSuccess = curlHttpUploadDownload.Run();
        EXPECT_EQ(isSuccess, true);
    }

    NN_NETTEST_LOG("\nExit NATF Process\n\n");
}  // NOLINT(impl/function_size)
