﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <map>
#include <nn/nn_Log.h>

namespace {

    const uint32_t       MemoryPoolSize = (512 * 1024);
    const uint32_t       ThreadStackSize = (64 * 1024);
    nn::lmem::HeapHandle g_HeapHandle;
    uint8_t              g_HeapMemoryPool[MemoryPoolSize] NN_ALIGNAS(nn::os::ThreadStackAlignment);

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    void* AllocAligned(size_t size, size_t align)
    {
        return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size, align);
    }

    void Free(void* p)
    {
        nn::lmem::FreeToExpHeap(g_HeapHandle, p);
    }
}

void CleanUpThread(void * data);

class DhcpTestHarness
{
public:
    DhcpTestHarness() : exiting(false), m_CleanupThreadStack(nullptr)
    {

    }

    struct ThreadData
    {
        ThreadData(nn::os::ThreadType * thread, void * stack, void * data) : thread(thread), stack(stack), data(data){}
        nn::os::ThreadType * thread;
        void * stack;
        void * data;
    };

    void Initialize()
    {
        m_CleanupThreadStack = AllocAligned(ThreadStackSize, nn::os::ThreadStackAlignment);
        nn::os::CreateThread(&m_CleanupThread, CleanUpThread, this, m_CleanupThreadStack, ThreadStackSize, 0);

        nn::htcs::Initialize(&Allocate, &Deallocate);
        nn::os::InitializeMutex(&m_ThreadDataMutex, true, 0);

        nn::os::StartThread(&m_CleanupThread);


    }
    void Finalize()
    {
        nn::os::DestroyThread(&m_CleanupThread);
        Free(m_CleanupThreadStack);

        nn::htcs::Finalize();
        nn::os::FinalizeMutex(&m_ThreadDataMutex);

    }


    nn::Result CreateThread(nn::os::ThreadType ** thread, nn::os::ThreadFunction function, void * data = nullptr, int dataSize = 0)
    {
        nn::os::LockMutex(&m_ThreadDataMutex);
            //Alocate Thread Stack
            void * stack = AllocAligned(ThreadStackSize, nn::os::ThreadStackAlignment);

            //Copy User Data
            void * userData = nullptr;
            if (data != nullptr && dataSize > 0)
            {
                userData = AllocAligned(dataSize,16);
                if (userData)
                {
                    memcpy(userData, data, dataSize);
                }
            }
            //Create Thread
            void * threadMem = AllocAligned(sizeof(nn::os::ThreadType),16);
            *thread = new (threadMem) nn::os::ThreadType;

            nn::Result res = nn::os::CreateThread(*thread, function, userData, stack, ThreadStackSize, 0);


            //New Thread data added to thread data vec
            m_ThreadDataMap[*thread] = new ThreadData(*thread, stack, userData);
            //Set User Data to the thread

            //Start Thread
            nn::os::StartThread(*thread);
        nn::os::UnlockMutex(&m_ThreadDataMutex);
        return res;
    }

    void DestroyThread(nn::os::ThreadType * thread)
    {
        nn::os::LockMutex(&m_ThreadDataMutex);

            //Destroy Thread
            nn::os::DestroyThread(thread);
            //Free thread stack
            Free(m_ThreadDataMap.at(thread)->stack);

            if (m_ThreadDataMap.at(thread)->data)
            {
                Free(m_ThreadDataMap.at(thread)->data);
            }

            //Delete thread data
            delete (m_ThreadDataMap.at(thread));
            //Erase thread data map entry
            m_ThreadDataMap.erase(thread);
            //delete thread;
            thread->~ThreadType();
            Free(thread);
        nn::os::UnlockMutex(&m_ThreadDataMutex);

    }

    void WaitForCommandCompletion()
    {
        while (true)
        {
            nn::os::LockMutex(&m_ThreadDataMutex);
            if (m_ThreadDataMap.size() == 0)
            {
                nn::os::UnlockMutex(&m_ThreadDataMutex);
                break;
            }
            nn::os::UnlockMutex(&m_ThreadDataMutex);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }

    }

    void CleanUpDeadThreads()
    {
        while (!exiting || !m_ThreadDataMap.empty())
        {
            nn::os::LockMutex(&m_ThreadDataMutex);
            if (m_ThreadDataMap.size() > 0)
            {
                auto iter = m_ThreadDataMap.begin();
                auto iterEnd = m_ThreadDataMap.end();
                nn::os::ThreadType * thread = nullptr;
                for (; iter != iterEnd; ++iter)
                {
                    if (iter->first->_state == nn::os::ThreadType::State_Exited)
                    {
                        thread = iter->first;
                        break;
                    }
                }


                if (thread != nullptr)
                {
                    nn::os::UnlockMutex(&m_ThreadDataMutex);
                    DestroyThread(thread);
                    continue;
                }
                else
                {

                    nn::os::UnlockMutex(&m_ThreadDataMutex);
                }

            }
            else
            {

                nn::os::UnlockMutex(&m_ThreadDataMutex);
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
    }
private:
    nn::os::MutexType m_ThreadDataMutex;
    std::map<nn::os::ThreadType *,ThreadData *> m_ThreadDataMap;

    bool exiting;

    nn::os::ThreadType m_CleanupThread;
    void * m_CleanupThreadStack;

};
