﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstdio>
#include <memory>

#include <nn/os.h>
#include <nn/htcs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/init.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>
#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/nifm.h>
#include "DhcpTestHarness.h"

namespace
{
    DhcpTestHarness * g_testHarness;
    const int udpTestSendRecvNum = 4000;

}
enum
{
    IP4_HDRLEN = 20,      // IPv4 header length
    ICMP_HDRLEN = 8         // ICMP header length for echo request, excludes data
};

struct EchoRequestInfo
{
    nn::socket::SockAddrIn destAddr;
    char interfaceName[nn::socket::IfNamSiz];
    uint32_t count;
    uint32_t dataSize;
};


void CommandSetup(void * data)
{

}
void CommandConfigure(void * data)
{
    nn::bsdsocket::cfg::CancelIf("usb0");
    nn::bsdsocket::cfg::SetIfDown("usb0");
}
void CommandOption(void * data)
{

}
void CommandBoot(void * data)
{
    nn::bsdsocket::cfg::IfSettings ifcfg;
    memset(&ifcfg, 0, sizeof(ifcfg));
    ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
    ifcfg.mtu  = 1500;
    ifcfg.duplicateIpWaitTime = 2000;
    nn::Result result = nn::bsdsocket::cfg::SetIfUp("usb0", &ifcfg);
    if(result.IsFailure())
    {
        NN_LOG("Failed to boot interface %s - %d:%d\n",
                "usb0",
                result.GetModule(),
                result.GetDescription());
    }

}
void CommandRelease(void * data)
{
    nn::bsdsocket::cfg::SetIfDown("usb0", nn::bsdsocket::cfg::ShutdownOption_Release);
}

void CommandReset(void * data)
{
    nn::bsdsocket::cfg::CancelIf("usb0");
    nn::bsdsocket::cfg::SetIfDown("usb0");

}
void CommandReboot(void * data)
{

    nn::bsdsocket::cfg::IfState interfaceState;
    nn::bsdsocket::cfg::GetIfState("usb0",&interfaceState);

    nn::bsdsocket::cfg::CancelIf("usb0");
    nn::bsdsocket::cfg::SetIfDown("usb0");

    nn::bsdsocket::cfg::IfSettings ifcfg;
    memset(&ifcfg, 0, sizeof(ifcfg));

    ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_DhcpReboot;
    ifcfg.mtu  = 1500;
    ifcfg.duplicateIpWaitTime = 2000;
    ifcfg.u.modeDhcpReboot.requestedAddr = interfaceState.addr;

    nn::Result result = nn::bsdsocket::cfg::SetIfUp("usb0", &ifcfg);
    if(result.IsFailure())
    {
        NN_LOG("Failed to boot interface %s - %d:%d\n",
                "usb0",
                result.GetModule(),
                result.GetDescription());
    }
}
void CommandRenew(void * data)
{
    nn::bsdsocket::cfg::IfState interfaceState;
    nn::bsdsocket::cfg::GetIfState("usb0", &interfaceState);

    nn::bsdsocket::cfg::CancelIf("usb0");
    nn::bsdsocket::cfg::SetIfDown("usb0");

    nn::bsdsocket::cfg::IfSettings ifcfg;
    memset(&ifcfg, 0, sizeof(ifcfg));

    ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_DhcpReboot;
    ifcfg.mtu = 1500;
    ifcfg.duplicateIpWaitTime = 2000;
    ifcfg.u.modeDhcpReboot.requestedAddr = interfaceState.addr;
    ifcfg.u.modeDhcpReboot.serverAddr = interfaceState.serverAddr;

    nn::Result result = nn::bsdsocket::cfg::SetIfUp("usb0", &ifcfg);
    if (result.IsFailure())
    {
        NN_LOG("Failed to boot interface %s - %d:%d\n",
            "usb0",
            result.GetModule(),
            result.GetDescription());
    }
}
void CommandStaticIpAssign(void * data)
{

}
void CommandDhcpInformSend(void * data)
{

}

// Compute the internet checksum (RFC 1071).

static uint16_t Checksum(uint16_t *addr, int len)
{
    int count = len;
    uint32_t sum = 0;
    uint16_t answer = 0;

    // Sum up 2-byte values until none or only one byte left.
    while (count > 1) {
        sum += *(addr++);
        count -= 2;
    }

    // Add left-over byte, if any.
    if (count > 0) {
        sum += *(uint8_t *)addr;
    }

    // Fold 32-bit sum into 16 bits; we lose information by doing this,
    //// increasing the chances of a collision.
    // sum = (lower 16 bits) + (upper 16 bits shifted right 16 bits)
    while (sum >> 16) {
        sum = (sum & 0xffff) + (sum >> 16);
    }

    // Checksum is one's compliment of sum.
    answer = ~sum;

    return (answer);
}

int initialized = 0;
namespace
{
    char NN_ALIGNAS(4096) g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
}

void CommandEchoRequestSend(void * data)
{

    nn::socket::Icmp icmpHeader;
    nn::bsdsocket::cfg::IfState ifState;
    const int maxPacketSize = 1024;
    uint8_t packet[maxPacketSize];

    //get echo request info that was passed to us
    EchoRequestInfo * pInfo = (EchoRequestInfo *)data;


    int datalen = pInfo->dataSize;
    if (ICMP_HDRLEN + datalen > maxPacketSize)
    {
        NN_LOG("ERROR!, Echo data length cannot exceed %d, datalen = %d", maxPacketSize - (IP4_HDRLEN + ICMP_HDRLEN), datalen);
        return;
    }

    if (!initialized)
    {
        nn::socket::Initialize(g_SocketMemoryPoolBuffer, nn::socket::DefaultSocketMemoryPoolSize, nn::socket::DefaultSocketAllocatorSize, 4);
        initialized = 1;
    }


    // ICMP header

    // Message Type (8 bits): echo request
    icmpHeader.icmp_type = nn::socket::IcmpType::Icmp_Echo;

    // Message Code (8 bits): echo request
    icmpHeader.icmp_code = static_cast<nn::socket::IcmpCode>(0);

    // Identifier (16 bits): usually pid of sending process - pick a number
    icmpHeader.icmp_id = nn::socket::InetHtons(1000);

    // Prepare packet.

    for (int i = 0; i < pInfo->count; ++i)
    {
      //  NN_LOG("C Interface name: %s\n", pInfo->interfaceName);

        // Sequence Number (16 bits): starts at 0
        icmpHeader.icmp_seq = nn::socket::InetHtons(i);

        // Next part of packet is upper layer protocol header.
        memcpy((packet), &icmpHeader, ICMP_HDRLEN);

        // Finally, add the ICMP data.
        memset(packet + ICMP_HDRLEN, 0xAF, datalen);

        // ICMP header checksum (16 bits): set to 0 when calculating checksum
        icmpHeader.icmp_cksum = 0;
        // Calculate ICMP header checksum
        icmpHeader.icmp_cksum = Checksum((uint16_t *)(packet), ICMP_HDRLEN + datalen);
        memcpy((packet), &icmpHeader, ICMP_HDRLEN);

        // Submit request for a raw socket descriptor.
        int socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Raw, nn::socket::Protocol::IpProto_Icmp);
        if (socket < 0)
        {
            NN_LOG("FAILED nn::socket::Type::Sock_Raw Creation!\n");
            return;
        }

        // Send packet.
        if (nn::socket::SendTo(socket, packet, ICMP_HDRLEN + datalen, nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr *>(&pInfo->destAddr), sizeof(nn::socket::SockAddr)) < 0) {
            NN_LOG("sendto() failed,  Error: %d\n", nn::socket::GetLastError());
            return;
        }

        nn::socket::Close(socket);
    }



}

void CommandRestartLogCheck(void * data)
{

}

void UdpTestServer(void * data)
{


    int fd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    int port = 6000;
    int broadcastEnable = 1;

    NN_LOG("UdpTestServer THREAD POINTER VALUE: 0x%x\n", nn::os::GetCurrentThread());
    if (fd < 0)
    {
        NN_LOG("FAIL Server Socket!\n");
    }
    int res = nn::socket::SetSockOpt(fd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast,
        &broadcastEnable, sizeof(broadcastEnable));
    if (res < 0)
    {
        NN_LOG("FAIL Server SetSockOpt!\n");
    }

    nn::socket::SockAddrIn addr;
    memset(&addr, 0, sizeof(addr));
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(port);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;

    res = nn::socket::Bind(fd, reinterpret_cast<nn::socket::SockAddr *>(&addr), sizeof(addr));
    if (res < 0)
        NN_LOG("FAIL Server Bind!\n");
    nn::socket::SockAddrIn remoteAddr;
    int i = 0;
    char buf[1024];
    int total = 0;
    while (i <= udpTestSendRecvNum)
    {
        unsigned slen = sizeof(nn::socket::SockAddr);
        res = nn::socket::RecvFrom(fd, buf, 1024 - 1, nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), &slen);

        if (res < 0)
            NN_LOG("FAIL Server RecvFrom!\n");
        total += res;
        if(i % 50 == 0)
        {
            NN_LOG("Received Broadcast #%i, total bytes: %d\n",i, total);
        }
        if ((i + 1) % 2001 == 0)
        {
            NN_LOG("Shutdown all sockets, and see what happens. \n");
            int res = nn::socket::ShutdownAllSockets(false);
            NN_LOG("Shutdown %d sockets\n", res);
        }
        ++i;
        //nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    //delete [] buf;


}



void UdpTest(void * data)
{
    NN_LOG("Initialize NIFM\n");
    nn::nifm::Initialize();
    nn::nifm::SubmitNetworkRequestAndWait();
    NN_LOG("Initialize Socket with size Memory pool size %d\n",nn::socket::DefaultSocketMemoryPoolSize);
    NN_LOG("UdpTest THREAD POINTER VALUE: 0x%x\n", nn::os::GetCurrentThread());
    if (!initialized)
    {
        nn::socket::Initialize(g_SocketMemoryPoolBuffer, nn::socket::DefaultSocketMemoryPoolSize, nn::socket::DefaultSocketAllocatorSize, 4);
    }

    nn::os::ThreadType * thread = nullptr;
    NN_LOG("Create Test Receive Server\n");
    g_testHarness->CreateThread(&thread, UdpTestServer, nullptr);

    NN_LOG("Create Socket\n");
    int fd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    if (fd < 0)
        NN_LOG("FAIL Socket!\n");

    int broadcastEnable = 1;
    char buf[1024];

    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(6000);
    nn::socket::InetPton(nn::socket::Family::Af_Inet, "192.168.0.255", &addr.sin_addr);
    NN_LOG("Set Socket Broadcast\n");
    int res = nn::socket::SetSockOpt(fd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast,
        &broadcastEnable, sizeof(broadcastEnable));
    if(res < 0)
       NN_LOG("FAIL SETSOCKOPT!\n");
    NN_LOG("Begin Send\n");

    for(int i = 0; i <= udpTestSendRecvNum; ++i)
    {
       int res = nn::socket::SendTo(fd, buf, sizeof(buf), nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr *>(&addr), sizeof(addr));
       if (res < 0)
       {
           NN_LOG("FAIL SendTo! Error Code: %d \n", nn::socket::GetLastError());

       }
       if (i % 1000 == 0)
       {
           NN_LOG("Sent Broadcast Packet #%i\n", i);
       }
    }


}



void CleanUpThread(void * data)
{
    ((DhcpTestHarness *)data)->CleanUpDeadThreads();
}




void ProccessCommand(char * commandBuff)
{
    //copy command
    char * currentBegin = strtok(commandBuff,"\n");
    if(currentBegin == nullptr)
    {
        NN_LOG("ERROR No Command Sent!\n");
        return;
    }
    size_t commandLength = strlen(currentBegin);
    char * command  = new char[commandLength + 1];
    memcpy(command,currentBegin,commandLength);
    command[commandLength] = 0;

    // copy host name
    currentBegin = strtok(nullptr,"\n");
    if(currentBegin == nullptr)
    {
        NN_LOG("ERROR No Hostname Sent!\n");
        return;
    }
    unsigned hostNameLength = strlen(currentBegin);
    char * hostName  = new char[hostNameLength + 1];
    memcpy(hostName,currentBegin,hostNameLength);
    hostName[hostNameLength] = 0;

    NN_LOG("Recieved Command: %s For Hostname: %s\n", command, hostName);

    if(strcmp(command,"setup") == 0)
    {  }
    else if(strcmp(command,"configure") == 0)
    {
        g_testHarness->WaitForCommandCompletion();
        nn::os::ThreadType * thread = nullptr;
        g_testHarness->CreateThread(&thread, CommandConfigure, nullptr);
    }
    else if(strcmp(command,"option") == 0)
    {  }
    else if(strcmp(command,"boot") == 0)
    {
        g_testHarness->WaitForCommandCompletion();
        nn::os::ThreadType * thread = nullptr;
        g_testHarness->CreateThread(&thread,CommandBoot,nullptr);
    }
    else if(strcmp(command,"release") == 0)
    {
        g_testHarness->WaitForCommandCompletion();
        nn::os::ThreadType * thread = nullptr;
        g_testHarness->CreateThread(&thread, CommandRelease, nullptr);
    }
    else if(strcmp(command,"renew") == 0)
    {
        g_testHarness->WaitForCommandCompletion();
        nn::os::ThreadType * thread = nullptr;
        g_testHarness->CreateThread(&thread, CommandRenew, nullptr);
    }
    else if(strcmp(command,"reboot") == 0)
    {
        g_testHarness->WaitForCommandCompletion();
        nn::os::ThreadType * thread = nullptr;
        g_testHarness->CreateThread(&thread, CommandReboot, nullptr);
    }
    else if(strcmp(command,"reset") == 0)
    {
        nn::os::ThreadType * thread = nullptr;
        g_testHarness->CreateThread(&thread, CommandReset, nullptr);
    }
    else if(strcmp(command,"staticipassign") == 0)
    { }
    else if(strcmp(command,"dhcpinformsend") == 0)
    { }
    else if(strcmp(command,"echorequestsend") == 0)
    {
        currentBegin = strtok(nullptr, "\n");
        if (currentBegin == nullptr)
        {
            NN_LOG("ERROR No Destination Sent!\n");
            return;
        }
        EchoRequestInfo info;
        memset(&info.destAddr, 0, sizeof(info.destAddr));
        info.destAddr.sin_family = nn::socket::Family::Af_Inet;
        nn::socket::InetPton(nn::socket::Family::Af_Inet, currentBegin, &info.destAddr.sin_addr);

        currentBegin = strtok(nullptr, "\n");
        if (currentBegin == nullptr)
        {
            NN_LOG("ERROR No Interface Name Sent!\n");
            return;
        }

        unsigned length = strlen(currentBegin);
        if (length >= nn::socket::IfNamSiz - 3)
        {
            NN_LOG("ERROR Interface Name Too Long!\n");
            return;
        }
        memcpy(info.interfaceName, "usb", 3);
        memcpy(info.interfaceName + 3, currentBegin, length);
        info.interfaceName[length + 3] = 0;

        currentBegin = strtok(nullptr, "\n");
        if (currentBegin == nullptr)
        {
            NN_LOG("ERROR No Repeat Count Sent!\n");
            return;
        }

        sscanf(currentBegin, "%d", &info.count);


        currentBegin = strtok(nullptr, "\n");
        if (currentBegin == nullptr)
        {
            NN_LOG("ERROR No Data Size Sent!\n");
            return;
        }

        sscanf(currentBegin, "%d", &info.dataSize);

        g_testHarness->WaitForCommandCompletion();
        nn::os::ThreadType * thread = nullptr;
        g_testHarness->CreateThread(&thread, CommandEchoRequestSend, &info,sizeof(info));
    }
    else if(strcmp(command,"restartlogcheck") == 0)
    {

    }
    else if (strcmp(command, "udptest") == 0)
    {
        nn::os::ThreadType * thread = nullptr;
        g_testHarness->CreateThread(&thread, UdpTest, nullptr);
    }
    else
    {
        NN_LOG("ERROR Command Not Recognized: %s",command);
    }
    delete [] command;
    delete [] hostName;
} //NOLINT(impl/function_size)


void ReceiveCommands(int sock)
{
    char buf[1028];
    nn::htcs::ssize_t receivedBytes = nn::htcs::Recv(sock, buf, sizeof(buf) - 1, 0);
    if (receivedBytes < 0)
    {
        NN_LOG("Recv error %d\n", nn::htcs::GetLastError());
        return;
    }
    else if (receivedBytes == 0)
    {
        NN_LOG("gracefully closed\n");
        return;
    }

    buf[receivedBytes] = '\0';
    ProccessCommand(buf);

    nn::htcs::ssize_t sentBytes = nn::htcs::Send(sock, buf, receivedBytes, 0);
    if (sentBytes < 0)
    {
        NN_LOG("send error %d\n", nn::htcs::GetLastError());
    }
}

void DhcpTestControllerServer()
{

    nn::htcs::SockAddrHtcs serviceAddr;
    serviceAddr.family = nn::htcs::HTCS_AF_HTCS;
    serviceAddr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(serviceAddr.portName.name, "DhcpTestController");

    int serviceSocket = nn::htcs::Socket();
    nn::htcs::Bind(serviceSocket, &serviceAddr);

    int res = nn::htcs::Listen(serviceSocket, 20);
    if(res == -1)
    {
        NN_LOG("Command Connection Listen Error! error num: %d",nn::htcs::GetLastError());
    }
    NN_LOG("Dhcp Test Harness Waiting For Commands.\n");
    while(true)
    {
        // 接続を待ち受ける

        int sock = nn::htcs::Accept(serviceSocket, nullptr);
        if(sock == -1)
        {
             NN_LOG("Command Connection Accept Error! error num: %d",nn::htcs::GetLastError());
             continue;
        }


        // エコーバックテスト
        ReceiveCommands(sock);

        // 切断
        nn::htcs::Close(sock);
    }

    nn::htcs::Close(serviceSocket);
}

extern "C"
void nninitStartup()
{
    const size_t MemoryHeapSize = 8 * 1024 * 1024;
    nn::os::SetMemoryHeapSize(MemoryHeapSize);

    uintptr_t address;
    nn::os::AllocateMemoryBlock(&address, MemoryHeapSize);

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MemoryHeapSize);

    g_HeapHandle =
        nn::lmem::CreateExpHeap(
        g_HeapMemoryPool,
        sizeof(g_HeapMemoryPool),
        nn::lmem::CreationOption_NoOption
        );

}


// TORIAEZU: ソケット型
extern "C" void nnMain()
{
    void * testHarnessMem = AllocAligned(sizeof(DhcpTestHarness),16);
    g_testHarness = new (testHarnessMem) DhcpTestHarness();

    g_testHarness->Initialize();

    DhcpTestControllerServer();

    g_testHarness->Finalize();

    Free(g_testHarness);

    return;
}
