﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 Test process for Network
 *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/socket.h>
#include <nn/nn_Log.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiForTest.h>

#include "testNet_ApiCommon.h"
#include "natf/Utils/CommandLineParser.h" // NATF::Utils::ParserGroup
#include "natf/natf.h" // NATF::BaseTest

namespace {

template <size_t MemoryPoolSize>
class TestConfigWithMemory
    : public nn::socket::ConfigDefault
{
public:
    NN_IMPLICIT TestConfigWithMemory() NN_NOEXCEPT
        : nn::socket::ConfigDefault(m_MemoryPool,
                                    sizeof(m_MemoryPool),
                                    MemoryPoolSize / 2)
    {
    }

    NN_IMPLICIT TestConfigWithMemory(int concurrencyCountMax) NN_NOEXCEPT
        : nn::socket::ConfigDefault(m_MemoryPool,
                                    sizeof(m_MemoryPool),
                                    MemoryPoolSize / 2)
    {
        SetConcurrencyCountMax(concurrencyCountMax);
    }

private:
#if defined NN_BUILD_CONFIG_OS_WIN32
    uint8_t m_MemoryPool[1];
#else
    uint8_t m_MemoryPool[MemoryPoolSize] NN_ALIGNAS(nn::socket::MemoryPoolAlignment);
#endif
};

    TestConfigWithMemory< 16 * 1024 * 1024> g_SocketConfigWithMemory;
    int g_TestTeardownOptions = -1;
    nn::nifm::NetworkConnection *g_pNifmConnection = nullptr;

} // namespace unnamed

namespace NATF {
namespace API {

    bool ParseOptions(char* testName, nn::util::Uuid* pNetProfile) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::Result rval;

        NATF::Utils::ParserGroup parser;
        nn::util::Uuid netProfileLocal;
        char testNameLocal[NATF::BaseTest::NameBufferLen];

        int argC;
        const char * const * argV;

        ERROR_IF(nullptr == testName, "TestSetup: testName nullptr");
        ERROR_IF(nullptr == pNetProfile, "TestSetup: pNetProfile nullptr");

        NETTEST_GET_ARGS(argC, argV);

        NN_LOG("ParseOptions: argC=%d\n", argC);
        for( int i = 0; i < argC; ++i )
        {
            NN_LOG("ParseOptions: argV[%d]=%s\n", i, argV[i]);
        }

        parser.AddParser(NATF::Utils::StringParser  ("--Name", nullptr, testNameLocal, sizeof(testNameLocal)));
        parser.AddParser(NATF::Utils::UuidParser    ("--NetProfile", &nn::util::InvalidUuid, netProfileLocal));

        ERROR_IF( !parser.Parse(argC, argV), "Failed to parse command line arguments");

        strcpy(testName, testNameLocal);
        *pNetProfile = netProfileLocal;

out:
        return isSuccess;
    }

    bool TestSetup(unsigned int testSetupOptionsBitfield) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::Result rval;

        nn::util::Uuid netProfile;
        char testName[NATF::BaseTest::NameBufferLen];

        ::g_TestTeardownOptions = 0;
        ::g_pNifmConnection = nullptr;

        ERROR_IF(!ParseOptions(testName, &netProfile), "TestSetup: ParseOptions failed");

        NN_LOG("TestSetup: testName=%s\n", testName);

        // Setup nifm
        if( 0 != (testSetupOptionsBitfield & TestSetupOptions_Nifm) )
        {
            /* since the following will utilize the system network interface, we must initialize
               network interface manager (NIFM) */
            PRINT_AND_CALL(rval = nn::nifm::Initialize());
            ERROR_IF(rval.IsFailure(), "nn::nifm::Initialize failed");

            PRINT_AND_CALL(rval = nn::nifm::SetExclusiveClient(nn::nifm::GetClientId()));
            ERROR_IF(rval.IsFailure(), "nn::nifm::SetExclusiveClient failed");

            ::g_pNifmConnection = new nn::nifm::NetworkConnection;
            ERROR_IF(nullptr == ::g_pNifmConnection, "nn::nifm::NetworkConnection constructor failed");

            nn::nifm::RequestHandle requestHandle = ::g_pNifmConnection->GetRequestHandle();
            nn::nifm::SetRequestConnectionConfirmationOption(requestHandle,nn::nifm::ConnectionConfirmationOption_Prohibited);
            nn::nifm::SetRequestPersistent(requestHandle, true);

            if( nn::util::InvalidUuid != netProfile )
            {
                char uuidBuffer[nn::util::Uuid::StringSize];
                NN_LOG("Setting Uuid.\n");
                PRINT_AND_CALL(rval = nn::nifm::SetRequestNetworkProfileId(requestHandle, netProfile));
                ERROR_IF(0 > sprintf(uuidBuffer, "Failed to set network profile id! Uuid: %s", netProfile.ToString(uuidBuffer, sizeof(uuidBuffer))), "TestSetup: sprint failed");
                ERROR_IF(rval.IsFailure(), uuidBuffer);
            }

            // Submit asynchronous request to NIFM
            ::g_pNifmConnection->SubmitRequest();

            // Wait while NIFM brings up the interface
            while(!::g_pNifmConnection->IsAvailable())
            {
                NN_LOG("Waiting for network interface...\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            ::g_TestTeardownOptions |= TestSetupOptions_Nifm;
        }

        // Setup socket
        if( 0 != (testSetupOptionsBitfield & TestSetupOptions_Socket) )
        {
            PRINT_AND_CALL(rval = nn::socket::Initialize(g_SocketConfigWithMemory));
            ERROR_IF(rval.IsFailure(), "nn::socket::Initialize failed");

            ::g_TestTeardownOptions |= TestSetupOptions_Socket;
        }

out:
        return isSuccess;
    }

    bool TestTeardown() NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::Result rval;

        ERROR_IF(-1 == ::g_TestTeardownOptions, "TestTeardown: precondition failed (bitfield -1)");

        // Teardown socket
        if( 0 != (::g_TestTeardownOptions & TestSetupOptions_Socket) )
        {
            PRINT_AND_CALL(rval = nn::socket::Finalize());
            ERROR_IF(rval.IsFailure(), "nn::socket::Finalize failed");
        }

        if( nullptr != ::g_pNifmConnection )
        {
            g_pNifmConnection->CancelRequest();
            delete ::g_pNifmConnection;
            ::g_pNifmConnection = nullptr;
        }

        ::g_TestTeardownOptions = -1;

out:
        return isSuccess;
    }


    void FillBuffer(char* pBuffer, uint32_t dataPosition, uint32_t fillLen) NN_NOEXCEPT
    {
        char val = static_cast<char>(dataPosition);
        for(uint32_t i = 0; i < fillLen; ++i)
        {
            pBuffer[i] = val;
            ++val;
        }
    }

    bool WaitOnReadable(int socket, uint32_t timeoutMs) NN_NOEXCEPT
    {
        nn::socket::PollFd fdPoll;
        fdPoll.fd = socket;
        fdPoll.events = nn::socket::PollEvent::PollRdNorm;
        fdPoll.revents = nn::socket::PollEvent::PollNone;

        int rval = nn::socket::Poll(&fdPoll, 1, timeoutMs);
        if( rval < 0 )
        {
            NN_LOG("Error: Poll: rval: %d errno: %d\n", rval, nn::socket::GetLastError());
            return false;
        }
        else if( rval == 0 )
        {
            NN_LOG("Poll: Timed out!\n");
            return false;
        }
        else if( !(fdPoll.revents & nn::socket::PollEvent::PollRdNorm) )
        {
            NN_LOG("Error: Poll: Socket not readable after poll! rval: %d errno: %d\n", rval, nn::socket::GetLastError());
            return false;
        }

        return true;
    }

    MD5Hash::Result CalcMd5(uint32_t dataLen) NN_NOEXCEPT
    {
        MD5Hash md5;
        MD5Hash::Result hashResult;
        char pBuffer[BufferLen];
        uint32_t bytesWritten = 0;

        do
        {
            uint32_t maxWrite = (dataLen - bytesWritten < sizeof(pBuffer)) ? dataLen - bytesWritten : sizeof(pBuffer);
            FillBuffer(pBuffer, bytesWritten, maxWrite);
            bytesWritten += maxWrite;
            md5.Update((uint8_t*)pBuffer, maxWrite);
        } while( bytesWritten < dataLen );

        md5.Final(hashResult);
        return hashResult;
    }

    bool RecvData(int socket) NN_NOEXCEPT
    {
        bool isSuccess = true;
        ssize_t recvRet = 0;
        ssize_t sendRet = 0;
        uint8_t response = 1;
        uint32_t expectedRecvLen = 0;
        uint32_t totalRecvd = 0;
        MD5Hash::Result expectedHash;
        MD5Hash::Result recvHash;
        MD5Hash recvMd5;

        NN_LOG("Receiving...\n");

        // Wait for data to read from socket
        isSuccess = WaitOnReadable(socket, PollTimeoutMs);
        ERROR_IF(!isSuccess, "Socket not readable!\n\n");

        // Receive the amount of data about to be sent by client
        recvRet = nn::socket::Recv(socket, &expectedRecvLen, sizeof(expectedRecvLen), nn::socket::MsgFlag::Msg_None);
        ERROR_IF(recvRet != sizeof(expectedRecvLen), "Failed to recv first message. rval: %d, errno: %d\n\n", (int)recvRet, nn::socket::GetLastError());
        expectedRecvLen = nn::socket::InetNtohl(expectedRecvLen);
        NN_LOG("Received first Message! Expected data size: %d bytes\n", (int)expectedRecvLen);

        do
        {
            char pBuffer[BufferLen];
            uint32_t recvChunk = ((expectedRecvLen - totalRecvd) < sizeof(pBuffer)) ? (expectedRecvLen - totalRecvd) : sizeof(pBuffer);

            // Wait for data to read from socket
            isSuccess = WaitOnReadable(socket, PollTimeoutMs);
            ERROR_IF(!isSuccess, "Socket not readable!\n\n");

            // Read data from socket
            recvRet = nn::socket::Recv(socket, pBuffer, recvChunk, nn::socket::MsgFlag::Msg_None);
            ERROR_IF(recvRet <= 0, "Failed to recv on socket. rval: %d, errno: %d\n\n", (int)recvRet, nn::socket::GetLastError());

            totalRecvd += (uint32_t)recvRet;
            recvMd5.Update((uint8_t*)pBuffer, (uint32_t)recvRet);
        } while(totalRecvd < expectedRecvLen);

        NN_LOG("Total Received: %d bytes\n", (int)totalRecvd);

        recvMd5.Final(recvHash);
        expectedHash = CalcMd5(expectedRecvLen);
        if( !(expectedHash == recvHash) )
        {
            NN_LOG("Error: Hash does not match expected!\n\n");
            isSuccess = false;
            response = 0;
        }

        sendRet = nn::socket::Send(socket, &response, sizeof(response), nn::socket::MsgFlag::Msg_None);
        ERROR_IF(sendRet != sizeof(response), "Failed to send response. rval: %d, errno: %d\n\n", (int)sendRet, nn::socket::GetLastError());

out:
        return isSuccess;
    }

    bool SendData(int socket, uint32_t dataLen) NN_NOEXCEPT
    {
        bool isSuccess = true;
        uint8_t response = 0;
        ssize_t recvRet = 0;
        ssize_t sendRet = 0;
        ssize_t totalSent = 0;
        uint32_t sendSizeBytes = 0;

        NN_LOG("Sending...\n");

        // Tell peer the size of data about to send.
        sendSizeBytes = nn::socket::InetHtonl(dataLen);
        sendRet = nn::socket::Send(socket, &sendSizeBytes, sizeof(sendSizeBytes), nn::socket::MsgFlag::Msg_None);
        ERROR_IF(sendRet != sizeof(sendSizeBytes), "Failed to send! errno: %d\n\n", nn::socket::GetLastError());

        do
        {
            char pBuffer[BufferLen];
            ssize_t sentBytes = 0;
            uint32_t sendChunk = (dataLen - (uint32_t)totalSent < sizeof(pBuffer)) ? dataLen - (uint32_t)totalSent : sizeof(pBuffer);
            FillBuffer(pBuffer, (uint32_t)totalSent, sendChunk);
            do
            {
                sendRet = nn::socket::Send(socket, &pBuffer[sentBytes], sendChunk - sentBytes, nn::socket::MsgFlag::Msg_None);
                ERROR_IF(sendRet < 0, "Failed to send on socket. errno: %d\n\n", nn::socket::GetLastError());
                sentBytes += sendRet;
            } while( sentBytes < (ssize_t)sendChunk );

            totalSent += sendChunk;
        } while(totalSent < (ssize_t)dataLen);

        NN_LOG("Total Sent: %d bytes\n", (int)totalSent);

        // Wait for data to read from socket
        isSuccess = WaitOnReadable(socket, PollTimeoutMs);
        ERROR_IF(!isSuccess, "Socket not readable!\n\n");

        // Read response from socket
        recvRet = nn::socket::Recv(socket, &response, sizeof(response), nn::socket::MsgFlag::Msg_None);
        ERROR_IF(recvRet != sizeof(response), "Failed to recv response. rval: %d, errno: %d\n\n", (int)recvRet, nn::socket::GetLastError());
        ERROR_IF(response != 1, "Peer responded with failure!\n\n");

        NN_LOG("Peer responded with success!\n");

out:
        return isSuccess;
    }
}} // namespace NATF::API
