﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;
static bool                 g_startThread = false;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );
    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait for Echo Server thread to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy Echo Server Thread
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,RecvFrom_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,RecvFrom_Various_Tests)
{
    nn::socket::SockAddrIn  outAddr = { 0 }, recvAddr = { 0 };
    int                     tcpSock = -1, udpSock = -1, udpServSock = -1;
    int                     serverSock = -1, acceptedSock = -1, cliSock = -1;
    int                     rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx;
    bool                    isSuccess = true;
    char                    buf1[1024];
    nn::socket::SockLenT    outAddrLen = sizeof( nn::socket::SockAddr );
    nn::socket::SockLenT    recvAddrLen = sizeof( recvAddr );
    uint64_t                unused1, unused2;

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////
    //  Establish UDP Target
    ///////////

    // Client: Target Address
    memset(&outAddr, 0, sizeof(outAddr));
    outAddr.sin_port        = nn::socket::InetHtons( 8053 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_echoServer->serverTCP )  ||
              ( idx == g_echoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        recvAddrLen = sizeof( recvAddr );
        rc = nn::socket::RecvFrom( idx, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr *>(&recvAddr), &recvAddrLen );
        ERROR_IF_AND_COUNT( rc > 0, "RecvFrom() succeeded but should have failed processing descriptor: %d\n", idx );

        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nRecvFrom() bad fds: %d should have failed with nn::socket::Errno::EBadf, but actually failed with: %d\n", idx, myError );
                goto out;
            }

            continue;  // Jump to top
        }

        if ( rc == 0 )
        {
            NN_LOG( "\nRecvFrom() fds: %d:  Succeeded but sent (no bytes)!\n", idx );
            goto out;
        }
    }

    NN_LOG( "\n" );


    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test Simple Network Echo Recieve
    ///////////

    NN_LOG( "Testing: [TCP] - Normal Send/Recieve\n" );

    rc = NATF::API::COMMON::WriteData( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ), unused1, unused2 );
    ERROR_IF( rc < 1, "[TCP]: Failed to write data to the Socket!\n" );

    // Wait for Echo back
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Got right response?
    rc = nn::socket::RecvFrom( tcpSock, buf1, sizeof( buf1), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ),
                        "Failed to read data from the TCP Socket!  Bytes read: %d, expected: %d\n", rc, strlen(  msgs[0] ) );

    ERROR_IF_AND_COUNT( strncmp( buf1, msgs[0], strlen( msgs[0] ) ) != 0,
                        "Failed to read data, Expected: [%.*s], Got [%.*s]\n", strlen( msgs[0] ), msgs[0], rc, buf1 );


    ////////////
    // [UDP] - Test Simple Network Echo Recieve
    ///////////

    NN_LOG( "Testing: [UDP] - Normal Send/Recieve\n" );

    rc = NATF::API::COMMON::WriteData( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ), unused1, unused2 );
    ERROR_IF( rc < 1, "[UDP] Failed to write data to the Socket!\n" );

    // Wait for Echo back
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Got right response?
    rc = nn::socket::RecvFrom( udpSock, buf1, sizeof( buf1), nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr *>(&recvAddr), &recvAddrLen );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ),
                        "Failed to read data from the TCP Socket!  Bytes read: %d, expected: %d\n", rc, strlen(  msgs[0] ) );

    ERROR_IF_AND_COUNT( strncmp( buf1, msgs[0], strlen( msgs[0] ) ) != 0 ,
                        "Failed to read data, Expected: [%.*s], Got [%.*s]\n", strlen( msgs[0] ), msgs[0], rc, buf1 );

    NN_LOG( "RecvFrom Response from Host: [%s:%d]\n", nn::socket::InetNtoa( recvAddr.sin_addr ),
                                                      nn::socket::InetNtohs( recvAddr.sin_port ) );

    ////////////
    // [UDP] - Test Simple Network Echo Recieve - No recieve SockAddr
    ///////////

    NN_LOG( "Testing: [UDP] - Normal Send/Recieve - No Recieve SockAddr\n" );

    rc = NATF::API::COMMON::WriteData( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ), unused1, unused2 );
    ERROR_IF( rc < 1, "[UDP] Failed to write data to the Socket!\n" );

    // Wait for Echo back
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Got right response?
    recvAddrLen = 0;
    rc = nn::socket::RecvFrom( udpSock, buf1, sizeof( buf1), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), &recvAddrLen );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ),
                        "Failed to read data from the TCP Socket!  Bytes read: %d, expected: %d\n", rc, strlen(  msgs[0] ) );

    ERROR_IF_AND_COUNT( strncmp( buf1, msgs[0], strlen( msgs[0] ) ) != 0 ,
                        "Failed to read data, Expected: [%.*s], Got [%.*s]\n", strlen( msgs[0] ), msgs[0], rc, buf1 );

    ////////////
    // [TCP] - nn::socket::MsgFlag::Msg_Peek
    ///////////

    NN_LOG( "Testing: [TCP] - nn::socket::MsgFlag::Msg_Peek\n" );

    rc = NATF::API::COMMON::WriteData( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ), unused1, unused2 );
    ERROR_IF( rc < 1, "[TCP]: Failed to write data to the Socket!\n" );

    // Wait for Echo back
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    for( idx = 0; idx < 5; idx++ )
    {
        NN_LOG( "TCP: nn::socket::MsgFlag::Msg_Peek [%d]\n", idx );
        // Got right response?
        memset( buf1, 0, sizeof( buf1 ) );
        rc = nn::socket::RecvFrom( tcpSock, buf1, sizeof( buf1), nn::socket::MsgFlag::Msg_Peek, nullptr, 0 );
        ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ),
                            "Failed to read data from the TCP Socket!  Bytes read: %d, expected: %d\n", rc, strlen(  msgs[0] ) );

        ERROR_IF_AND_COUNT( strncmp( buf1, msgs[0], strlen( msgs[0] ) ) != 0,
                            "Failed to read data, Expected: [%.*s], Got [%.*s]\n", strlen( msgs[0] ), msgs[0], rc, buf1 );
    }

    // Flush Socket
    rc = nn::socket::RecvFrom( tcpSock, buf1, sizeof( buf1), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    NN_LOG( "PASSED!\n" );


    ////////////
    // [UDP] - nn::socket::MsgFlag::Msg_Peek
    ///////////

    NN_LOG( "Testing: [UDP] - nn::socket::MsgFlag::Msg_Peek\n" );

    rc = NATF::API::COMMON::WriteData( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ), unused1, unused2 );
    ERROR_IF( rc < 1, "[UDP]: Failed to write data to the Socket!\n" );

    // Wait for Echo back
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    for( idx = 0; idx < 5; idx++ )
    {
        NN_LOG( "UDP: nn::socket::MsgFlag::Msg_Peek [%d]\n", idx );
        // Got right response?
        memset( buf1, 0, sizeof( buf1 ) );
        rc = nn::socket::RecvFrom( udpSock, buf1, sizeof( buf1), nn::socket::MsgFlag::Msg_Peek, nullptr, 0 );
        ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ),
                            "Failed to read data from the UDP Socket!  Bytes read: %d, expected: %d\n", rc, strlen(  msgs[0] ) );

        ERROR_IF_AND_COUNT( strncmp( buf1, msgs[0], strlen( msgs[0] ) ) != 0,
                            "Failed to read data, Expected: [%.*s], Got [%.*s]\n", strlen( msgs[0] ), msgs[0], rc, buf1 );
    }

    // Flush Socket
    rc = nn::socket::RecvFrom( udpSock, buf1, sizeof( buf1), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    NN_LOG( "PASSED!\n" );

    ////////////
    // [TCP] - nn::socket::MsgFlag::Msg_DontWait + nn::socket::MsgFlag::Msg_WaitAll
    ///////////

    NN_LOG( "Testing: [TCP] - nn::socket::MsgFlag::Msg_DontWait + nn::socket::MsgFlag::Msg_WaitAll\n" );

    rc = nn::socket::RecvFrom( tcpSock, buf1, sizeof( buf1),  nn::socket::MsgFlag::Msg_DontWait | nn::socket::MsgFlag::Msg_WaitAll, nullptr, 0 );
    ERROR_IF_AND_COUNT( rc > -1, "RecvFrom succeeded but should have failed!\n" );
    ERROR_IF_AND_COUNT( nn::socket::GetLastError() != nn::socket::Errno::EAgain,
                        "Expected errno nn::socket::Errno::EAgain but actually got errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [TCP] - Test NULL Recv Buffer
    ///////////


    NN_LOG( "Testing: [TCP] - NULL Receive Buffer - nn::socket::Errno::EInval\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::RecvFrom( tcpSock, nullptr, 1, nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc > -1, "RecvFrom() succeeded but should have failed! [TCP]\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "RecvFrom() expected Errno: nn::socket::Errno::EInval. actually got: %d\n", myError );


    ////////////
    // [UDP] - Test NULL Send Buffer
    ///////////

    NN_LOG( "Testing: [UDP] - NULL Recieve Buffer - nn::socket::Errno::EInval\n" );

    outAddrLen = sizeof( outAddr );
    rc = nn::socket::RecvFrom( udpSock, nullptr, 1, nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc > -1, "RecvFrom() succeeded but should have failed! [UDP]\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "RecvFrom() expected Errno: nn::socket::Errno::EInval. actually got: %d\n", myError );


    ////////////
    // [TCP] - Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [TCP] - (No Address Len)\n" );

    outAddrLen = 0;
    rc = nn::socket::RecvFrom( tcpSock, buf1, 0, nn::socket::MsgFlag::Msg_DontWait, nullptr, 0 );
    ERROR_IF_AND_COUNT( rc != 0, "[TCP] RecvFrom() failed but should have succeeded! Errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [UDP] - Test NULL No Address Length
    ///////////

    NN_LOG( "Testing: [UDP] - (No Address Len)\n" );

    outAddrLen = 0;
    rc = nn::socket::RecvFrom( udpSock, buf1, 0, nn::socket::MsgFlag::Msg_DontWait, reinterpret_cast<nn::socket::SockAddr *>(&outAddr), &outAddrLen );
    ERROR_IF_AND_COUNT( rc != 0, "[UDP] RecvFrom() failed but should have succeeded! Errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // [UDP] - Connect established destination - Same Address
    ///////////

    NN_LOG( "Testing: [UDP]- Connect() establishes destination - Same Address\n" );

    outAddrLen = 0;
    rc = nn::socket::RecvFrom( udpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_DontWait, nullptr, 0 );
    ERROR_IF_AND_COUNT( rc > -1, "[UDP] RecvFrom() succeeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock , "[UDP] RecvFrom() got errno: %d\n", myError );


    /////////////////
    // [TCP] - nn::socket::Errno::EConnReset - The remote socket is forcibly closed
    /////////////////

    NN_LOG( "Testing: [TCP]- (OTHERSIDE connection RESET\n" );

    {
        // Sevrver: Make a listen socket
        rc = NATF::API::COMMON::MakeListenSocket( serverSock, 9000, "127.0.0.1" );
        ERROR_IF( rc < 0, "Failed to create server listen socket\n" );

        // Server is a BLOCKING socket
        rc = NATF::API::COMMON::SetBlocking( serverSock );

        // Connect a Client Soscket
        cliSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9000 );
        ERROR_IF( cliSock < 0, "Failed to Connect to server on port 9000!\n" );

        // Client is a BLOCKING socket
        rc = NATF::API::COMMON::SetBlocking( cliSock );

        // Server: Accept the Client socket
        acceptedSock = nn::socket::Accept( serverSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );

        // Accepted Socket is a BLOCKING socket
        rc = NATF::API::COMMON::SetBlocking( acceptedSock );

        ////////////////
        // Close remote end of socket -- suddenly...
        ////////////////

        nn::socket::Close( acceptedSock );
        acceptedSock = -1;

        nn::socket::Close( serverSock );
        serverSock = -1;


        /////////////
        // RecvFrom() should fail - nn::socket::Errno::EConnReset
        /////////////

        rc = nn::socket::RecvFrom( cliSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_WaitAll, nullptr, 0 );
        ERROR_IF_AND_COUNT( rc != 0, "RecvFrom() should have failed with rc (0 == ZERO).  But I got rc: %d\n", rc );

        NN_LOG( "*****************************\n" );
        NN_LOG( "PASSED!  (OTHERSIDE Connection RESET)\n" );
        NN_LOG( "*****************************\n" );


        ////////////////
        // Close client
        ////////////////

        nn::socket::Close( cliSock );
        cliSock = -1;

    }


out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( serverSock > -1 )
    {
        nn::socket::Close( serverSock );
        serverSock = -1;
    }

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

}   // NOLINT(impl/function_size)


namespace {

typedef struct
{
    int   sockfds;
    int   count;

} RECVFROM_PAYLOAD_TYPE;

} // Anonymous Namespace


static void
RecvFrom_Threaded_Counter( void  * inArg )
{
    RECVFROM_PAYLOAD_TYPE *  payload;
    int                      idx, rc;
    uint64_t                 unused1, unused2;
    bool                     isSuccess = true;


    // Restore Payload
    payload = (RECVFROM_PAYLOAD_TYPE *) inArg;

    // Write 10 'A' chars slowly.,..
    NN_LOG( "Writing 'A' at position [" );
    for( idx = 0; idx < payload->count; idx++ )
    {
        NN_LOG( "%d, ", idx );

        rc = NATF::API::COMMON::WriteData( payload->sockfds, (unsigned char *) "A", 1, unused1, unused2 );
        ERROR_IF( rc != 1, "Failed to write 'A' char at position [%d]\n", idx );

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG( "]\n"  );

out:

    NN_LOG( "Thread Exiting!\n" );

}


TEST(ApiUnit,RecvFrom_MSG_WAITALL_DONTWAIT)
{
    RECVFROM_PAYLOAD_TYPE  payload;
    int                    threadId = -1;
    int                    tcpSock = -1;
    int                    rc = -1;
    bool                   isSuccess = true;
    char                   buf1[10];

    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    //////////////////////////////
    // B L O C K I N G    T E S T
    //////////////////////////////

    NN_LOG( "Testing: RecvFrom nn::socket::MsgFlag::Msg_WaitAll - RecvFrom should BLOCK\n" );

    rc = NATF::API::COMMON::SetBlocking( tcpSock );
    ERROR_IF_AND_COUNT( rc < 0, "Failed make TCP socket BLOCKING!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make Threaded Test
    ///////////

    payload.sockfds = tcpSock;
    payload.count   = 10;
    threadId = g_pThreadedTest->CreateThread( &RecvFrom_Threaded_Counter, (void *) &payload, kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    ////////////
    // Block RecvFrom() until all bytes receivec
    ///////////

    rc = nn::socket::RecvFrom( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_WaitAll, nullptr, 0 );
    ERROR_IF_AND_COUNT( rc != 10, "Expected to receive 10 bytes, actually received [%d] bytes\n", rc );

    NN_LOG( "=============\n" );
    NN_LOG( "PASSED!\n" );
    NN_LOG( "=============\n" );

    // Wait for test thread to die
    g_pThreadedTest->WaitThread( threadId );

    // Destroy test Thread
    g_pThreadedTest->DestroyThread( threadId );


    /////////////////////////////////
    // N O N - B L O C K I N G    T E S T
    ////////////////////////////////

    NN_LOG( "Testing: RecvFrom nn::socket::MsgFlag::Msg_DontWait - RecvFrom should NOT BLOCK\n" );


    ////////////
    // Make Threaded Test
    ///////////

    payload.sockfds = tcpSock;
    payload.count  = 10;

    threadId = g_pThreadedTest->CreateThread( &RecvFrom_Threaded_Counter, (void *) &payload, kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    ////////////
    // Perform Non-Blocking Receive from..
    ///////////

    rc = nn::socket::RecvFrom( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_DontWait, nullptr, 0 );
    ERROR_IF (rc == 10, "RecvFrom() blocked until 10 bytes were received, but should have unblocked before that!\n" );
    NN_LOG( "RecvFrom() Thread - UNBLOCKS (now)\n" );

    // Wait for test thread to die
    g_pThreadedTest->WaitThread( threadId );

    // Destroy test Thread
    g_pThreadedTest->DestroyThread( threadId );

    NN_LOG( "=============\n" );
    NN_LOG( "PASSED!\n" );
    NN_LOG( "=============\n" );

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

}




TEST(ApiUnit,RecvFrom_SO_OOB)
{

    int                     cliSock = -1, listenSock = -1, acceptedSock = -1;
    int                     rc = -1;
    bool                    isSuccess = true, isOOB = false;
    int                     disableFlag = 0, count = 0, atMark = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    nn::socket::SockLenT    optlen = sizeof( disableFlag );
    char                    buf1[1024] = {};

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////////
    // Create Server
    ////////////////

    rc = NATF::API::COMMON::MakeListenSocket( listenSock, 9001, g_echoServer->kInterfaceIPv4Addr );
    ERROR_IF_AND_COUNT( listenSock < 0, "Failed to creat Server socket\n" );
    NN_LOG( "Server established at: loopback:9001\n" );


    ////////////////
    // Create Client
    ////////////////

    cliSock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 9001 );
    ERROR_IF_AND_COUNT( cliSock < 0, "Failed to create Server socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );


    ////////////////
    // Server Accepts New Client
    ////////////////

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server Turns off OOBINLINE
    ////////////////

    rc = nn::socket::SetSockOpt( acceptedSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline,(void *) &disableFlag, optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_OobInline - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server turns on NON-BLOCKING
    ////////////////

    rc = nn::socket::Fcntl( acceptedSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed for nn::socket::FcntlFlag::O_NonBlock - Errno: %d\n", nn::socket::GetLastError() );


    //////////////
    // Client Writes (regular) and (OOB) data to server..
    //////////////

    NN_LOG( "SendTo(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc = nn::socket::SendTo( cliSock, (unsigned char *) msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ), "Client failed to write; %s\n", msgs[0] );

    NN_LOG( "SendTo nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc = nn::socket::SendTo( cliSock, (unsigned char *) msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[1] ), "Client failed to Send (OOB); %s\n", msgs[1], nullptr, 0 );

    NN_LOG( "SendTo(): %.*s\n", strlen( msgs[2] ), msgs[2] );
    rc = nn::socket::SendTo( cliSock, (unsigned char *) msgs[2], strlen( msgs[2] ), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[2] ), "Client failed to write; %s\n", msgs[2] );


    ///////////////
    // Server Reads (Regular) data BEFORE the OOB data (aka mark)
    ///////////////

    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ; ; )
    {
        atMark = nn::socket::SockAtMark( acceptedSock );
        if (atMark < 0 )
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError() );
        }
        if ( atMark == 1 )
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc = nn::socket::RecvFrom( acceptedSock, (unsigned char *) &buf1[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : nn::socket::MsgFlag::Msg_None ), static_cast<nn::socket::SockAddr *>(nullptr), 0 );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                rc = nn::socket::RecvFrom( acceptedSock, (unsigned char *) &buf1[count], 1, nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
                ERROR_IF( rc < 0, "RecvFrom() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                NN_LOG( "%c", (char) buf1[count] );
                count++;
                continue;
            }
            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "RecvFrom() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", (char) buf1[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, buf1 );

    // Validate returned data:
    if ( count == strlen( msgs[3] )  &&
         memcmp( msgs[3], buf1, count ) == 0 )
    {
        NN_LOG( "====================\n" );
        NN_LOG( "PASSED!  OOB and Normal Data returned correctly!\n" );
        NN_LOG( "====================\n" );
    }
    else
    {
        NN_LOG( "====================\n" );
        NN_LOG( "FAILED!  OOB Data [%.*s] and Actually Recieved Data [%.*s] do not match!\n",
                               count, buf1, strlen( msgs[3] ), msgs[3] );
        NN_LOG( "====================\n" );
    }


out:

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

}   // NOLINT(impl/function_size)

TEST( ApiUnit,RecvFrom_SO_OOBINLINE)
{
    int                     cliSock = -1, listenSock = -1, acceptedSock = -1;
    int                     rc = -1;
    bool                    isSuccess = true, isOOB = false;
    int                     enableFlag = 1, count = 0, atMark = 0;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    nn::socket::SockLenT    optlen = sizeof( enableFlag );
    char                    buf1[1024] = {};

    const char * msgs[]
    {
         "The rain in Spain...",
         "Flows",
         " mainly on the plain.",
         "The rain in Spain...Flows mainly on the plain."
    };

    ////////////////
    // Create Server
    ////////////////

    rc = NATF::API::COMMON::MakeListenSocket( listenSock, 9001, g_echoServer->kInterfaceIPv4Addr );
    ERROR_IF_AND_COUNT( listenSock < 0, "Failed to creat Server socket\n" );
    NN_LOG( "Server established at: loopback:9001\n" );


    ////////////////
    // Create Client
    ////////////////

    cliSock = NATF::API::COMMON::MakeTCPConnection( g_echoServer->kInterfaceIPv4Addr, 9001 );
    ERROR_IF_AND_COUNT( cliSock < 0, "Failed to create Server socket\n" );
    NN_LOG( "Client established at: loopback:9001\n" );


    ////////////////
    // Server Accepts New Client
    ////////////////

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server Turns ON OOBINLINE
    ////////////////

    rc = nn::socket::SetSockOpt( acceptedSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline,(void *) &enableFlag, optlen );
    ERROR_IF( rc < 0, "SetSockOpt() failed for nn::socket::Option::So_OobInline - Errno: %d\n", nn::socket::GetLastError() );


    ////////////////
    // Server turns ON NON-BLOCKING
    ////////////////

    rc = nn::socket::Fcntl( acceptedSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed for nn::socket::FcntlFlag::O_NonBlock - Errno: %d\n", nn::socket::GetLastError() );


    //////////////
    // Client Writes (regular) and (OOB) data to server..
    //////////////

    NN_LOG( "SendTo(): %.*s\n", strlen( msgs[0] ), msgs[0] );
    rc = nn::socket::SendTo( cliSock, (unsigned char *) msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None , static_cast<nn::socket::SockAddr *>(nullptr), 0);
    ERROR_IF_AND_COUNT( rc != strlen( msgs[0] ), "Client failed to write; %s\n", msgs[0] );

    NN_LOG( "SendT0() nn::socket::MsgFlag::Msg_Oob: %.*s\n", strlen( msgs[1] ), msgs[1] );
    rc = nn::socket::SendTo( cliSock, (unsigned char *) msgs[1], strlen( msgs[1] ), nn::socket::MsgFlag::Msg_Oob, static_cast<nn::socket::SockAddr *>(nullptr), 0);
    ERROR_IF_AND_COUNT( rc != strlen( msgs[1] ), "Client failed to Send (OOB); %s\n", msgs[1] );

    NN_LOG( "SendTo(): %.*s\n", strlen( msgs[2] ), msgs[2] );
    rc = nn::socket::SendTo( cliSock, (unsigned char *) msgs[2], strlen( msgs[2] ), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc != strlen( msgs[2] ), "Client failed to write; %s\n", msgs[2] );


    ///////////////
    // Server Reads (Regular) data BEFORE the OOB data (aka mark)
    ///////////////

    NN_LOG( "Read before Mark: [" );
    isOOB = false;
    for( ; ; )
    {
        atMark = nn::socket::SockAtMark( acceptedSock );
        if (atMark < 0 )
        {
            NN_LOG( "SockAtMark() failed - errno: %d\n", nn::socket::GetLastError() );
        }
        if ( atMark == 1 )
        {
            NN_LOG("[At Mark!]\n" );
            isOOB = true;
        }
        else
        {
            isOOB = false;
        }

        rc = nn::socket::RecvFrom( acceptedSock, (unsigned char *) &buf1[count], 1, (isOOB == true ? nn::socket::MsgFlag::Msg_Oob : nn::socket::MsgFlag::Msg_None ), static_cast<nn::socket::SockAddr *>(nullptr), 0  );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();

            // TRICKY: If nn::socket::Errno::EInval is returned it means "read again" without the OOB flag
            if ( myError == nn::socket::Errno::EInval )
            {
                rc = nn::socket::RecvFrom( acceptedSock, (unsigned char *) &buf1[count], 1, nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
                ERROR_IF( rc < 0, "RecvFrom() failed without nn::socket::MsgFlag::Msg_Oob - Errno: %d\n", myError );
                NN_LOG( "%c", (char) buf1[count] );
                count++;
                continue;
            }
            // If it's not nn::socket::Errno::EInval then see if we are done reading
            if ( myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock )
            {
                break;
            }

            NN_LOG( "RecvFrom() failed with errno: %d\n", nn::socket::GetLastError() );
            goto out;
        }
        NN_LOG( "%c", (char) buf1[count] );
        count++;
        continue;
    }
    NN_LOG( "]\n" );

    printf( "Actual Received Data: [%.*s]\n", count, buf1 );

    // Validate returned data:
    if ( count == strlen( msgs[3] )  &&
         memcmp( msgs[3], buf1, count ) == 0 )
    {
        NN_LOG( "====================\n" );
        NN_LOG( "PASSED!  OOB and Normal Data returned correctly!\n" );
        NN_LOG( "====================\n" );
    }
    else
    {
        NN_LOG( "====================\n" );
        NN_LOG( "FAILED!  OOB Data [%.*s] and Actually Recieved Data [%.*s] do not match!\n",
                               count, buf1, strlen( msgs[3] ), msgs[3] );
        NN_LOG( "====================\n" );
    }


out:

    if ( cliSock > -1 )
    {
        nn::socket::Close( cliSock );
        cliSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

}   // NOLINT(impl/function_size)


TEST(ApiUnit,RecvFrom_Test_RCVLOWAT)
{
    RECVFROM_PAYLOAD_TYPE  payload;
    int                    threadId = -1;
    int                    tcpSock = -1;
    int                    rc = -1, optval = 10;
    bool                   isSuccess = true;
    char                   buf1[20];

    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    //////////////////////////////
    // S E T   R C V L O W A T
    //////////////////////////////

    NN_LOG( "Testing: RecvFrom with nn::socket::Option::So_RcvLoWat - RecvFrom should UNBLOCK\n" );

    rc = NATF::API::COMMON::SetBlocking( tcpSock );
    ERROR_IF_AND_COUNT( rc < 0, "Failed make TCP socket BLOCKING!  Errno:%d\n", nn::socket::GetLastError() );

    optval = 10;
    rc = nn::socket::SetSockOpt( tcpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvLoWat, &optval, sizeof( optval ) );
    ERROR_IF( rc < 0, "SetSockOpt failed for nn::socket::Option::So_RcvLoWat - Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Make Threaded Test
    ///////////

    payload.sockfds = tcpSock;
    payload.count   = 20;

    threadId = g_pThreadedTest->CreateThread( &RecvFrom_Threaded_Counter, (void *) &payload, kThreadPriority );
    if ( threadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    ////////////
    // Block RecvFrom() until all bytes receive
    ///////////

    rc = nn::socket::RecvFrom( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc != 10, "Expected to receive 10 bytes, actually received [%d] bytes\n", rc );

    NN_LOG( "=============\n" );
    NN_LOG( "PASSED!  Unblocked at [10] same settings as RCVLOWAT\n" );
    NN_LOG( "=============\n" );

    // Wait for test thread to die
    g_pThreadedTest->WaitThread( threadId );

    // Destroy test Thread
    g_pThreadedTest->DestroyThread( threadId );
    threadId = -1;

    // Close Open Socket
    nn::socket::Close( tcpSock );
    tcpSock = -1;

out:

    if ( threadId > -1 )
    {
        // Wait for test thread to die
        g_pThreadedTest->WaitThread( threadId );

        // Destroy test Thread
        g_pThreadedTest->DestroyThread( threadId );

        threadId = -1;
    }

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

}


TEST(ApiUnit,RecvFrom_Test_RCVTIMEO)
{
    nn::socket::TimeVal myTime = {};
    int                 tcpSock = -1;
    int                 rc = -1;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;
    bool                isSuccess = true;
    char                buf1[20] = {};

    ////////////
    // Make TCP Connection
    ///////////

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    //////////////////////////////
    // S E T   R C V T I M E O
    //////////////////////////////

    NN_LOG( "Testing: RecvFrom with nn::socket::Option::So_RcvTimeo - RecvFrom should UNBLOCK\n" );

    rc = NATF::API::COMMON::SetBlocking( tcpSock );
    ERROR_IF_AND_COUNT( rc < 0, "Failed to make TCP socket BLOCKING!  Errno:%d\n", nn::socket::GetLastError() );

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;

    rc = nn::socket::SetSockOpt( tcpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, (void *) &myTime, sizeof( myTime ) );
    ERROR_IF( rc < 0, "SetSockOpt failed for nn::socket::Option::So_RcvTimeo - Errno:%d\n", nn::socket::GetLastError() );

    ////////////////////
    // RecvFrom() should unblock..
    ///////////////////

    rc = nn::socket::RecvFrom( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None, static_cast<nn::socket::SockAddr *>(nullptr), 0 );
    ERROR_IF( rc > 0, "RecvFrom succeeded for nn::socket::Option::So_RcvTimeo, but was supposed to FAIL!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF( myError != nn::socket::Errno::EAgain && myError != nn::socket::Errno::EWouldBlock,
             "RecvFrom() failed for nn::socket::Option::So_RcvTimeo - Errno:%d\n", myError );


    NN_LOG( "=============\n" );
    NN_LOG( "PASSED!  Unblocked after [5] second timeout time\n" );
    NN_LOG( "=============\n" );

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }
}

namespace {

typedef struct
{
    int   threadId;
    int   myIdx;
    int   sockfds;
    int   count;
    int   desiredCore;

} RECVFROM_PAYLOAD_TYPE2;

}



static void
RecvFrom_From_UDP_Clients( void  * inArg )
{
    RECVFROM_PAYLOAD_TYPE2 * payload;
    int                      idx, rc;
    uint64_t                 unused1, unused2;
    bool                     isSuccess = true;

    // Restore Payload
    payload = (RECVFROM_PAYLOAD_TYPE2 *) inArg;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, payload->desiredCore );

    while( g_startThread == false )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( 500 ) );
    }

    NN_LOG( "Thread %d Starting!\n", payload->myIdx );

    for( idx = 0; idx < payload->count; idx++ )
    {
        rc = NATF::API::COMMON::WriteData( payload->sockfds, (unsigned char *) "A", 1, unused1, unused2 );
        ERROR_IF( rc != 1, "Failed to write 'A' char at position [%d]\n", idx );

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

out:

    NN_LOG( "Thread %d Exiting!\n", payload->myIdx );
}


TEST(ApiUnit,RecvFrom_From_UDP_Clients)
{
    RECVFROM_PAYLOAD_TYPE2  payload[5];
    nn::socket::TimeVal          myTime;
    nn::socket::SockAddrIn      outAddr, recvAddr;
    int                     idx = 0, rc = 0;
    int                     udpSock;
    nn::socket::SockLenT               recvAddrLen;
    bool                    isSuccess = true;
    char                    buf1[100];
    int                     core = 1;
    int                     portCount = 1;

    // Initialize Payload
    for( idx = 0; idx < 5; idx++ )
    {
        payload[ idx ].threadId = -1;
        payload[ idx ].myIdx    = -1;
        payload[ idx ].sockfds  = -1;
        payload[ idx ].count    = -1;
    }

    ////////////
    // Make UDP Server
    ///////////

    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Socket failed with errno: %d\n", nn::socket::GetLastError() )

    // Client: Target Address
    memset( &outAddr, 0, sizeof(outAddr) );
    outAddr.sin_port        = nn::socket::InetHtons( 7000 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    myTime.tv_sec  = 5;
    myTime.tv_usec = 0;
    rc = nn::socket::SetSockOpt( udpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, (void *) &myTime, sizeof( myTime ) );
    ERROR_IF( rc < 0, "SetSockOpt failed for nn::socket::Option::So_RcvTimeo - Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Bind( udpSock, reinterpret_cast<nn::socket::SockAddr *>(&outAddr), sizeof( outAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP socket - Errno: %dn", nn::socket::GetLastError() );


    ////////////
    // Make Threaded Test
    ///////////

    // Move for loop up +1 to avoid add by +0
    g_startThread = false;
    portCount = 7001;
    core = 1;
    for( idx = 0; idx < 5; idx++ )
    {
        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        payload[idx].myIdx       = idx;
        payload[idx].count       = 10;
        payload[idx].desiredCore = core;

        payload[idx].sockfds = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", portCount, "127.0.0.1", 7000 );
        if ( payload[idx].sockfds < 0 )
        {
            NN_LOG( "Failed to create a new UDP Socket!\n" );
            goto out;
        }
        portCount++;

        payload[idx].threadId = g_pThreadedTest->CreateThread( &RecvFrom_From_UDP_Clients, (void *) &payload[idx], kThreadPriority );
        if ( payload[idx].threadId < 0 )
        {
            NN_LOG( "Failed to create Client (thread)!\n" );
            goto out;
        }
    }

    // Catch 50 messages
    g_startThread = true;
    for( idx = 0; idx < 50; idx++ )
    {
        recvAddrLen = sizeof( recvAddr );
        rc = nn::socket::RecvFrom( udpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr *>(&recvAddr), &recvAddrLen );
        ERROR_IF( rc < 0, "RecvFrom()  - Errno: %d\n", nn::socket::GetLastError() );

        NN_LOG( "Recieved UDP from [%s:%d]\n", nn::socket::InetNtoa(  recvAddr.sin_addr ),
                                               nn::socket::InetNtohs( recvAddr.sin_port ) );
    }


out:

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    // Free Payload
    for( idx = 0; idx < 5; idx++ )
    {
        if ( payload[idx].threadId > -1 )
        {
            g_pThreadedTest->WaitThread(    payload[idx].threadId );
            g_pThreadedTest->DestroyThread( payload[idx].threadId );
            payload[idx].threadId = -1;
        }

        if ( payload[ idx ].sockfds > -1 )
        {
            nn::socket::Close( payload[ idx ].sockfds );
            payload[ idx ].sockfds = -1;
        }
    }
}

TEST(ApiUnit,RecvFrom_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,RecvFrom_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
