﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_LoopbackDnsServer.h"     // Loopback DNS

#include <nn/socket/socket_ApiPrivate.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/socket/resolver/resolver_Client.h>
#endif

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static ThreadedController * g_pThreadedTest = NULL;
static LoopbackDnsServer *  g_loopbackDnsServer = NULL;

static int                  kMaxNumberOfThreads = 32;
static int                  kThreadPriority = 10;

static uint8_t              g_WorkVaruchar;

// Standard Google Lookup
//
// ; <<>> DiG 9.10.3-P4 <<>> +noedns=0 google.com
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 35193
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;google.com.                    IN      A
// ;; ANSWER SECTION:
// google.com.             150     IN      A       216.58.193.78
//
// ;; Query time: 31 msefalsec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 02 12:59:44 PDT 2016
// ;; MSG SIZE  rcvd: 44

static const unsigned char GetHErrno_Google_com_simple[] = {

0x96, 0xac, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x06, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x03, 0x63, 0x6f, 0x6d, 0x00,
0x00, 0x01, 0x00, 0x01, 0xc0, 0x0c, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00,
0x00, 0xcd, 0x00, 0x04, 0xd8, 0x3a, 0xc1, 0x4e

};


// $ dig -x 192.203.230.10 +noedns
//
// ; <<>> DiG 9.10.3-P4 <<>> -x 192.203.230.10 +noedns
// ;; global options: +cmd
// ;; Got answer:
// ;; ->>HEADER<<- opcode: QUERY, status: NOERROR, id: 30073
// ;; flags: qr rd ra; QUERY: 1, ANSWER: 1, AUTHORITY: 0, ADDITIONAL: 0
//
// ;; QUESTION SECTION:
// ;10.230.203.192.in-addr.arpa.   IN      PTR
//
// ;; ANSWER SECTION:
// 10.230.203.192.in-addr.arpa. 10066 IN   PTR     e.root-servers.net.
//
// ;; Query time: 2 msec
// ;; SERVER: 10.1.19.29#53(10.1.19.29)
// ;; WHEN: Tue Aug 23 12:27:05 PDT 2016
// ;; MSG SIZE  rcvd: 77

static const unsigned char GetHErrno_Amazon_com_simple[] = {

0xf3, 0x78, 0x81, 0x80, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
0x02, 0x31, 0x30, 0x03, 0x32, 0x33, 0x30, 0x03, 0x32, 0x30, 0x33, 0x03,
0x31, 0x39, 0x32, 0x07, 0x69, 0x6e, 0x2d, 0x61, 0x64, 0x64, 0x72, 0x04,
0x61, 0x72, 0x70, 0x61, 0x00, 0x00, 0x0c, 0x00, 0x01, 0xc0, 0x0c, 0x00,
0x0c, 0x00, 0x01, 0x00, 0x00, 0x26, 0xaa, 0x00, 0x14, 0x01, 0x65, 0x0c,
0x72, 0x6f, 0x6f, 0x74, 0x2d, 0x73, 0x65, 0x72, 0x76, 0x65, 0x72, 0x73,
0x03, 0x6e, 0x65, 0x74, 0x00

};

static nn::socket::ResolverOption
DisableDnsCache()
{
    nn::socket::ResolverOption   myOption;

    NN_LOG( "DisableDnsCache() - Begin\n" );
    myOption.key = static_cast<nn::socket::ResolverOptionKey>( nn::socket::ResolverOptionKey::RequestEnableDnsCacheBoolean );
    myOption.type = nn::socket::ResolverOptionType::Boolean;
    myOption.size = sizeof(bool);
    myOption.data.booleanValue = false;
    NN_LOG( "DisableDnsCache() - End\n" );

    return myOption;
}

static void
StartLookbackDnsServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Loopback DNS server]: Thread starting\n" );

    LoopbackDnsServer * loopbackDnsServer = (LoopbackDnsServer *) inArg;

    // Start the DNS Server Main Loop
    loopbackDnsServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Loopback DNS server]: Thread exiting\n" );
}



static bool
InitializeTesting()
{
    nn::Result                result;
    int                       rc;
    bool                      isSuccess = true, rval;

    NN_LOG( "In\n\n" );

#ifdef ENABLE_FAILING_TESTS
NN_LOG("** TestModeVerbose is ENABLED **\n");
#endif

    // Initialize work var
    g_WorkVaruchar = false;

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate DNS Server
    ///////////////////////////

    g_loopbackDnsServer = new LoopbackDnsServer();
    if ( g_loopbackDnsServer == NULL )
    {
        NN_LOG( "Failed to allocate a LoopbackDnsServer.  Can't start Loopback port DNS Server\n" );
        goto out;
    }

    ///////////////////////////
    ////  Change DNS Server Address - to (loopback)
    ///////////////////////////

    rval = g_loopbackDnsServer->SetDnsResolver( "127.0.0.1", 8053 );
    ERROR_IF( rval != true, "Failed calling LoopbackDnsServer::SetDnsResolver" );

    ///////////////////////////
    //// Threaded DNS Server
    ///////////////////////////

    NN_LOG( "===> Starting [loopback] DNS Server\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumberOfThreads );

    // Create a thread to run the Server in
    rc = g_pThreadedTest->CreateThread( &StartLookbackDnsServer, (void *) g_loopbackDnsServer, kThreadPriority );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to create and start DNS Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started DNS Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_loopbackDnsServer != NULL )
    {
        delete g_loopbackDnsServer;
        g_loopbackDnsServer = NULL;
    }

    return( false );
}



static bool
CallGetHostByName( const char * in_domainname )
{
    nn::socket::HostEnt * respHosts;
    nn::socket::InAddr ** addr_list;
    int                 idx;

    // Disable DNS Cache
    nn::socket::ResolverOption myOption = DisableDnsCache();

    PRINT_AND_CALL(respHosts = nn::socket::GetHostEntByName( in_domainname, &myOption, 1));
    if ( respHosts == NULL )
    {
        NN_LOG( "nn::socket::GetHostByName() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
                 nn::socket::GetLastError(), *(nn::socket::GetHError() ),
                 nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }
    else
    {
        NN_LOG( "Host name       : [%s]\n", respHosts->h_name );
        NN_LOG( "Address Type    : [%d]\n", respHosts->h_addrtype );
        NN_LOG( "Len of Addr     : [%d]\n", respHosts->h_length );

        for(idx = 0; respHosts->h_aliases[idx] != NULL; idx++)
        {
            NN_LOG( "Alias: [%s]\n", respHosts->h_aliases[idx] );
        }

        addr_list = (nn::socket::InAddr **) respHosts->h_addr_list;
        for(idx = 0; addr_list[idx] != NULL; idx++)
        {
            NN_LOG( "IP Address: [%s]\n", nn::socket::InetNtoa( (*addr_list[idx] ) ) );
        }

        NN_LOG( "There are a total of [%d] IPv4 Addresses\n", idx );
    }

    return( true );

out:

    return( false );
}


static bool
CallGetHostByAddr( const char *    in_ipAddr )
{
    nn::socket::InAddr      ipv4Addr;
    nn::socket::HostEnt *    respHosts = NULL;
    nn::socket::InAddr **   addr_list = NULL;
    int                 idx = 0, rc = 0;
    bool                isSuccess = true;

    // Disable DNS Cache
    nn::socket::ResolverOption myOption = DisableDnsCache();

    // Translate Target IP Address into Lookup Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, in_ipAddr, &ipv4Addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    PRINT_AND_CALL(respHosts = nn::socket::GetHostEntByAddr( &ipv4Addr, 4, nn::socket::Family::Af_Inet, &myOption, 1));
    if ( respHosts == NULL )
    {
        NN_LOG( "nn::socket::GetHostByAddr() Failed with: errno: %d, h_errno: %d, hstrerror: %s\n",
                 nn::socket::GetLastError(), *(nn::socket::GetHError() ),
                 nn::socket::HStrError(  *(nn::socket::GetHError() ) ) );
        goto out;
    }
    else
    {
        NN_LOG( "Host name       : [%s]\n", respHosts->h_name );
        NN_LOG( "Address Type    : [%d]\n", respHosts->h_addrtype );
        NN_LOG( "Len of Addr     : [%d]\n", respHosts->h_length );

        for(idx = 0; respHosts->h_aliases[idx] != NULL; idx++)
        {

           NN_LOG( "Alias: [%s]\n", respHosts->h_aliases[idx] );
        }

        addr_list = (nn::socket::InAddr **) respHosts->h_addr_list;
        for(idx = 0; addr_list[idx] != NULL; idx++)
        {

           NN_LOG( "IP Address: [%s]\n", nn::socket::InetNtoa( (*addr_list[idx] ) ) );
        }

        NN_LOG( "There are a total of [%d] IPv4 Addresses\n", idx );
    }

    return( true );

out:

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping DNS Server\n" );

    // If the loopback DNS server is allocated
    if ( g_loopbackDnsServer != NULL )
    {
        g_loopbackDnsServer->Stop();
        delete g_loopbackDnsServer;
        g_loopbackDnsServer = NULL;
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitAllThreads();

    // Destroy all Threads
    g_pThreadedTest->DestroyAllThreads();

    // Delete Object Pointer
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;

    NN_LOG( "DNS Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

out:

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );
}

TEST(ApiUnit,GetHErrno_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,GetHErrno_Various_Stuff)
{
    int               idx = 0;
    bool              rval, isSuccess = true;
    char *            errorMsg;


    ///////////////////
    // Test HStrError()
    ///////////////////

    for( idx = ( static_cast<int>(nn::socket::HErrno::Netdb_Internal) - 5) ; idx < (static_cast<int>(nn::socket::HErrno::No_Data) + 5); idx++ )
    {
        errorMsg = (char *) nn::socket::HStrError( static_cast<nn::socket::HErrno>(idx) );
        ERROR_IF_AND_COUNT( errorMsg == NULL, "Called HStrError() with index: %d, but got a NULL response!\n", idx );
        NN_LOG( "Index: %d, Error Message: %s\n", idx, errorMsg );
    }

    // Normal DNS Rqst/Resp Parsers
    g_loopbackDnsServer->SetDNSRequestCB( g_loopbackDnsServer->MakeDnsRequestCB );
    g_loopbackDnsServer->SetDNSResponseCB( g_loopbackDnsServer->MakeDnsResponseCB );


    ///////////////////
    // Simple Lookup
    ///////////////////

    // GetHostByName()
    g_loopbackDnsServer->UDPResponse = (unsigned char *) GetHErrno_Google_com_simple;
    g_loopbackDnsServer->UDPResponseLen = sizeof( GetHErrno_Google_com_simple );

    NN_LOG( "Query: [GetHostByName][Google] Simple\n" );
    rval = CallGetHostByName( "google.com" );
    ERROR_IF_AND_COUNT( rval == false, "GHBN - Valid query asked for over UDP, but GHBN did not process the data!" );

    // GetHostByAddr()
    g_loopbackDnsServer->UDPResponse = (unsigned char *) GetHErrno_Amazon_com_simple;
    g_loopbackDnsServer->UDPResponseLen = sizeof( GetHErrno_Amazon_com_simple );

    NN_LOG( "Query: [GetHostByAddr][Amazon] Simple\n" );
    rval = CallGetHostByAddr( "192.203.230.10" );
    ERROR_IF_AND_COUNT( rval == false, "GHBA - Valid query asked for over UDP, but GHBA did not process the data!" );


    ///////////////////
    //  D O N E
    ///////////////////

out:

    return;
}  // NOLINT(impl/function_size)



TEST(ApiUnit,GetHErrno_GHBN_Response_Codes)
{
    unsigned char googleReply[512];
    int           idx;
    unsigned char respCode;
    bool          rval = false, isSuccess = false;

    memset( googleReply, 0, sizeof( googleReply ) );
    memcpy( googleReply, GetHErrno_Google_com_simple, sizeof( GetHErrno_Google_com_simple ) );

    for( idx = 1; idx < 11; idx++ )
    {
        respCode = ( 8 << 4 ) | ( idx );
        NN_LOG( "DNS response Code: %d, Encoded in Header as: 0x%02x\n", idx, respCode );

        // Set DNS Response Code
        googleReply[3] = respCode;

        // GetHostByName()
        g_loopbackDnsServer->UDPResponse = (unsigned char *) googleReply;
        g_loopbackDnsServer->UDPResponseLen = sizeof( GetHErrno_Google_com_simple );

        NN_LOG( "Query: [GetHostByName][Google]\n" );
        rval = CallGetHostByName( "google.com" );
        ERROR_IF_AND_COUNT( rval == true, "GHBN - Received an Error DNS Response, but indicates response succeeded!" );
        NN_LOG( "Got GHBN h_errno: %d\n", *nn::socket::GetHError() );

        switch( idx )
        {
        case 1:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 2:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::Try_Again,   "Expected: Host name lookup failure!" );  break;
        case 3:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::Host_Not_Found, "Expected: Unknown host!" );  break;
        case 4:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::Try_Again,   "Expected: Host name lookup failure!" );  break;
        case 5:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::Try_Again,   "Expected: Host name lookup failure!" );  break;
        case 6:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 7:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 8:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 9:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 10: ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        default:   break;
        }
    }

out:

    return;
}

TEST(ApiUnit,GetHErrno_GHBA_Response_Codes)
{
    unsigned char amazonReply[512];
    int           idx;
    unsigned char respCode;
    bool          rval = false, isSuccess = false;

    memset( amazonReply, 0, sizeof( amazonReply ) );
    memcpy( amazonReply, GetHErrno_Amazon_com_simple, sizeof( GetHErrno_Amazon_com_simple ) );

    for( idx = 1; idx < 11; idx++ )
    {
        respCode = ( 8 << 4 ) | ( idx );
        NN_LOG( "DNS response Code: %d, Encoded in Header as: 0x%02x\n", idx, respCode );

        // Set DNS Response Code
        amazonReply[3] = respCode;

        // GetHostByName()
        g_loopbackDnsServer->UDPResponse = (unsigned char *) amazonReply;
        g_loopbackDnsServer->UDPResponseLen = sizeof( GetHErrno_Amazon_com_simple );

        NN_LOG( "Query: [GetHostByAddr][Amazon] Simple\n" );
        rval = CallGetHostByAddr( "192.203.230.10" );
        ERROR_IF_AND_COUNT( rval == true, "GHBA - Received an Error DNS Response, but indicates response succeeded!" );
        NN_LOG( "Got GHBA h_errno: %d\n", *nn::socket::GetHError() );

        switch( idx )
        {
        case 1:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 2:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::Try_Again,   "Expected: Host name lookup failure!" );  break;
        case 3:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::Host_Not_Found, "Expected: Unknown host!" );  break;
        case 4:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::Try_Again,   "Expected: Host name lookup failure!" );  break;
        case 5:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::Try_Again,   "Expected: Host name lookup failure!" );  break;
        case 6:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 7:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 8:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 9:  ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        case 10: ERROR_IF_AND_COUNT( *nn::socket::GetHError() != nn::socket::HErrno::No_Recovery, "Expected: Unknown server error!" );  break;
        default:   break;
        }
    }

out:

    return;
}


TEST(ApiUnit,GetHErrno_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,GetHErrno_Window)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
