﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/util/util_FormatString.h>
#include <nn/nn_SdkLog.h>
#include <nn/socket.h>

#include <cstring>
#include <algorithm> //std::max

#include "testNet_ApiCommon.h"
#include "Complex/testNet_UnitCommon.h"
#include "Complex/testNet_SelectUnitData.h"
#include "Complex/testNet_SelectUnitServer.h"

namespace NATF {
namespace API {

/**
 * @brief select unit client
 */
SelectUnitServer::SelectUnitServer(SelectUnitData* pSelectUnitData) :
    LockedReferenceCountObjectImpl(__FUNCTION__),
    UnitTestThreadBase(SocketContainerActorAsString(SOCKET_CONTAINER_ACTOR_SERVER), 500),
    SelectUnitNetworkCommon(pSelectUnitData, SOCKET_CONTAINER_ACTOR_SERVER, this)
{
    //UNIT_TEST_TRACE("");
    nn::util::SNPrintf(m_SelectUnitServerName, sizeof(m_SelectUnitServerName),  "s:%d", m_pSelectUnitData->GetPort());
};

SelectUnitServer::~SelectUnitServer()
{
    //UNIT_TEST_TRACE("");
};

void SelectUnitServer::Run()
{
    RunLoop();
}

void SelectUnitServer::InitializeSockets(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    int rc = -1;
    nn::socket::SockAddrIn saServer = { 0 };
    nn::socket::SockAddrIn saInfo = { 0 };
    nn::socket::SockLenT  saLen = sizeof(saInfo);

    saServer.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Loopback);
    saServer.sin_port        = nn::socket::InetHtons(0);
    saServer.sin_family      = nn::socket::Family::Af_Inet;

    if (SOCK_TYPE_FLAGS_SOCK_STREAM == container.m_SocketTypeFlags)
    {
        if (-1 == (rc = nn::socket::Bind(container.m_Socket,
                                         (nn::socket::SockAddr *)&saServer,
                                         sizeof(nn::socket::SockAddr))))
        {
            SVALIDATE_FAIL(m_pValidator,
                           true,
                           "Bind (listener) failed socket: %d, errno: %d",
                           container.m_Socket, nn::socket::GetLastError());
            goto bail;
        }
        else if(-1 == ( rc = nn::socket::GetSockName(container.m_Socket,
                                                     reinterpret_cast<nn::socket::SockAddr *>(&saInfo),
                                                     &saLen) ))
        {
            SVALIDATE_FAIL(m_pValidator,
                           true,
                           "GetSockName (listener) failed socket: %d, errno: %d",
                           container.m_Socket, nn::socket::GetLastError());
            goto bail;
        };

        m_pSelectUnitData->SetPort(static_cast<unsigned int>(nn::socket::InetNtohs(saInfo.sin_port)));

        if (-1 == (rc = nn::socket::Listen(container.m_Socket, container.m_Backlog)))
        {
            SVALIDATE_FAIL(m_pValidator,
                           true,
                           "Listen failed on socket: %d, errno: %d",
                           container.m_Socket, nn::socket::GetLastError());
            goto bail;
        };

        UNIT_TEST_TRACE("listener %d on port %d with backlog of %d",
                        container.m_Socket,
                        m_pSelectUnitData->GetPort(),
                        container.m_Backlog);
    }
    else if (SOCK_TYPE_FLAGS_SOCK_DGRAM == container.m_SocketTypeFlags)
    {
        // unimplemented
        NN_ASSERT(false);
    }
    else
    {
        NN_ASSERT(false);
    };

bail:
    return;
};

bool SelectUnitServer::EarlyBlockingInitialize(nn::os::SemaphoreType& blockingSemaphore)
{
    //UNIT_TEST_TRACE("");
    bool rc = false;
    GetSiftedList();
    for (auto iter = m_SocketContainerList.begin(); iter != m_SocketContainerList.end(); ++iter)
    {
        InitializeSockets(*iter);
    }
    return rc;
}

int SelectUnitServer::OnNewConnectionEvent(SocketContainer& listenerContainer)
{
    //UNIT_TEST_TRACE("");
    int eventsHandled = 0;
    int rc = -1;
    nn::socket::SockAddrIn socketOutAddress;
    nn::socket::SockLenT socketLengthOut = sizeof(nn::socket::SockAddrIn);
    SocketContainer acceptedSocket;

    if (-1 != (rc = nn::socket::Accept(listenerContainer.m_Socket, reinterpret_cast<nn::socket::SockAddr*>(&socketOutAddress), &socketLengthOut)))
    {
        // create a container that represents the embryonic socket; data comes later
        UNIT_TEST_TRACE("accepted socket %d from listener %d errno: %d", rc, listenerContainer.m_Socket, nn::socket::GetLastError());

        acceptedSocket.m_Socket = rc;
        acceptedSocket.m_ContainerTypeFlags = static_cast<SocketContainerTypeFlags>(SOCKET_CONTAINER_TYPE_EMBRYONIC | SOCKET_CONTAINER_TYPE_READ);
        acceptedSocket.m_Role = SOCKET_CONTAINER_ACTOR_EMBRYONIC;
        acceptedSocket.m_SocketControlFlags = listenerContainer.m_SocketControlFlags;

        switch (acceptedSocket.m_SocketControlFlags)
        {
            // TODO: make this more flags than value
        case SOCKET_CONTROL_FLAGS_DEFAULT:
            break;
        case SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING:
            nn::socket::Fcntl(acceptedSocket.m_Socket, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock);
            break;
        default:
            UNIT_TEST_TRACE("unhandled default case hit for accept socket flags", SocketContainerActorAsString(m_Role));
            NN_ASSERT(false);
            goto bail;
        };

        AddSocketContainerToWaitingQueue(acceptedSocket);
        ++eventsHandled;

        if (--listenerContainer.m_Backlog == 0)
        {
            // can't shut down an unconnected socket
            UNIT_TEST_TRACE("queuing remove of listener socket %d", listenerContainer.m_Socket);
            listenerContainer.m_CurrentState = static_cast<SocketContainerTypeFlags>(listenerContainer.m_CurrentState | SOCKET_CONTAINER_TYPE_SHUTDOWN);
        };
    }
    else
    {
        SVALIDATE_FAIL(m_pValidator, true, "unable to accept socket %d, errno: %d", listenerContainer.m_Socket, rc, nn::socket::GetLastError());
        goto bail;
    };

bail:
    return eventsHandled;
};

int SelectUnitServer::OnEmbryonicReadEvent(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    int eventsHandled = 0;
    ssize_t rc = -1;
    uint8_t buf[1024] = { '\0' };

    SocketContainer packet;

    UNIT_TEST_TRACE("embryonic read: event encountered on socket: %d", container.m_Socket);

    rc = ReceiveContainer(packet, container, "embryonic read", nn::socket::MsgFlag::Msg_Oob);
    ++eventsHandled;

    if (rc <= 0)
    {
        goto bail;
    };

    container.m_AddressFamilyFlags = packet.m_AddressFamilyFlags;
    container.m_SocketTypeFlags = packet.m_SocketTypeFlags;
    container.m_ProtocolFlags = packet.m_ProtocolFlags;
    container.m_Backlog = packet.m_Backlog;
    container.m_SocketControlFlags = packet.m_SocketControlFlags;
    container.m_ContainerTypeFlags = packet.m_ContainerTypeFlags;
    container.m_Role = m_Role;

    if (0 != (container.m_ContainerTypeFlags & SOCKET_CONTAINER_TYPE_READ))
    {
        SendContainer(container, "embryonic/read", nn::socket::MsgFlag::Msg_None);
        container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState | SOCKET_CONTAINER_TYPE_READ);
    }
    else if (0 != (container.m_ContainerTypeFlags & SOCKET_CONTAINER_TYPE_WRITE))
    {
        UNIT_TEST_TRACE("delaying container send until write event")
            container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState | SOCKET_CONTAINER_TYPE_READ);
    }
    else if (0 != (container.m_ContainerTypeFlags & SOCKET_CONTAINER_TYPE_EXCEPTION))
    {
        // todo: can't do two way exception data because winsock is broken (send 40 bytes, get 39)
        //rc = SendContainer(container, "embryonic/exception", nn::socket::MsgFlag::Msg_Oob );
        rc = SendBytes(container, buf, 1, nn::socket::MsgFlag::Msg_Oob);
        container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState | SOCKET_CONTAINER_TYPE_EXCEPTION);
        container.m_CurrentState = SOCKET_CONTAINER_TYPE_SHUTDOWN;
    }
    else
    {
        NN_ASSERT(false);
    }
    container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState & ~SOCKET_CONTAINER_TYPE_EMBRYONIC);

bail:
    return eventsHandled;
}

int SelectUnitServer::OnReadEvent(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    int eventsHandled = 0;
    ssize_t rc = -1;
    uint8_t buf[1024] = { '\0' };
    nn::socket::Errno errorNumber = nn::socket::Errno::ESuccess;

    memset(buf, '\0', sizeof(buf));

    UNIT_TEST_TRACE("read: attempting to recv %d bytes from socket %d", SocketContainer::SizeOf(), container.m_Socket);
    rc = ReceiveBytes(container, buf, SocketContainer::SizeOf(), nn::socket::MsgFlag::Msg_None);
    ++eventsHandled;
    errorNumber = nn::socket::GetLastError();
    if (0 == rc
        || (nn::socket::Errno::EPipe == errorNumber && -1 == rc)
        || (nn::socket::Errno::EConnReset == errorNumber && -1 == rc)) // siglo for windows
    {
        container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState | SOCKET_CONTAINER_TYPE_SHUTDOWN);
        goto bail;
    }
    else if (-1 == rc)
    {
        SVALIDATE_FAIL(m_pValidator, true, "read: ReceiveBytes failed, socket: %d, rc: %d, errno: %d", container.m_Socket, rc, errorNumber);
        container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState | SOCKET_CONTAINER_TYPE_ERROR);
        goto bail;
    }
    else
    {
        UNIT_TEST_TRACE("read: recvd %d bytes from socket %d", rc, container.m_Socket);
        container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState | SOCKET_CONTAINER_TYPE_SHUTDOWN);
    };

bail:
    return eventsHandled;
};

int SelectUnitServer::OnWriteEvent(SocketContainer& container)
{
    //UNIT_TEST_TRACE("");
    int eventsHandled = 0;

    SendContainer(container, "write", nn::socket::MsgFlag::Msg_None);
    ++eventsHandled;

    container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState | SOCKET_CONTAINER_TYPE_SHUTDOWN);
    return eventsHandled;
};

int SelectUnitServer::OnExceptionEvent(SocketContainer& container)
{
    int eventsHandled = 0;
    ssize_t rc = -1;
    SocketContainer junk;

    UNIT_TEST_TRACE("exception: attempting to recv %d bytes from socket %d", SocketContainer::SizeOf(), container.m_Socket);
    rc = ReceiveContainer(junk, container, "exception", nn::socket::MsgFlag::Msg_Oob);
    ++eventsHandled;
    if (0 > rc)
    {
        goto bail;
    };

    rc = SendBytes(container, reinterpret_cast<uint8_t*>(const_cast<char*>("a")), 1, nn::socket::MsgFlag::Msg_Oob);
    //rc = SendContainer(container, "exception/send", nn::socket::MsgFlag::Msg_Oob);
    container.m_CurrentState = static_cast<SocketContainerTypeFlags>(container.m_CurrentState | SOCKET_CONTAINER_TYPE_SHUTDOWN);

bail:
    return eventsHandled;
};


}}; // NATF::API
