﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Complex/testNet_PollBeforeConnect.h"

namespace NATF {
namespace API {

NN_ALIGNAS(4096) uint8_t g_PollBeforeConnectSocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

PollBeforeConnectTestThread::PollBeforeConnectTestThread(SimpleValidator* pSimpleValidator,
                                                         int sockfd,
                                                         nn::socket::PollEvent events,
                                                         int pollTimeoutInSeconds,
                                                         int waitForDoneTimeoutInSeconds,
                                                         int pollExpectedReturnValue) :
    LockedReferenceCountObjectImpl(__FUNCTION__),
    UnitTestThreadBase(__FUNCTION__, (waitForDoneTimeoutInSeconds * 1000) ),
    m_pValidator(pSimpleValidator),
    m_Sockfd(sockfd),
    m_PollEvents(events),
    m_PollTimeoutInSeconds(pollTimeoutInSeconds),
    m_waitForDoneTimeoutInSeconds(waitForDoneTimeoutInSeconds),
    m_PollReturnValue(-1),
    m_PollExpectedReturnValue(pollExpectedReturnValue)
{
    NN_ASSERT(NULL != m_pValidator);
    m_pValidator->addReference();
};

PollBeforeConnectTestThread::~PollBeforeConnectTestThread()
{
    m_pValidator->releaseReference();
    nn::socket::Close(m_Sockfd);
};

/** @brief run function, waits for testTimeout or signal */
void PollBeforeConnectTestThread::Run()
{
    nn::socket::PollFd fds;
    int pollTimeoutInMilliSeconds = 0;
    {
        std::lock_guard<std::mutex> lock(m_AccessLock);
        fds.fd = m_Sockfd;
        fds.events = m_PollEvents;
        pollTimeoutInMilliSeconds = m_PollTimeoutInSeconds * 1000;
    };

    int rc = -1;

    UNIT_TEST_TRACE("Waiting on poll");

    rc = nn::socket::Poll(&fds, 1, pollTimeoutInMilliSeconds);
    {
        std::lock_guard<std::mutex> lock(m_AccessLock);
        m_PollReturnValue = rc;

        // the return value might be 1 if we close the socket out from under poll,
        // so first thing we do is check whether or not m_Sockfd was set to -1
        // and if not then we proceed
        if (m_Sockfd == -1)
        {
            UNIT_TEST_TRACE("poll was interrupted (m_Sockfd == -1); skipping call to Close");
        }
        else
        {
            if (m_PollReturnValue != m_PollExpectedReturnValue)
            {
                SVALIDATE_FAIL(m_pValidator,
                               true,
                               "poll returned: %d, expected: %d, errno: %d, sockfd: %d",
                               m_PollReturnValue, m_PollExpectedReturnValue, nn::socket::GetLastError(), m_Sockfd);
            }
            else
            {
                UNIT_TEST_TRACE("poll returned %d, expected: %d", m_PollReturnValue, m_PollExpectedReturnValue);
            };

            UNIT_TEST_TRACE("Calling Close");

            rc = nn::socket::Close(m_Sockfd);
            if (rc < 0)
            {
                SVALIDATE_FAIL(m_pValidator, true, " close failed, rc: %d, errno: %d, sockfd: %d",
                               rc, nn::socket::GetLastError(), m_Sockfd);
            };
            m_Sockfd = -1;
        };
    };

    return;
};

int PollBeforeConnectTestThread::Interrupt()
{
    UNIT_TEST_TRACE("");
    int rc = -1;
    std::lock_guard<std::mutex> lock(m_AccessLock);

    if (-1 == m_Sockfd)
    {
        UNIT_TEST_TRACE("Test previously ran to completion, unable to interrupt");
    }
    else
    {
        rc = nn::socket::Close(m_Sockfd);
        if (rc < 0)
        {
            SVALIDATE_FAIL(m_pValidator, true, " Interrupt / close failed, rc: %d, errno: %d, sockfd: %d",
                           rc, nn::socket::GetLastError(), m_Sockfd);
        };
        m_Sockfd = -1;
        rc = 0;
    };

    return rc;
};

bool PollBeforeConnectTestThread::WaitOrTimeoutGetSocketClose(bool shouldComplete, bool & didSocketClose)
{
    bool rc = UnitTestThreadBase::WaitForDoneWithTimeout();

    std::lock_guard<std::mutex> lock(m_AccessLock);
    didSocketClose = m_Sockfd == -1;

    if (true == shouldComplete && false == rc)
    {
        SVALIDATE_FAIL(m_pValidator, true, "Thread blocked for longer than %d seconds.", m_waitForDoneTimeoutInSeconds);
    }
    else if (false == shouldComplete && true == rc)
    {
        SVALIDATE_FAIL(m_pValidator, true, "Thread should still be blocked after %d seconds but finished instead.", m_waitForDoneTimeoutInSeconds);
    };

    return rc;
};

void RunPollBeforeConnectUnitTest(nn::socket::PollEvent events, int pollTimeoutInMilliseconds, int waitForDoneTimeoutInSeconds, bool shouldCompleteBeforeTimeout, int pollExpectedReturnValue)
{
    bool nifmSetupSuccess = false;

    if (pollTimeoutInMilliseconds >= waitForDoneTimeoutInSeconds)
    {
        UNIT_TEST_TRACE("Test Data Error: waitForDoneTimeoutInSeconds (%d) needs to be greater than pollTimeoutInMilliseconds",
                        waitForDoneTimeoutInSeconds, pollTimeoutInMilliseconds);
        ADD_FAILURE();
        return;
    }
    else if (false == (nifmSetupSuccess = NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket)))
    {
        NN_LOG("\nError: NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket) failed.\n");
        goto bail;
    }
    else
    {
        int sockfd = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        UNIT_TEST_TRACE("socket fd is: %d", sockfd);
        bool didSocketClose = false;
        AutoReleaseObject<SimpleValidator> validator(new SimpleValidator(), false);
        AutoReleaseObject<PollBeforeConnectTestThread> testThread(new PollBeforeConnectTestThread(&(*validator),
                                                                                                  sockfd,
                                                                                                  events,
                                                                                                  pollTimeoutInMilliseconds,
                                                                                                  waitForDoneTimeoutInSeconds,
                                                                                                  pollExpectedReturnValue));

        (*testThread).Start();
        (*testThread).WaitOrTimeoutGetSocketClose(shouldCompleteBeforeTimeout, didSocketClose);
        if (false == didSocketClose)
        {
            (*testThread).Interrupt();
        };

        if ( true == (*validator).DidFail() )
        {
            // FAIL message will have been printed
            ADD_FAILURE();
        };
    };

bail:
    if ( true == nifmSetupSuccess && false == NATF::API::TestTeardown() )
    {
        NN_LOG("\nError: NATF::API::TestTeardown() failed. \n");;
    };

    return;
};

static const int g_WaitForDoneTimeoutInSeconds = 2;

// negative 1 (i.e. wait forever)
TEST(PollBeforeConnect, ZERO_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollNone;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        false;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        true;
#elif defined(__FreeBSD__)
        true;
#else
        false;
#error "This target os is untested"
#endif
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLIN_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollIn;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        false;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        true;
#elif defined(__FreeBSD__)
        true;
#else
        false;
#error "This target os is untested"
#endif
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLPRI_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollPri;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        false;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        true;
#elif defined(__FreeBSD__)
        true;
#else
        false;
#error "This target os is untested"
#endif
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLIN_POLLPRI_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollPri;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        false;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        true;
#elif defined(__FreeBSD__)
        true;
#else
        false;
#error "This target os is untested"
#endif
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        false;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        true;
#elif defined(__FreeBSD__)
        true;
#else
        false;
#error "This target os is untested"
#endif
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLWRNORM_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollWrNorm;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        false;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        true;
#elif defined(__FreeBSD__)
        true;
#else
        false;
#error "This target os is untested"
#endif
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLWRNORM_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollWrNorm;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        false;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        true;
#elif defined(__FreeBSD__)
        true;
#else
        false;
#error "This target os is untested"
#endif
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLERR_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollErr;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = -1;

    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLERR_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollErr;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLWRNORM_POLLERR_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollErr;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLWRNORM_POLLERR_negative1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollErr;
    int timeout = -1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = -1;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

// 0 seconds; return immediately

TEST(PollBeforeConnect, ZERO_0second_timeout)
{
    UNIT_TEST_TRACE("");
    nn::socket::PollEvent events = nn::socket::PollEvent::PollNone;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLIN_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollIn;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLPRI_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollPri;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLIN_POLLPRI_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollPri;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLWRNORM_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollWrNorm;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLWRNORM_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollWrNorm;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLERR_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollErr;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        -1;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        0;
#elif defined(__FreeBSD__)
        0;
#else
        -1;
#error "This target os is untested"
#endif

    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLERR_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollErr;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        -1;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        0;
#elif defined(__FreeBSD__)
        0;
#else
        -1;
#error "This target os is untested"
#endif

    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLWRNORM_POLLERR_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollErr;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        -1;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        0;
#elif defined(__FreeBSD__)
        0;
#else
        -1;
#error "This target os is untested"
#endif

    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLWRNORM_POLLERR_0second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollErr;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        -1;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        0;
#elif defined(__FreeBSD__)
        0;
#else
        -1;
#error "This target os is untested"
#endif
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

// wait for 1 second
TEST(PollBeforeConnect, ZERO_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollNone;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLIN_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollIn;
    int timeout = 0;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLPRI_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollPri;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLLIN_POLLPRI_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollPri;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLWRNORM_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollWrNorm;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLWRNORM_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollWrNorm;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue = 0;
    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLERR_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollErr;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        -1;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        0;
#elif defined(__FreeBSD__)
        0;
#else
        -1;
#error "This target os is untested"
#endif

    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLERR_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollErr;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        -1;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        0;
#elif defined(__FreeBSD__)
        0;
#else
        -1;
#error "This target os is untested"
#endif

    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLWRNORM_POLLERR_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollErr;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        -1;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        0;
#elif defined(__FreeBSD__)
        0;
#else
        -1;
#error "This target os is untested"
#endif

    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

TEST(PollBeforeConnect, POLL_POLLRDNORM_POLLWRNORM_POLLERR_1second_timeout)
{
    UNIT_TEST_TRACE("");

    nn::socket::PollEvent events = nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollErr;
    int timeout = 1;
    bool shouldCompleteBeforeTimeout = true;
    int pollExpectedReturnValue =
#if defined(NN_BUILD_CONFIG_OS_WIN)
        -1;
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        0;
#elif defined(__FreeBSD__)
        0;
#else
        -1;
#error "This target os is untested"
#endif

    RunPollBeforeConnectUnitTest(events, timeout, g_WaitForDoneTimeoutInSeconds, shouldCompleteBeforeTimeout, pollExpectedReturnValue);
};

}}
