﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
  Test process for Network
  *---------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/socket.h>
#include <nn/nn_Log.h>
#include <nn/os/os_Thread.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/socket/socket_ApiPrivate.h>

#include "testNet_ApiCommon.h"
#include "natf/Utils/md5.h"
#include "natf/Utils/CommandLineParser.h"
#include "natf/Utils/InitApis.h"
#include <nn/dns/parser.h>
#include <nnc/dns/parser.h>

namespace NATF { namespace API {

TEST(DnsParser, CreateInaddrArpaStringTest)
{
    const char* correct = "101.120.128.138.in-addr.arpa"; // landsberger.com
    uint32_t ip;
    nn::socket::InetPton(nn::socket::Family::Af_Inet, "138.128.120.101", &ip);
    char arpaBuffer[static_cast<int>(nn::dns::parser::LibraryConstant::InaddrArpaStringBufferSize)] = { 0 };

    ASSERT_TRUE(-1 != nn::dns::parser::CreateInaddrArpaString(arpaBuffer, sizeof(arpaBuffer), ip));
    ASSERT_TRUE(0 == strcmp(correct, arpaBuffer));
};

TEST(DnsParser, MemoryBlockEqualTest)
{
    nn::dns::parser::MemoryBlock a = {reinterpret_cast<const uint8_t*>(0x12345678), reinterpret_cast<const uint8_t*>(0x23456789)};
    nn::dns::parser::MemoryBlock b = {reinterpret_cast<const uint8_t*>(0x12345678), reinterpret_cast<const uint8_t*>(0x23456789)};
    nn::dns::parser::MemoryBlock c = {reinterpret_cast<const uint8_t*>(0x12345678), reinterpret_cast<const uint8_t*>(0x23456780)};

    ASSERT_TRUE(MemoryBlockIsEqual(&a, &b));
    ASSERT_FALSE(MemoryBlockIsEqual(&a, &c));
};

TEST(DnsParser, HeaderTest)
{
    nn::dns::parser::Header h1, h2;
    const uint8_t buffer[] = { 0x96, 0xac,  // transaction id
                               0x81, 0x80,  // flags
                               0x00, 0x01,  // question count
                               0x00, 0x02,  // answer count
                               0x00, 0x03,  // authority count
                               0x00, 0x04 }; // additional count
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    // initialize empty object
    h1.Initialize(nullptr);
    h2.Initialize(nullptr);

    // empty object tests
    ASSERT_TRUE(h1.SizeOf() == h2.SizeOf());
    ASSERT_TRUE(h1 == h2);
    ASSERT_TRUE(nullptr == h1.GetMessage());
    ASSERT_TRUE(nullptr == h2.GetMessage());
    ASSERT_TRUE(h1 == h1);
    ASSERT_TRUE(h2 == h2);

    // read the object from a buffer
    ASSERT_TRUE(-1 != h1.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(h1.GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(h1.GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(h1.GetQuestionCount()   == 1);
    ASSERT_TRUE(h1.GetAnswerCount()     == 2);
    ASSERT_TRUE(h1.GetAuthorityCount()  == 3);
    ASSERT_TRUE(h1.GetAdditionalCount() == 4);
    ASSERT_TRUE(h1 == h1);

    // send the object to another buffer and compare
    ASSERT_TRUE(sizeof(buffer) == h1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(-1 != h2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(h1 == h2);
    ASSERT_TRUE(sizeof(buffer) == h2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    h2 = nn::dns::parser::Header();
    ASSERT_FALSE(h1 == h2);

    // assign and compare
    h2 = h1;
    ASSERT_TRUE(h1 == h2);
};

TEST(DnsParser, LabelTest)
{
    nn::dns::parser::Label l1, l2;

    const uint8_t buffer[] = { 0x03, 0x77, 0x77, 0x77, // label1: www
                               0x03, 0x66, 0x6F, 0x6F, // label2: foo
                               0x02, 0x75, 0x73,       // label3: us
                               0x00 };                 // label: end

    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    char outName[sizeof(name)] = { 0 };

    uint8_t dataBuffer[sizeof(name) + static_cast<int>(nn::dns::parser::LabelConstant::StringLabelPadding)] = { 0 };

    // initialize empty object
    l1.Initialize(nullptr);
    l2.Initialize(nullptr);

    // empty object tests
    ASSERT_TRUE(l1.SizeOf() == l2.SizeOf());
    ASSERT_TRUE(l1 == l2);
    ASSERT_TRUE(nullptr == l1.GetMessage());
    ASSERT_TRUE(nullptr == l2.GetMessage());
    ASSERT_TRUE(l1 == l1);
    ASSERT_TRUE(l2 == l2);

    // read the object from a buffer
    ASSERT_TRUE(-1 != l1.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(sizeof(name) == l1.GetStringBufferSize());

    ASSERT_TRUE(strlen(name) + 2 == l1.SizeOf());
    ASSERT_TRUE(-1 != l1.ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(l1 == l1);

    // send the object to another buffer and compare
    ASSERT_TRUE(sizeof(buffer) == l1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(-1 != l2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(l1 == l2);
    ASSERT_TRUE(sizeof(buffer) == l2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a string
    ASSERT_TRUE(-1 != nn::dns::parser::Label::ToBufferFromDomainNameString(dataBuffer, sizeof(dataBuffer), name));
    ASSERT_TRUE(-1 != l2.FromBuffer(dataBuffer, sizeof(dataBuffer)));
    //ASSERT_TRUE(-1 != l2.FromString(name));
    ASSERT_TRUE(sizeof(name) == l2.GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == l2.SizeOf());
    ASSERT_TRUE(-1 != l2.ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));

    // clear and compare
    l2 = nn::dns::parser::Label();
    ASSERT_FALSE(l1 == l2);

    // assign and compare
    l2 = l1;
    ASSERT_TRUE(l1 == l2);
};

TEST(DnsParser, LongName1)
{
    const char bigdomain[] = "012345678901234567890123456789012345678901234567890123456789012."
        "012345678901234567890123456789012345678901234567890123456789012."
        "012345678901234567890123456789012345678901234.com";
    uint8_t dataBuffer[sizeof(bigdomain) + static_cast<int>(nn::dns::parser::LabelConstant::StringLabelPadding)] = { 0 };
    char outString[sizeof(bigdomain)] = { 0 };
    nn::dns::parser::Label l1;
    l1.Initialize(nullptr);

    ASSERT_TRUE(-1 != nn::dns::parser::Label::ToBufferFromDomainNameString(dataBuffer, sizeof(dataBuffer), bigdomain));
    ASSERT_TRUE(-1 != l1.FromBuffer(dataBuffer, sizeof(dataBuffer)));
    ASSERT_TRUE(-1 != l1.ToString(outString, sizeof(outString)));
    ASSERT_TRUE(0 == strcmp(bigdomain, outString));
};

TEST(DnsParser, LongName2)
{
    nn::dns::parser::Message m1, m2;

    const uint8_t buffer[] = { 0x71, 0xf3, //txid
                               0x81, 0x80, //flags
                               0x00, 0x01, //qcount
                               0x00, 0x01, //ancount
                               0x00, 0x00, //nscount
                               0x00, 0x00, //adcount
                               //qlabel part 1 - 63 characters
                               0x3f, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34,
                               0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30,
                               0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36,
                               0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32,
                               //qlabel part 2 - 63 characters
                               0x3f, 0x39, 0x38, 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, 0x39, 0x38, 0x37, 0x36, 0x35,
                               0x34, 0x33, 0x32, 0x31, 0x30, 0x39, 0x38, 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, 0x39,
                               0x38, 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, 0x39, 0x38, 0x37, 0x36, 0x35, 0x34, 0x33,
                               0x32, 0x31, 0x31, 0x39, 0x38, 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x30, 0x39, 0x38, 0x37,
                               //qlabel part 3 - 45 characters
                               0x2d, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34,
                               0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30,
                               0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x30, 0x31, 0x32, 0x33, 0x34,
                               //qlabel part 4 - 3 characters
                               0x03, 0x63, 0x6f, 0x6d,
                               //qlabel part 5 - null
                               0x00,
                               0x00, 0x01, //qtype
                               0x00, 0x01, //qclass
                               0xc0, 0x0c, //label pointer
                               0x00, 0x01, //type
                               0x00, 0x01, //class
                               0x00, 0x00, 0x00, 0xcd, //ttl
                               0x00, 0x04, //length
                               0xd8, 0x3a, 0xc1, 0x4e //data
    };

    uint8_t outBuffer[sizeof(buffer)] = { 0 };
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));
    ASSERT_TRUE(-1 != m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);

    nn::dns::parser::Label l1;
    const char name[] = "012345678901234567890123456789012345678901234567890123456789012.987654321098765432109876543210987654321098765432109876543210987.012345678901234567890123456789012345678901234.com";
    uint8_t dataBuffer[sizeof(name) + 2] = { 0 };

    ASSERT_TRUE(-1 != nn::dns::parser::Label::ToBufferFromDomainNameString(dataBuffer, sizeof(dataBuffer), name));
    ASSERT_TRUE(-1 != l1.FromBuffer(dataBuffer, sizeof(dataBuffer)));
    int rc;

    nn::dns::parser::RecordIterator riterator;
    for (rc = riterator.Initialize(m1,
                                   nn::dns::parser::MessageSectionConstant::Answer);
         1 == rc;
         rc = riterator.GetNext())
    {
        nn::dns::parser::Record* pRecord = &riterator.GetCurrent();
        size_t nameStringSize = pRecord->GetName().GetStringBufferSize();
        char* nameString = (char*)alloca(nameStringSize);
        pRecord->GetName().ToString(nameString, nameStringSize);
    };
};

TEST(DnsParser, CircularLabel)
{
    nn::dns::parser::Message m1;
    nn::dns::parser::Label l1;

    const uint8_t msgBuffer[] = { 0x96, 0xac,               // transaction id
                                  0x81, 0x80,               // flags
                                  0x00, 0x01,               // qcount
                                  0x00, 0x00,               // ancount
                                  0x00, 0x00,               // nscount
                                  0x00, 0x00,               // adcount
                                  0xC0, 0x0C,               // qlabel: pointer to self (will fail)
                                  0x00, 0x01,               // qtype
                                  0x00, 0x01 };               // qclass

    m1.Initialize();

    ASSERT_TRUE(-1 == m1.FromBuffer(msgBuffer, sizeof(msgBuffer)));
};

/*
TEST(DnsParser, OverlapLeftLabel)
{
nn::dns::parser::Message m1;
nn::dns::parser::Label l1;

const uint8_t msgBuffer[] = { 0x96, 0xac,               // transaction id
                              0x81, 0x80,               // flags
                              0x00, 0x01,               // qcount
                              0x00, 0x01,               // ancount
                              0x00, 0x00,               // nscount
                              0x00, 0x00,               // adcount
                              0x03, 0x77, 0x77, 0x77,   // qlabel: www
                              0x03, 0x66, 0x6F, 0x6F,   // qlabel: foo
                              0x02, 0x75, 0x73,         // qlabel: us
                              0x00,                     // qlabel: null
                              0x00, 0x01,               // qtype
                              0x00, 0x01,               // qclass
                              0xC0, 0x0B,               // label pointer -- octet 0x0B is NOT a label, but 0x0C is. You cannot have overlapping pointers.
                              0x00, 0x01,               // type (address)
                              0x00, 0x01,               // class (internet)
                              0x00, 0x00, 0x00, 0x8E,   // ttl: 142
                              0x00, 0x04,               // data length
                              0xD8, 0x3A, 0xC1, 0x44 }; // 216.58.193.68

    m1.Initialize();

    ASSERT_TRUE(-1 == m1.FromBuffer(msgBuffer, sizeof(msgBuffer)));
};

TEST(DnsParser, OverlapRightLabel)
{
    nn::dns::parser::Message m1;
    nn::dns::parser::Label l1;

    const uint8_t msgBuffer[] = { 0x96, 0xac,               // transaction id
                                  0x81, 0x80,               // flags
                                  0x00, 0x01,               // qcount
                                  0x00, 0x01,               // ancount
                                  0x00, 0x00,               // nscount
                                  0x00, 0x00,               // adcount
                                  0x03, 0x77, 0x77, 0x77,   // qlabel: www
                                  0x03, 0x66, 0x6F, 0x6F,   // qlabel: foo
                                  0x02, 0x75, 0x73,         // qlabel: us
                                  0x00,                     // qlabel: null
                                  0x00, 0x01,               // qtype
                                  0x00, 0x01,               // qclass
                                  0xC0, 0x16,               // label pointer -- octet 0x15 is part of a label but, but 0x17 isn't. You cannot have overlapping pointers.
                                  0x00, 0x01,               // type (address)
                                  0x00, 0x01,               // class (internet)
                                  0x00, 0x00, 0x00, 0x8E,   // ttl: 142
                                  0x00, 0x04,               // data length
                                  0xD8, 0x3A, 0xC1, 0x44 }; // 216.58.193.68

    m1.Initialize();

    ASSERT_TRUE(-1 == m1.FromBuffer(msgBuffer, sizeof(msgBuffer)));
};
*/

TEST(DnsParser, LabelGraphCycle)
{
    nn::dns::parser::Message m1;
    nn::dns::parser::Label l1;

    const uint8_t msgBuffer[] = { 0x96, 0xac,                    // transaction id
                                  0x81, 0x80,                    // flags
                                  0x00, 0x01,                    // qcount
                                  0x00, 0x01,                    // ancount
                                  0x00, 0x00,                    // nscount
                                  0x00, 0x00,                    // adcount
                                  0x03, 0x77, 0x77, 0x77,        // qlabel: www
                                  0xC0, 0x15,                     // qlabel: pointer into record
                                  0x00, 0x01,                    // qtype
                                  0x00, 0x01,                    // qclass
                                  0x04, 0x62, 0x6c, 0x61, 0x68,  // label: blah
                                  0xC0, 0x0C,                    // pointer back to question
                                  0x00, 0x01,                    // type (address)
                                  0x00, 0x01,                    // class (internet)
                                  0x00, 0x00, 0x00, 0x8E,        // ttl: 142
                                  0x00, 0x04,                    // data length
                                  0xD8, 0x3A, 0xC1, 0x44 };      // 216.58.193.68

    m1.Initialize();

    ASSERT_TRUE(-1 == m1.FromBuffer(msgBuffer, sizeof(msgBuffer)));
};

TEST(DnsParser, LabelCompare)
{
    nn::dns::parser::Label l1, l2, l3;
    nn::dns::parser::Message m1;

    const uint8_t msgBuffer[] = { 0x96, 0xac,               // transaction id
                                  0x81, 0x80,               // flags
                                  0x00, 0x01,               // qcount
                                  0x00, 0x01,               // ancount
                                  0x00, 0x00,               // nscount
                                  0x00, 0x00,               // adcount
                                  0x03, 0x77, 0x77, 0x77,   // qlabel: www
                                  0x03, 0x66, 0x6F, 0x6F,   // qlabel: foo
                                  0x02, 0x75, 0x73,         // qlabel: us
                                  0x00,                     // qlabel: null
                                  0x00, 0x01,               // qtype
                                  0x00, 0x01,               // qclass
                                  0xC0, 0x0C,               // label pointer
                                  0x00, 0x01,               // type (address)
                                  0x00, 0x01,               // class (internet)
                                  0x00, 0x00, 0x00, 0x8E,   // ttl: 142
                                  0x00, 0x04,               // data length
                                  0xD8, 0x3A, 0xC1, 0x44 }; // 216.58.193.68

    const uint8_t labelString[] = { 0x03, 0x77, 0x77, 0x77, // label1: www
                                    0x03, 0x66, 0x6F, 0x6F, // label2: foo
                                    0x02, 0x75, 0x73,       // label3: us
                                    0x00 };                 // label: end

    nn::dns::parser::QuestionIterator qiter;
    nn::dns::parser::RecordIterator riter;

    // initialize empty object
    m1.Initialize();
    l1.Initialize(nullptr);
    l2.Initialize(&m1);

    // set up
    ASSERT_TRUE(-1 != m1.FromBuffer(msgBuffer, sizeof(msgBuffer)));
    ASSERT_TRUE(-1 != l1.FromBuffer(labelString, sizeof(labelString)));

    uint8_t pointer_to_question[] = { 0xC0, 0x0C };
    ASSERT_TRUE(-1 != l2.FromBuffer(pointer_to_question, sizeof(pointer_to_question)));

    // compare
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(qiter.GetCurrent().GetName() == l1);
    ASSERT_TRUE(l1 == qiter.GetCurrent().GetName());

    ASSERT_TRUE(qiter.GetCurrent().GetName() == l2);
    ASSERT_TRUE(l2 == qiter.GetCurrent().GetName());


    ASSERT_TRUE(1 == riter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Answer));

    ASSERT_TRUE(qiter.GetCurrent().GetName() == riter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());

    ASSERT_TRUE(l1  == riter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetName() == l1);

    ASSERT_TRUE(l2  == riter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetName() == l2);

    ASSERT_TRUE(l2 == l2);

    l3.Initialize(&m1);
    uint8_t pointer_to_question2[] = { 0xC0, 0x0C };
    ASSERT_TRUE(-1 != l3.FromBuffer(pointer_to_question2, sizeof(pointer_to_question2)));
    ASSERT_TRUE(l3 == l2);
    ASSERT_TRUE(l2 == l3);
};

TEST(DnsParser, QuestionTest)
{
    nn::dns::parser::Question q1, q2;
    const uint8_t buffer[] = { 0x03,0x77, 0x77, 0x77,  // label1: www
                               0x03, 0x66, 0x6F, 0x6F, // label2: foo
                               0x02, 0x75, 0x73,       // label3: us
                               0x00,                   // label: end
                               0x00, 0x01,             // type (address)
                               0x00, 0x01 };           // class (internet)
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    char outName[sizeof(name)] = { 0 };

    // initialize empty object
    q1.Initialize(nullptr);
    q2.Initialize(nullptr);

    // empty object tests
    ASSERT_TRUE(q1.SizeOf() == q2.SizeOf());
    ASSERT_TRUE(q1 == q2);
    ASSERT_TRUE(nullptr == q1.GetMessage());
    ASSERT_TRUE(nullptr == q2.GetMessage());
    ASSERT_TRUE(q1 == q1);
    ASSERT_TRUE(q2 == q2);

    // read the object from a buffer
    ASSERT_TRUE(-1 != q1.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(sizeof(name) == q1.GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == q1.GetName().SizeOf());
    ASSERT_TRUE(-1 != q1.GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == q1.GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == q1.GetClass());
    ASSERT_TRUE(q1 == q1);

    // send the object to another buffer and compare
    ASSERT_TRUE(sizeof(buffer) == q1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == q2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(q1 == q2);
    ASSERT_TRUE(sizeof(buffer) == q2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    q2 = nn::dns::parser::Question();
    ASSERT_FALSE(q1 == q2);

    // assign and compare
    q2 = q1;
    ASSERT_TRUE(q1 == q2);
};

TEST(DnsParser, RecordTest)
{
    // in the wild most records you will see have labels with pointers
    // however since there is no message for this test the record
    // needs to stand alone and so it has three labels with strings
    nn::dns::parser::Record r1, r2;
    const uint8_t buffer[] = { 0x03,0x77, 0x77, 0x77,    // label1: www
                               0x03, 0x66, 0x6F, 0x6F,   // label2: foo
                               0x02, 0x75, 0x73,         // label3: us
                               0x00,                     // label: end
                               0x00, 0x01,               // type (address)
                               0x00, 0x01,               // class (internet)
                               0x00, 0x00, 0x00, 0x8E,   // ttl: 142
                               0x00, 0x04,               // data length
                               0xD8, 0x3A, 0xC1, 0x44 }; // 216.58.193.68
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    char outName[sizeof(name)] = { 0 };

    // initialize empty object
    r1.Initialize(nullptr);
    r2.Initialize(nullptr);

    // empty object tests
    ASSERT_TRUE(r1.SizeOf() == r2.SizeOf());
    ASSERT_TRUE(r1 == r2);
    ASSERT_TRUE(nullptr == r1.GetMessage());
    ASSERT_TRUE(nullptr == r2.GetMessage());
    ASSERT_TRUE(r1 == r1);
    ASSERT_TRUE(r2 == r2);

    // read the object from a buffer
    ASSERT_TRUE(-1 != r1.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(-1 != r1.GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == r1.GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == r1.GetClass());
    ASSERT_TRUE(142 == r1.GetTimeToLive());
    ASSERT_TRUE(4 == r1.GetLength());
    ASSERT_TRUE(nn::socket::InetHtonl(0xD83AC144) == *reinterpret_cast<const uint32_t*>(r1.GetData()));

    // send the object to another buffer and compare
    ASSERT_TRUE(sizeof(buffer) == r1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(-1 != r2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(r1 == r2);
    ASSERT_TRUE(sizeof(buffer) == r2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    r2 = nn::dns::parser::Record();
    ASSERT_FALSE(r1 == r2);

    // assign and compare
    r2 = r1;
    ASSERT_TRUE(r1 == r2);
};

TEST(DnsParser, QuestionMessageTest)
{
    nn::dns::parser::Message m1, m2;
    const uint8_t buffer[] = { 0x96, 0xac,             // transaction id
                               0x81, 0x80,             // flags
                               0x00, 0x01,             // qcount
                               0x00, 0x00,             // ancount
                               0x00, 0x00,             // nscount
                               0x00, 0x00,             // adcount
                               0x03, 0x77, 0x77, 0x77, // qlabel: www
                               0x03, 0x66, 0x6F, 0x6F, // qlabel: foo
                               0x02, 0x75, 0x73,       // qlabel: us
                               0x00,                   // qlabel: null
                               0x00, 0x01,             // qtype
                               0x00, 0x01 };           // qclass
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    char outName[sizeof(name)] = { 0 };

    nn::dns::parser::QuestionIterator qiter;

    // initialize empty buffer
    m1.Initialize();
    m2.Initialize();

    // empty object tests
    ASSERT_TRUE(m1.SizeOf() == m2.SizeOf());
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(m2 == m2);

    // read the object from a buffer and test the various layers
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));
    // header
    ASSERT_TRUE(m1.GetHeader().GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(m1.GetHeader().GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(m1.GetHeader().GetQuestionCount()   == 1);
    ASSERT_TRUE(m1.GetHeader().GetAnswerCount()     == 0);
    ASSERT_TRUE(m1.GetHeader().GetAuthorityCount()  == 0);
    ASSERT_TRUE(m1.GetHeader().GetAdditionalCount() == 0);
    ASSERT_TRUE(m1.GetHeader() == m1.GetHeader());

    // question
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(sizeof(name) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // send the object to another buffer and compare
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(sizeof(buffer) == m2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    m2 = nn::dns::parser::Message();
    ASSERT_FALSE(m1 == m2);

    // assign and compare
    m2 = m1;
    ASSERT_TRUE(m1 == m2);
};

TEST(DnsParser, SingleAnswerRecordMessageTest)
{
    nn::dns::parser::Message m1, m2;
    const uint8_t buffer[] = { 0x96, 0xac,               // transaction id
                               0x81, 0x80,               // flags
                               0x00, 0x01,               // qcount
                               0x00, 0x01,               // ancount
                               0x00, 0x00,               // nscount
                               0x00, 0x00,               // adcount
                               0x03, 0x77, 0x77, 0x77,   // qlabel: www
                               0x03, 0x66, 0x6F, 0x6F,   // qlabel: foo
                               0x02, 0x75, 0x73,         // qlabel: us
                               0x00,                     // qlabel: null
                               0x00, 0x01,               // qtype
                               0x00, 0x01,               // qclass
                               0xC0, 0x0C,               // label pointer
                               0x00, 0x01,               // type (address)
                               0x00, 0x01,               // class (internet)
                               0x00, 0x00, 0x00, 0x8E,   // ttl: 142
                               0x00, 0x04,               // data length
                               0xD8, 0x3A, 0xC1, 0x44 }; // 216.58.193.68
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    char outName[sizeof(name)] = { 0 };

    nn::dns::parser::QuestionIterator qiter;
    nn::dns::parser::RecordIterator riter;

    // initialize empty buffer
    m1.Initialize();
    m2.Initialize();

    // empty object tests
    ASSERT_TRUE(m1.SizeOf() == m2.SizeOf());
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(m2 == m2);

    // read the object from a buffer and test the various layers
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));

    // header
    ASSERT_TRUE(m1.GetHeader().GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(m1.GetHeader().GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(m1.GetHeader().GetQuestionCount()   == 1);
    ASSERT_TRUE(m1.GetHeader().GetAnswerCount()     == 1);
    ASSERT_TRUE(m1.GetHeader().GetAuthorityCount()  == 0);
    ASSERT_TRUE(m1.GetHeader().GetAdditionalCount() == 0);
    ASSERT_TRUE(m1.GetHeader() == m1.GetHeader());

    // question
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(sizeof(name) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // compare record
    ASSERT_TRUE(1 == riter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Answer));
    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetType() == qiter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(142 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(4 == riter.GetCurrent().GetLength());
    ASSERT_TRUE(nn::socket::InetHtonl(0xD83AC144) == *reinterpret_cast<const uint32_t*>(riter.GetCurrent().GetData()));

    // send the object to another buffer and compare
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(sizeof(buffer) == m2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    m2 = nn::dns::parser::Message();
    ASSERT_FALSE(m1 == m2);

    // assign and compare
    m2 = m1;
    ASSERT_TRUE(m1.GetBuffer() == m2.GetBuffer());
    ASSERT_TRUE(m1.GetBufferSize() == m2.GetBufferSize());
    ASSERT_TRUE(m1 == m2);
};

TEST(DnsParser, SingleAuthorityRecordMessageTest)
{
    nn::dns::parser::Message m1, m2;
    const uint8_t buffer[] = { 0x96, 0xac,               // transaction id
                               0x81, 0x80,               // flags
                               0x00, 0x01,               // qcount
                               0x00, 0x00,               // ancount
                               0x00, 0x01,               // nscount
                               0x00, 0x00,               // adcount
                               0x03, 0x77, 0x77, 0x77,   // qlabel: www
                               0x03, 0x66, 0x6F, 0x6F,   // qlabel: foo
                               0x02, 0x75, 0x73,         // qlabel: us
                               0x00,                     // qlabel: null
                               0x00, 0x01,               // qtype
                               0x00, 0x01,               // qclass
                               0xC0, 0x0C,               // label pointer
                               0x00, 0x01,               // type (address)
                               0x00, 0x01,               // class (internet)
                               0x00, 0x00, 0x00, 0x8E,   // ttl: 142
                               0x00, 0x04,               // data length
                               0xD8, 0x3A, 0xC1, 0x44 }; // 216.58.193.68
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    char outName[sizeof(name)] = { 0 };

    nn::dns::parser::QuestionIterator qiter;
    nn::dns::parser::RecordIterator riter;

    // initialize empty buffer
    m1.Initialize();
    m2.Initialize();

    // empty object tests
    ASSERT_TRUE(m1.SizeOf() == m2.SizeOf());
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(m2 == m2);

    // read the object from a buffer and test the various layers
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));

    // header
    ASSERT_TRUE(m1.GetHeader().GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(m1.GetHeader().GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(m1.GetHeader().GetQuestionCount()   == 1);
    ASSERT_TRUE(m1.GetHeader().GetAnswerCount()     == 0);
    ASSERT_TRUE(m1.GetHeader().GetAuthorityCount()  == 1);
    ASSERT_TRUE(m1.GetHeader().GetAdditionalCount() == 0);
    ASSERT_TRUE(m1.GetHeader() == m1.GetHeader());

    // question
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(sizeof(name) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // compare record
    ASSERT_TRUE(1 == riter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Authority));
    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetType() == qiter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(142 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(4 == riter.GetCurrent().GetLength());
    ASSERT_TRUE(nn::socket::InetHtonl(0xD83AC144) == *reinterpret_cast<const uint32_t*>(riter.GetCurrent().GetData()));

    // send the object to another buffer and compare
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(sizeof(buffer) == m2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    m2 = nn::dns::parser::Message();
    ASSERT_FALSE(m1 == m2);

    // assign and compare
    m2 = m1;
    ASSERT_TRUE(m1.GetBuffer() == m2.GetBuffer());
    ASSERT_TRUE(m1.GetBufferSize() == m2.GetBufferSize());
    ASSERT_TRUE(m1 == m2);
};

TEST(DnsParser, SingleAdditionalRecordMessageTest)
{
    nn::dns::parser::Message m1, m2;
    const uint8_t buffer[] = { 0x96, 0xac,               // transaction id
                               0x81, 0x80,               // flags
                               0x00, 0x01,               // qcount
                               0x00, 0x00,               // ancount
                               0x00, 0x00,               // nscount
                               0x00, 0x01,               // adcount
                               0x03, 0x77, 0x77, 0x77,   // qlabel: www
                               0x03, 0x66, 0x6F, 0x6F,   // qlabel: foo
                               0x02, 0x75, 0x73,         // qlabel: us
                               0x00,                     // qlabel: null
                               0x00, 0x01,               // qtype
                               0x00, 0x01,               // qclass
                               0xC0, 0x0C,               // label pointer
                               0x00, 0x01,               // type (address)
                               0x00, 0x01,               // class (internet)
                               0x00, 0x00, 0x00, 0x8E,   // ttl: 142
                               0x00, 0x04,               // data length
                               0xD8, 0x3A, 0xC1, 0x44 }; // 216.58.193.68
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    char outName[sizeof(name)] = { 0 };

    nn::dns::parser::QuestionIterator qiter;
    nn::dns::parser::RecordIterator riter;

    // initialize empty buffer
    m1.Initialize();
    m2.Initialize();

    // empty object tests
    ASSERT_TRUE(m1.SizeOf() == m2.SizeOf());
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(m2 == m2);

    // read the object from a buffer and test the various layers
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));

    // header
    ASSERT_TRUE(m1.GetHeader().GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(m1.GetHeader().GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(m1.GetHeader().GetQuestionCount()   == 1);
    ASSERT_TRUE(m1.GetHeader().GetAnswerCount()     == 0);
    ASSERT_TRUE(m1.GetHeader().GetAuthorityCount()  == 0);
    ASSERT_TRUE(m1.GetHeader().GetAdditionalCount() == 1);
    ASSERT_TRUE(m1.GetHeader() == m1.GetHeader());

    // question
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(sizeof(name) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // compare record
    ASSERT_TRUE(1 == riter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Additional));
    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetType() == qiter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(142 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(4 == riter.GetCurrent().GetLength());
    ASSERT_TRUE(nn::socket::InetHtonl(0xD83AC144) == *reinterpret_cast<const uint32_t*>(riter.GetCurrent().GetData()));

    // send the object to another buffer and compare
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(sizeof(buffer) == m2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    m2 = nn::dns::parser::Message();
    ASSERT_FALSE(m1 == m2);

    // assign and compare
    m2 = m1;
    ASSERT_TRUE(m1.GetBuffer() == m2.GetBuffer());
    ASSERT_TRUE(m1.GetBufferSize() == m2.GetBufferSize());
    ASSERT_TRUE(m1 == m2);
};

TEST(DnsParser, MultipleAnswerRecordMessageTest)
{
    nn::dns::parser::Message m1, m2;
    const uint8_t buffer[] = { 0x96, 0xac,                   // transaction id
                               0x81, 0x80,                   // flags
                               0x00, 0x01,                   // qcount
                               0x00, 0x02,                   // ancount
                               0x00, 0x00,                   // nscount
                               0x00, 0x00,                   // adcount
                               0x03, 0x77, 0x77, 0x77,       // qlabel: www
                               0x03, 0x66, 0x6F, 0x6F,       // qlabel: foo
                               0x02, 0x75, 0x73,             // qlabel: us
                               0x00,                         // qlabel: null
                               0x00, 0x01,                   // qtype
                               0x00, 0x01,                   // qclass
                               // answer #1
                               0xC0, 0x0C,                   // label pointer
                               0x00, 0x05,                   // type (cname)
                               0x00, 0x01,                   // class (internet)
                               0x00, 0x00, 0x00, 0x44,       // ttl (68)
                               0x00, 0x09,                   // length: 9 bytes
                               0x04, 0x62, 0x6c, 0x61, 0x68, // label: blah
                               0x02, 0x6c, 0x79,             // label: ly
                               0x00,                         // label: null
                               // answer #2
                               0xC0, 0x0C,                   // label pointer
                               0x00, 0x01,                   // type (address)
                               0x00, 0x01,                   // class (internet)
                               0x00, 0x00, 0x00, 0x8E,       // ttl: 142
                               0x00, 0x04,                   // data length
                               0xD8, 0x3A, 0xC1, 0x44 };     // 216.58.193.68
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    const char name2[] = "blah.ly";

    char outName[sizeof(name)] = { 0 };

    nn::dns::parser::QuestionIterator qiter;
    nn::dns::parser::RecordIterator riter;

    nn::dns::parser::Label label;

    // initialize empty buffer
    m1.Initialize();
    m2.Initialize();

    // empty object tests
    ASSERT_TRUE(m1.SizeOf() == m2.SizeOf());
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(m2 == m2);

    // read the object from a buffer and test the various layers
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));

    // header
    ASSERT_TRUE(m1.GetHeader().GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(m1.GetHeader().GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(m1.GetHeader().GetQuestionCount()   == 1);
    ASSERT_TRUE(m1.GetHeader().GetAnswerCount()     == 2);
    ASSERT_TRUE(m1.GetHeader().GetAuthorityCount()  == 0);
    ASSERT_TRUE(m1.GetHeader().GetAdditionalCount() == 0);
    ASSERT_TRUE(m1.GetHeader() == m1.GetHeader());

    // question
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(sizeof(name) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // compare record
    ASSERT_TRUE(1 == riter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Answer));
    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(nn::dns::parser::TypeConstant::CanonicalName == riter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(68 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(9 == riter.GetCurrent().GetLength());
    // foo.us pointer
    ASSERT_TRUE(-1 != riter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    // cname label in data
    ASSERT_TRUE(-1 != label.FromBuffer(riter.GetCurrent().GetData(), riter.GetCurrent().GetLength()));
    ASSERT_TRUE(-1 != label.ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name2, outName));

    ASSERT_TRUE(1 == riter.GetNext());

    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetType() == qiter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(142 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(4 == riter.GetCurrent().GetLength());
    ASSERT_TRUE(nn::socket::InetHtonl(0xD83AC144) == *reinterpret_cast<const uint32_t*>(riter.GetCurrent().GetData()));

    // send the object to another buffer and compare
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(sizeof(buffer) == m2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    m2 = nn::dns::parser::Message();
    ASSERT_FALSE(m1 == m2);

    // assign and compare
    m2 = m1;
    ASSERT_TRUE(m1.GetBuffer() == m2.GetBuffer());
    ASSERT_TRUE(m1.GetBufferSize() == m2.GetBufferSize());
    ASSERT_TRUE(m1 == m2);
};

TEST(DnsParser, MultipleAuthorityRecordMessageTest)
{
    nn::dns::parser::Message m1, m2;
    const uint8_t buffer[] = { 0x96, 0xac,                   // transaction id
                               0x81, 0x80,                   // flags
                               0x00, 0x01,                   // qcount
                               0x00, 0x00,                   // ancount
                               0x00, 0x02,                   // nscount
                               0x00, 0x00,                   // adcount
                               0x03, 0x77, 0x77, 0x77,       // qlabel: www
                               0x03, 0x66, 0x6F, 0x6F,       // qlabel: foo
                               0x02, 0x75, 0x73,             // qlabel: us
                               0x00,                         // qlabel: null
                               0x00, 0x01,                   // qtype
                               0x00, 0x01,                   // qclass
                               // authority #1
                               0xC0, 0x0C,                   // label pointer
                               0x00, 0x05,                   // type (cname)
                               0x00, 0x01,                   // class (internet)
                               0x00, 0x00, 0x00, 0x44,       // ttl (68)
                               0x00, 0x09,                   // length: 9 bytes
                               0x04, 0x62, 0x6c, 0x61, 0x68, // label: blah
                               0x02, 0x6c, 0x79,             // label: ly
                               0x00,                         // label: null
                               // authority #2
                               0xC0, 0x0C,                   // label pointer
                               0x00, 0x01,                   // type (address)
                               0x00, 0x01,                   // class (internet)
                               0x00, 0x00, 0x00, 0x8E,       // ttl: 142
                               0x00, 0x04,                   // data length
                               0xD8, 0x3A, 0xC1, 0x44 };     // 216.58.193.68
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    const char name2[] = "blah.ly";

    char outName[sizeof(name)] = { 0 };

    nn::dns::parser::QuestionIterator qiter;
    nn::dns::parser::RecordIterator riter;

    nn::dns::parser::Label label;

    // initialize empty buffer
    m1.Initialize();
    m2.Initialize();

    // empty object tests
    ASSERT_TRUE(m1.SizeOf() == m2.SizeOf());
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(m2 == m2);

    // read the object from a buffer and test the various layers
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));

    // header
    ASSERT_TRUE(m1.GetHeader().GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(m1.GetHeader().GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(m1.GetHeader().GetQuestionCount()   == 1);
    ASSERT_TRUE(m1.GetHeader().GetAnswerCount()     == 0);
    ASSERT_TRUE(m1.GetHeader().GetAuthorityCount()  == 2);
    ASSERT_TRUE(m1.GetHeader().GetAdditionalCount() == 0);
    ASSERT_TRUE(m1.GetHeader() == m1.GetHeader());

    // question
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(sizeof(name) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // compare record
    ASSERT_TRUE(1 == riter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Authority));
    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(nn::dns::parser::TypeConstant::CanonicalName == riter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(68 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(9 == riter.GetCurrent().GetLength());
    // foo.us pointer
    ASSERT_TRUE(-1 != riter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    // cname label in data
    ASSERT_TRUE(-1 != label.FromBuffer(riter.GetCurrent().GetData(), riter.GetCurrent().GetLength()));
    ASSERT_TRUE(-1 != label.ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name2, outName));

    ASSERT_TRUE(1 == riter.GetNext());

    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetType() == qiter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(142 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(4 == riter.GetCurrent().GetLength());
    ASSERT_TRUE(nn::socket::InetHtonl(0xD83AC144) == *reinterpret_cast<const uint32_t*>(riter.GetCurrent().GetData()));

    // send the object to another buffer and compare
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(sizeof(buffer) == m2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    m2 = nn::dns::parser::Message();
    ASSERT_FALSE(m1 == m2);

    // assign and compare
    m2 = m1;
    ASSERT_TRUE(m1.GetBuffer() == m2.GetBuffer());
    ASSERT_TRUE(m1.GetBufferSize() == m2.GetBufferSize());
    ASSERT_TRUE(m1 == m2);
};

TEST(DnsParser, MultipleAdditionalRecordMessageTest)
{
    nn::dns::parser::Message m1, m2;
    const uint8_t buffer[] = { 0x96, 0xac,                   // transaction id
                               0x81, 0x80,                   // flags
                               0x00, 0x01,                   // qcount
                               0x00, 0x00,                   // ancount
                               0x00, 0x00,                   // nscount
                               0x00, 0x02,                   // adcount
                               0x03, 0x77, 0x77, 0x77,       // qlabel: www
                               0x03, 0x66, 0x6F, 0x6F,       // qlabel: foo
                               0x02, 0x75, 0x73,             // qlabel: us
                               0x00,                         // qlabel: null
                               0x00, 0x01,                   // qtype
                               0x00, 0x01,                   // qclass
                               // authority #1
                               0xC0, 0x0C,                   // label pointer
                               0x00, 0x05,                   // type (cname)
                               0x00, 0x01,                   // class (internet)
                               0x00, 0x00, 0x00, 0x44,       // ttl (68)
                               0x00, 0x09,                   // length: 9 bytes
                               0x04, 0x62, 0x6c, 0x61, 0x68, // label: blah
                               0x02, 0x6c, 0x79,             // label: ly
                               0x00,                         // label: null
                               // authority #2
                               0xC0, 0x0C,                   // label pointer
                               0x00, 0x01,                   // type (address)
                               0x00, 0x01,                   // class (internet)
                               0x00, 0x00, 0x00, 0x8E,       // ttl: 142
                               0x00, 0x04,                   // data length
                               0xD8, 0x3A, 0xC1, 0x44 };     // 216.58.193.68
    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    const char name2[] = "blah.ly";

    char outName[sizeof(name)] = { 0 };

    nn::dns::parser::QuestionIterator qiter;
    nn::dns::parser::RecordIterator riter;

    nn::dns::parser::Label label;

    // initialize empty buffer
    m1.Initialize();
    m2.Initialize();

    // empty object tests
    ASSERT_TRUE(m1.SizeOf() == m2.SizeOf());
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(m2 == m2);

    // read the object from a buffer and test the various layers
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));

    // header
    ASSERT_TRUE(m1.GetHeader().GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(m1.GetHeader().GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(m1.GetHeader().GetQuestionCount()   == 1);
    ASSERT_TRUE(m1.GetHeader().GetAnswerCount()     == 0);
    ASSERT_TRUE(m1.GetHeader().GetAuthorityCount()  == 0);
    ASSERT_TRUE(m1.GetHeader().GetAdditionalCount() == 2);
    ASSERT_TRUE(m1.GetHeader() == m1.GetHeader());

    // question
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(sizeof(name) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // compare record
    ASSERT_TRUE(1 == riter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Additional));
    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(nn::dns::parser::TypeConstant::CanonicalName == riter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(68 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(9 == riter.GetCurrent().GetLength());
    // foo.us pointer
    ASSERT_TRUE(-1 != riter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    // cname label in data
    ASSERT_TRUE(-1 != label.FromBuffer(riter.GetCurrent().GetData(), riter.GetCurrent().GetLength()));
    ASSERT_TRUE(-1 != label.ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name2, outName));

    ASSERT_TRUE(1 == riter.GetNext());

    ASSERT_TRUE(riter.GetCurrent().GetName() == qiter.GetCurrent().GetName());
    ASSERT_TRUE(riter.GetCurrent().GetType() == qiter.GetCurrent().GetType());
    ASSERT_TRUE(riter.GetCurrent().GetClass() == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(142 == riter.GetCurrent().GetTimeToLive());
    ASSERT_TRUE(4 == riter.GetCurrent().GetLength());
    ASSERT_TRUE(nn::socket::InetHtonl(0xD83AC144) == *reinterpret_cast<const uint32_t*>(riter.GetCurrent().GetData()));

    // send the object to another buffer and compare
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(sizeof(buffer) == m2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    m2 = nn::dns::parser::Message();
    ASSERT_FALSE(m1 == m2);

    // assign and compare
    m2 = m1;
    ASSERT_TRUE(m1.GetBuffer() == m2.GetBuffer());
    ASSERT_TRUE(m1.GetBufferSize() == m2.GetBufferSize());
    ASSERT_TRUE(m1 == m2);
};

TEST(DnsParser, QuestionMultipleMessageTest)
{
    nn::dns::parser::Message m1, m2;
    const uint8_t buffer[] = { 0x96, 0xac,                   // transaction id
                               0x81, 0x80,                   // flags
                               0x00, 0x02,                   // qcount
                               0x00, 0x00,                   // ancount
                               0x00, 0x00,                   // nscount
                               0x00, 0x00,                   // adcount
                               //question #1
                               0x03, 0x77, 0x77, 0x77,       // qlabel: www
                               0x03, 0x66, 0x6F, 0x6F,       // qlabel: foo
                               0x02, 0x75, 0x73,             // qlabel: us
                               0x00,                         // qlabel: null
                               0x00, 0x01,                   // qtype
                               0x00, 0x01,                   // qclass
                               // question #2
                               0x04, 0x62, 0x6c, 0x61, 0x68, // label: blah
                               0x02, 0x6c, 0x79,             // label: ly
                               0x00,                         // label: null
                               0x00, 0x01,                   // qtype
                               0x00, 0x01 };                 // qclass

    uint8_t outBuffer[sizeof(buffer)] = { 0 };

    const char name[] = "www.foo.us";
    const char name2[] = "blah.ly";

    char outName[sizeof(name)] = { 0 };

    nn::dns::parser::QuestionIterator qiter;
    nn::dns::parser::RecordIterator riter;

    nn::dns::parser::Label label;

    // initialize empty buffer
    m1.Initialize();
    m2.Initialize();

    // empty object tests
    ASSERT_TRUE(m1.SizeOf() == m2.SizeOf());
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(m2 == m2);

    // read the object from a buffer and test the various layers
    ASSERT_TRUE(-1 != m1.FromBuffer(buffer, sizeof(buffer)));

    // header
    ASSERT_TRUE(m1.GetHeader().GetTransactionId()   == 0x96ac);
    ASSERT_TRUE(m1.GetHeader().GetFlags()           ==
                ((nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsResponseRecursionAvailableShift) |
                 (nn::dns::parser::HeaderFlagsConstant::True << nn::dns::parser::HeaderFlagsConstant::IsRecursionDesiredShift)));
    ASSERT_TRUE(m1.GetHeader().GetQuestionCount()   == 2);
    ASSERT_TRUE(m1.GetHeader().GetAnswerCount()     == 0);
    ASSERT_TRUE(m1.GetHeader().GetAuthorityCount()  == 0);
    ASSERT_TRUE(m1.GetHeader().GetAdditionalCount() == 0);
    ASSERT_TRUE(m1.GetHeader() == m1.GetHeader());

    // question 1
    ASSERT_TRUE(1 == qiter.Initialize(m1, nn::dns::parser::MessageSectionConstant::Question));
    ASSERT_TRUE(sizeof(name) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // question 2
    ASSERT_TRUE(1 == qiter.GetNext());
    ASSERT_TRUE(sizeof(name2) == qiter.GetCurrent().GetName().GetStringBufferSize());
    ASSERT_TRUE(strlen(name2) + 2 == qiter.GetCurrent().GetName().SizeOf());
    ASSERT_TRUE(-1 != qiter.GetCurrent().GetName().ToString(outName, sizeof(outName)));
    ASSERT_TRUE(0 == strcmp(name2, outName));
    ASSERT_TRUE(nn::dns::parser::TypeConstant::Address == qiter.GetCurrent().GetType());
    ASSERT_TRUE(nn::dns::parser::ClassConstant::Internet == qiter.GetCurrent().GetClass());
    ASSERT_TRUE(qiter.GetCurrent() == qiter.GetCurrent());

    // send the object to another buffer and compare
    ASSERT_TRUE(m1 == m1);
    ASSERT_TRUE(-1 != m1.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // read the other object from a buffer and compare
    ASSERT_TRUE(sizeof(buffer) == m2.FromBuffer(buffer, sizeof(buffer)));
    ASSERT_TRUE(m1 == m2);
    ASSERT_TRUE(sizeof(buffer) == m2.ToBuffer(outBuffer, sizeof(outBuffer)));
    ASSERT_TRUE(0 == memcmp(buffer, outBuffer, sizeof(buffer)));

    // clear and compare
    m2 = nn::dns::parser::Message();
    ASSERT_FALSE(m1 == m2);

    // assign and compare
    m2 = m1;
    ASSERT_TRUE(m1.GetBuffer() == m2.GetBuffer());
    ASSERT_TRUE(m1.GetBufferSize() == m2.GetBufferSize());
    ASSERT_TRUE(m1 == m2);
};


}}; // namespace NATF::API
