﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "natf.h"

#include <cstring>
#include <cstdio>
#include <cstdarg>
#include <cstdlib> // For malloc
#include <new>     // For placement new
#include "Utils/ExchangeInfo.h"
#include <nn/nifm/nifm_ApiIpAddress.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_TimeZoneApi.h>
#endif

namespace NATF
{
    const char* const BaseTest::RemoteHostName = "HTCS_REMOTE";

    // Default Constructor
    BaseModule::BaseModule(bool useSelect) NN_NOEXCEPT
        : m_pSelectOrPollFn(nullptr),
          m_pNext(nullptr),
          m_result(ModuleResult_Incomplete),
          m_pThread(nullptr)
    {
        if( useSelect )
        {
            m_pSelectOrPollFn = BaseModule::SelectWithPollSignature;
        }
        else
        {
            m_pSelectOrPollFn = nn::socket::Poll;
        }
    }

    // SelectWithPollSignature
    int BaseModule::SelectWithPollSignature(NetTest::PollFd* pFds, nn::socket::NfdsT fdCount, int timeoutMs) NN_NOEXCEPT
    {
        uint32_t tryCount = 2;
        int rval = 0;

        do
        {
            int maxFd = -1;

            NetTest::FdSet readSet;
            NetTest::FdSet writeSet;
            NetTest::FdSet errSet;
            NetTest::FdSet* pReadSet  = nullptr;
            NetTest::FdSet* pWriteSet = nullptr;
            NetTest::FdSet* pErrSet   = nullptr;

            nn::socket::TimeVal timeout;

            nn::socket::FdSetZero(&readSet);
            nn::socket::FdSetZero(&writeSet);
            nn::socket::FdSetZero(&errSet);

            for(NetTest::SockLen iPollFd = 0; iPollFd < fdCount; ++iPollFd)
            {
                if( (int)pFds[iPollFd].fd > maxFd )
                {
                    maxFd = (int)pFds[iPollFd].fd;
                }

                if( (pFds[iPollFd].events & (nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollRdBand | nn::socket::PollEvent::PollPri)) != nn::socket::PollEvent::PollNone )
                {
                    pReadSet = &readSet;
                    nn::socket::FdSetSet(pFds[iPollFd].fd, &readSet);
                }

                if( (pFds[iPollFd].events & (nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollWrBand)) != nn::socket::PollEvent::PollNone )
                {
                    pWriteSet = &writeSet;
                    nn::socket::FdSetSet(pFds[iPollFd].fd, &writeSet);
                }

                if( (pFds[iPollFd].events & nn::socket::PollEvent::PollErr) != nn::socket::PollEvent::PollNone )
                {
                    pErrSet = &errSet;
                    nn::socket::FdSetSet(pFds[iPollFd].fd, &errSet);
                }
            }

            timeout.tv_sec  = timeoutMs * 1000;
            timeout.tv_usec = (timeoutMs % 1000) * 1000;

            rval = nn::socket::Select(maxFd + 1, pReadSet, pWriteSet, pErrSet, &timeout);

            for(NetTest::SockLen iPollFd = 0; iPollFd < fdCount; ++iPollFd)
            {
                pFds[iPollFd].revents = nn::socket::PollEvent::PollNone;

                if( nn::socket::FdSetIsSet(pFds[iPollFd].fd, &readSet) )
                {
                    pFds[iPollFd].revents |= nn::socket::PollEvent::PollRdNorm;
                }

                if( nn::socket::FdSetIsSet(pFds[iPollFd].fd, &writeSet) )
                {
                    pFds[iPollFd].revents |= nn::socket::PollEvent::PollWrNorm;
                }

                if( nn::socket::FdSetIsSet(pFds[iPollFd].fd, &errSet) )
                {
                    pFds[iPollFd].revents |= nn::socket::PollEvent::PollErr;
                }
            }

            --tryCount;

            // Temp until SIGLONTD-4358 is resolved.
            if( rval == -1 )
            {
                if( (tryCount > 0) && (NetTest::GetLastError() == nn::socket::Errno::EBadf) )
                {
                    NATF_LOG("\n * WARNING: Select() errored with nn::socket::Errno::EBadf, retrying...\n\n");
                }
                else
                {
                    break;
                }
            }
            else
            {
                break;
            }
        } while( tryCount > 0 );

        return rval;
    }

    // SelectOrPoll
    int BaseModule::SelectOrPoll(NetTest::PollFd* pFds, nn::socket::NfdsT fdCount, int timeoutMs) const NN_NOEXCEPT
    {
        return m_pSelectOrPollFn(pFds, fdCount, timeoutMs);
    }

    // GetRemoteIp
    const char* BaseModule::GetRemoteIp() const NN_NOEXCEPT
    {
        return m_pThread->m_pTest->GetRemoteIp();
    }

    // WriteHostIp
    bool BaseModule::WriteHostIp(char* pStr, uint32_t bufferLen) const NN_NOEXCEPT
    {
        return m_pThread->m_pTest->WriteHostIp(pStr, bufferLen);
    }

    // GetResult
    ModuleResult BaseModule::GetResult() const NN_NOEXCEPT
    { return m_result; }

    // Log
    void BaseModule::Log(const char* pFormat, ...) const NN_NOEXCEPT
    {
        if( m_pThread )
        {
            static const unsigned StrBufLen = 2048;
            char pBuffer[StrBufLen];

            va_list args;
            va_start(args, pFormat);
            vsnprintf(pBuffer, StrBufLen, pFormat, args);
            m_pThread->Log("[%s] %s", this->GetName(), pBuffer);
            va_end (args);
        }
    }

    // LogErrorManual
    void BaseModule::LogErrorManual(const char* pFileName, const char* pFunctionName, unsigned lineNumber, const char* pFormat, ...) const NN_NOEXCEPT
    {
        if( m_pThread )
        {
            static const unsigned StrBufLen = 2048;
            char pBuffer[StrBufLen];

            va_list args;
            va_start(args, pFormat);
            vsnprintf(pBuffer, StrBufLen, pFormat, args);
            m_pThread->LogErrorManual(pFileName, pFunctionName, lineNumber, "[%s] %s", this->GetName(), pBuffer);
            va_end(args);
        }
    }

    // GetThreadName
    const char* BaseModule::GetThreadName() const NN_NOEXCEPT
    {
        if( m_pThread )
        {
            return m_pThread->GetName();
        }
        else
        {
            return nullptr;
        }
    }

    // Default Constructor
    TestThread::TestThread(const BaseTest* pTest, const char* pName, unsigned timeoutSec) NN_NOEXCEPT
        : m_pUnalignedBuf(nullptr),
          m_pHead(nullptr),
          m_pTail(nullptr),
          m_pNext(nullptr),
          m_isSlaveThread(false),
          m_timeoutSec(timeoutSec),
          m_state(ThreadState_Null),
          m_pTest(pTest)
    {
        strncpy(m_pName, pName, StrBufLen);
    }

    TestThread::~TestThread() NN_NOEXCEPT
    {
        Destroy();
    }

    // ThreadFunc (Static)
    void TestThread::ThreadFunc(void* pParam) NN_NOEXCEPT
    {
        TestThread* pThis = (TestThread*)pParam;

        pThis->m_state = ThreadState_Running;
        BaseModule* pCurr = pThis->m_pHead;
        while( pCurr )
        {
            pCurr->Log("- Starting\n\n");
            bool bPass = pCurr->Run();
            if( bPass )
            {
                pCurr->Log("PASSED!\n\n");
                pCurr->m_result = ModuleResult_Pass;
            }
            else
            {
                pCurr->Log("FAILED!\n\n");
                pCurr->m_result = ModuleResult_Fail;
            }

            NetTest::YieldThread();
            pCurr = pCurr->m_pNext;
        }

        pThis->m_state = ThreadState_Finished;
    }

    // GetName
    const char* TestThread::GetName() const NN_NOEXCEPT
    {
        return m_pName;
    }

    // GetTestName
    const char* TestThread::GetTestName() const NN_NOEXCEPT
    {
        return m_pTest->GetName();
    }

    bool TestThread::IsSlave() const NN_NOEXCEPT
    {
        return m_isSlaveThread;
    }

    // AddModule
    void TestThread::AddModule(BaseModule& module) NN_NOEXCEPT
    {
        module.m_pThread = this;
        module.m_pNext = nullptr;

        // Empty List
        if( m_pHead == nullptr )
        {
            m_pHead = m_pTail = &module;
        }
        else
        {
            m_pTail->m_pNext = &module;
            m_pTail = &module;
        }
    }

    // Start
    void TestThread::Start() NN_NOEXCEPT
    { NetTest::StartThread(&m_thread); }

    // GetTimeout
    unsigned TestThread::GetTimeout() const NN_NOEXCEPT
    { return m_timeoutSec; }

    // Destroy
    void TestThread::Destroy() NN_NOEXCEPT
    {
        if( m_state != ThreadState_Null )
        {
            Log("Destroying thread...\n");
            NetTest::DestroyThread(&m_thread);
            m_state = ThreadState_Null;
            Log("...Done\n");
        }
    }

    // GetState
    ThreadState TestThread::GetState() const NN_NOEXCEPT
    { return m_state; }

    // Log
    void TestThread::Log(const char* pFormat, ...) const NN_NOEXCEPT
    {
        static const unsigned LogBufLen = 2048;
        char pBuffer[LogBufLen];

        va_list args;
        va_start(args, pFormat);
        vsnprintf(pBuffer, LogBufLen, pFormat, args);
        m_pTest->Log("[%s]%s ", m_pName, pBuffer);
        va_end(args);
    }

    // LogErrorManual
    void TestThread::LogErrorManual(const char* pFileName, const char* pFunctionName, unsigned lineNumber, const char* pFormat, ...) const NN_NOEXCEPT
    {
        static const unsigned LogBufLen = 2048;
        char pBuffer[LogBufLen];

        va_list args;
        va_start(args, pFormat);
        vsnprintf(pBuffer, LogBufLen, pFormat, args);
        m_pTest->LogErrorManual(pFileName, pFunctionName, lineNumber, "[%s]%s\n", m_pName, pBuffer);
        va_end(args);
    }

    // LogResults
    bool TestThread::LogResults() const NN_NOEXCEPT
    {
        bool isSuccess = true;

        if( GetState() == ThreadState_Finished )
        {
            NATF_LOG("  %s thread finished cleanly\n", GetName());
        }
        else
        {
            NATF_LOG("  %s thread did NOT finish!\n", GetName());
        }

        NATF_LOG("  Modules Results for %s: \n", GetName());

        int moduleId = 0;
        BaseModule* pModule = m_pHead;
        while( pModule )
        {
            if( pModule->GetResult() == ModuleResult_Pass )
            {
                NATF_LOG("   Pass - %s_%d\n", pModule->GetName(), moduleId);
            }
            else if( pModule->GetResult() == ModuleResult_Fail )
            {
                isSuccess = false;
                NATF_LOG("   Fail - %s_%d\n", pModule->GetName(), moduleId);
            }
            else
            {
                isSuccess = false;
                NATF_LOG("   Incomplete - %s_%d\n", pModule->GetName(), moduleId);
            }

            ++moduleId;
            pModule = pModule->m_pNext;
        }

        NN_LOG("\n");
        return isSuccess;
    }

    // Default Constructor
    BaseTest::BaseTest(const char* pTestName, bool doExchangeIpInfo, Utils::InitApiFlags initFlags, const nn::util::Uuid& netProfile) NN_NOEXCEPT :
        m_doExchangeInfo(doExchangeIpInfo),
        m_initFlags(initFlags),
        m_netProfile(netProfile),
        m_pHead(nullptr),
        m_pTail(nullptr)
    {
        m_pNameBuffer[0] = '\0';
        SetName(pTestName);
    }

    // Destructor
    BaseTest::~BaseTest() NN_NOEXCEPT
    {
        CleanupThreads();
    }

    // SetName
    void BaseTest::SetName(const char* pTestName) NN_NOEXCEPT
    {
        if( pTestName )
        {
            int shouldBeLen = NETTEST_SNPRINTF(m_pNameBuffer, NameBufferLen, "%s", pTestName);
            int actualLen = static_cast<int>(strlen(m_pNameBuffer));

            if( shouldBeLen != actualLen )
            {
                Log(" Warning! Test name truncated! Max test name length: %d\n\n", NameBufferLen - 1);
            }
        }
        else
        {
            Log(" Warning! nullptr passed to BaseTest::SetName\n\n");
        }
    }

    // GetName
    const char* BaseTest::GetName() const NN_NOEXCEPT
    {
         return m_pNameBuffer;
    }

    // GetRemoteIp
    const char* BaseTest::GetRemoteIp() const NN_NOEXCEPT
    {
        if( !m_doExchangeInfo )
        {
            LogError("Warning: GetRemoteIp() was called when no ip address exchange occurred\n\n");
            return nullptr;
        }

        return m_exchangeInfo.GetRemoteIp();
    }

    bool BaseTest::WriteHostIp(char* pStr, uint32_t bufferLen) const NN_NOEXCEPT
    {
        size_t hostNameLen = strlen(RemoteHostName);

        const char* pRemoteIp = GetRemoteIp();
        if( nullptr == pRemoteIp )
        {
            LogError("Warning: WriteHostIp() was called when no ip address exchange occurred.\n\n");
            return false;
        }

        char* pWhere = strstr(pStr, RemoteHostName);
        if( pWhere )
        {
            size_t bytesAfter = strlen(pWhere + hostNameLen);
            if( (pWhere - pStr) + bytesAfter + hostNameLen >= bufferLen )
            {
                LogError("Warning: WriteHostIp() was called with a buffer that is too small.\n\n");
                return false;
            }

            size_t ipLen = strlen(pRemoteIp);
            memmove(pWhere + ipLen, pWhere + hostNameLen, bytesAfter);
            memcpy(pWhere, pRemoteIp, ipLen);
            *(pWhere + ipLen + bytesAfter) = '\0';

            return true;
        }

        return false;
    }

    // AddSlaveModule
    bool BaseTest::AddSlaveModule(const char* pThreadName, BaseModule& module) NN_NOEXCEPT
    {
        // Allocate thread object
        Log("Allocating thread stack...\n");
        unsigned char* pBuffer = (unsigned char*)malloc(sizeof(TestThread) + TestThread::StackAlign);
        if( !pBuffer )
        {
            LogError("Failed to allocate thread memory.\n\n");
            return false;
        }

        // Byte align the stack.
        unsigned char* pAlignedBuf = (pBuffer + TestThread::StackAlign - (reinterpret_cast<uintptr_t>(pBuffer) % TestThread::StackAlign));
        TestThread* pNewTestThread = new (pAlignedBuf) TestThread(this, pThreadName, TestThread::SlaveTimeoutSec);
        pNewTestThread->m_pUnalignedBuf = pBuffer;

        Log("Creating thread...\n");
        bool isSuccess = NetTest::CreateThread(&pNewTestThread->m_thread, TestThread::ThreadFunc, pNewTestThread, pNewTestThread->m_pStack, TestThread::StackSize);
        if( !isSuccess )
        {
            LogError("Failed to create thread!\n\n");
            pNewTestThread->~TestThread();
            free(pNewTestThread->m_pUnalignedBuf);

            return false;
        }

        pNewTestThread->m_state = ThreadState_Created;

        if( !m_pHead )
        {
            pNewTestThread->m_pNext = nullptr;
            m_pHead = m_pTail = pNewTestThread;
        }
        else
        {
            m_pTail->m_pNext = pNewTestThread;
            m_pTail = pNewTestThread;
        }

        pNewTestThread->m_isSlaveThread = true;
        pNewTestThread->AddModule(module);
        return true;
    }

    // CreateTestThread
    TestThread* BaseTest::CreateTestThread(const char* pThreadName, unsigned timeoutSec) NN_NOEXCEPT
    {
        // Allocate thread object
        Log("Allocating thread stack...\n");
        unsigned char* pBuffer = (unsigned char*)malloc(sizeof(TestThread) + TestThread::StackAlign);
        if( !pBuffer )
        {
            LogError("Failed to allocate thread memory.\n\n");
            return nullptr;
        }

        // Byte align the stack.
        unsigned char* pAlignedBuf = (pBuffer + TestThread::StackAlign - (reinterpret_cast<uintptr_t>(pBuffer) % TestThread::StackAlign));
        TestThread* pNewTestThread = new (pAlignedBuf) TestThread(this, pThreadName, timeoutSec);
        pNewTestThread->m_pUnalignedBuf = pBuffer;

        Log("Creating thread...\n");
        bool isSuccess = NetTest::CreateThread(&pNewTestThread->m_thread, TestThread::ThreadFunc, pNewTestThread, pNewTestThread->m_pStack, TestThread::StackSize);
        if( !isSuccess )
        {
            LogError("Failed to create thread!\n\n");
            pNewTestThread->~TestThread();
            free(pNewTestThread->m_pUnalignedBuf);

            return nullptr;
        }

        pNewTestThread->m_state = ThreadState_Created;

        if( !m_pHead )
        {
            pNewTestThread->m_pNext = nullptr;
            m_pHead = m_pTail = pNewTestThread;
        }
        else
        {
            m_pTail->m_pNext = pNewTestThread;
            m_pTail = pNewTestThread;
        }

        return pNewTestThread;
    }

    // Log
    void BaseTest::Log(const char* pFormat, ...) const NN_NOEXCEPT
    {
        static const unsigned StrBufLen = 2048;
        char pBuffer[StrBufLen];

        va_list args;
        va_start (args, pFormat);
        vsnprintf (pBuffer, StrBufLen, pFormat, args);
        int testDuration = static_cast<int>((NetTest::GetTick() - m_startTime).ToTimeSpan().GetSeconds());
        NATF_LOG("[%d][%s]%s ", testDuration, GetName(), pBuffer);
        va_end (args);
    }

    // LogErrorManual
    void BaseTest::LogErrorManual(const char* pFileName, const char* pFunctionName, unsigned lineNumber, const char* pFormat, ...) const NN_NOEXCEPT
    {
        static const unsigned StrBufLen = 2048;
        char pBuffer[StrBufLen];

        va_list args;
        va_start(args, pFormat);
        vsnprintf(pBuffer, StrBufLen, pFormat, args);
        int testDuration = static_cast<int>((NetTest::GetTick() - m_startTime).ToTimeSpan().GetSeconds());
        NATF_LOG("File: %s, Function: %s, Line: %d, Msg:\n[%d][%s]%s\n", pFileName, pFunctionName, lineNumber, testDuration, GetName(), pBuffer);
        va_end(args);
    }

    // CleanupThreads
    void BaseTest::CleanupThreads() NN_NOEXCEPT
    {
        TestThread* pThread = m_pHead;
        while( pThread )
        {
            TestThread* pNext = pThread->m_pNext;
            pThread->~TestThread();
            free(pThread->m_pUnalignedBuf);

            pThread = pNext;
        }

        m_pHead = m_pTail = nullptr;
    }

    bool BaseTest::PrivateInit(Utils::InitApi* pInitApi) NN_NOEXCEPT
    {
        nn::Result result;

        if( pInitApi != nullptr && m_initFlags != Utils::InitApiFlags::InitApiFlags_None )
        {
            if( !pInitApi->Init(m_netProfile) )
            {
                LogError(" ERROR: Failed to initialize an Api!\n");
                return false;
            }
        }

        if( m_doExchangeInfo )
        {
            // Fail if we have not initialized the network
            if( (m_initFlags & Utils::InitApiFlags::InitApiFlags_Network) == Utils::InitApiFlags::InitApiFlags_None )
            {
                LogError(" ERROR: InitApiFlags_Network must be set when exchanging ip info.\n\n");
                return false;
            }

            if( !m_exchangeInfo.Exchange() )
            {
                LogError(" ERROR: Failed to exchange ip info.\n\n");
                return false;
            }
        }

#ifndef NN_BUILD_CONFIG_OS_WIN32
        do
        {
            nn::time::PosixTime posixTime;
            nn::time::CalendarTime calendarTime;

            result = nn::time::Initialize();
            if(result.IsFailure())
            {
                Log(" WARNING: time::Initialize() failed! Desc: %d\n\n", result.GetDescription());
                break;
            }

            result = nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
            if(result.IsFailure())
            {
                Log(" WARNING: StandardUserSystemClock::GetCurrentTime() failed! Desc: %d\n\n", result.GetDescription());
                break;
            }

            result = nn::time::ToCalendarTime(&calendarTime, nullptr, posixTime);
            if(result.IsFailure())
            {
                Log(" WARNING: time::ToCalendarTime() failed! Desc: %d\n\n", result.GetDescription());
                break;
            }

            Log(
                "Date/Time:  %04d-%02d-%02d %02d:%02d:%02d\n\n",
                calendarTime.year, calendarTime.month, calendarTime.day,
                calendarTime.hour, calendarTime.minute, calendarTime.second);
        } while(NN_STATIC_CONDITION(false));
#endif

        Log(" Configuring test...\n");
        if( !Config() )
        {
            LogError(" ERROR: Failed to configure test!\n");
            return false;
        }

        Log(" Initializing test...\n");
        if( !Init() )
        {
            LogError(" ERROR: Failed to initialize test!\n\n");
            return false;
        }

        if ((m_initFlags & Utils::InitApiFlags::InitApiFlags_Nifm) != Utils::InitApiFlags::InitApiFlags_None)
        {
            nn::socket::InAddr localIP;
            result = nn::nifm::GetCurrentPrimaryIpAddress(&localIP);

            if (result.IsSuccess())
            {
                Log("Local IP address: %s\n", nn::socket::InetNtoa(localIP));
            }
            else
            {
                // Minor error does not affect test functionality so returns 'true' to let the test to continue
                Log("WARNING: Failed to obtain Local IP address! Desc: %d\n\n", result.GetDescription());
            }
        }

        return true;
    }

    // Run
    bool BaseTest::Run(bool initApis) NN_NOEXCEPT
    {
        bool isSuccess = true;
        Utils::InitApi* pInitApi = nullptr;
        Utils::InitApi initApi(m_initFlags);

        m_startTime = NetTest::GetTick();

        if(initApis)
        {
            pInitApi = &initApi;
        }

        if( !PrivateInit(pInitApi) )
        {
            return false;
        }

        // Start threads!
        TestThread* pThread = m_pHead;
        while( pThread )
        {
            pThread->Log(" Starting thread...\n");
            pThread->Start();
            pThread = pThread->m_pNext;
        }

        // Let the threads start up
        NetTest::SleepMs(1000);

        // Loop through the threads and check for timeouts
        NetTest::Tick start = NetTest::GetTick();
        bool bPending;
        do
        {
            bPending = false;
            NetTest::Tick now = NetTest::GetTick();
            NetTest::Time duration = NetTest::TickToTime(now - start);
            pThread = m_pHead;
            while( pThread )
            {
                // Is there a non-slave thread stil running? If so, check for timeout.
                if( !pThread->IsSlave() && pThread->GetState() == ThreadState_Running )
                {
                    if( duration.GetSeconds() > pThread->GetTimeout() )
                    {
                        pThread->Log(" WARNING: Thread has timed out!\n\n");
                        pThread->m_state = ThreadState_Terminated;
                    }
                    else
                    {
                        bPending = true;
                    }
                }

                pThread = pThread->m_pNext;
            }

            NetTest::SleepMs(1000);
        } while( bPending );

        // Signal all slave threads to stop
        pThread = m_pHead;
        while( pThread )
        {
            if( pThread->IsSlave() )
            {
                pThread->m_pHead->StopSlave();
            }

            pThread = pThread->m_pNext;
        }

        // Loop through all slave threads and wait for them to return
        start = NetTest::GetTick();
        do
        {
            bPending = false;
            NetTest::Tick now = NetTest::GetTick();
            NetTest::Time duration = NetTest::TickToTime(now - start);
            pThread = m_pHead;
            while( pThread )
            {
                // Is there a non-slave thread stil running? If so, check for timeout.
                if( pThread->IsSlave() && pThread->GetState() == ThreadState_Running )
                {
                    if( duration.GetSeconds() > pThread->GetTimeout() )
                    {
                        pThread->Log(" WARNING: Thread has timed out!\n\n");
                        pThread->m_state = ThreadState_Terminated;
                    }
                    else
                    {
                        bPending = true;
                    }
                }

                pThread = pThread->m_pNext;
            }

            NetTest::SleepMs(1000);
        } while( bPending );

        NATF_LOG(" Test: %s\n", GetName());
        pThread = m_pHead;
        while( pThread )
        {
            isSuccess &= pThread->LogResults();
            pThread = pThread->m_pNext;
        }

        Log("Cleaning up test...\n");
        if( !Cleanup() )
        {
            Log(" ERROR: Failed to cleanup test!\n\n");
            isSuccess = false;
        }

        Log("Cleaning up threads.\n");
        CleanupThreads();

#ifndef NN_BUILD_CONFIG_OS_WIN32
        Log("Finalizing Time Api...\n");
        nn::time::Finalize();
#endif

        Log("Finalizing Apis...\n");
        initApi.Finalize();

        Log("Test exiting.\n");
        return isSuccess;
    } // NOLINT(impl/function_size)
} // namespace natf

