﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/TcpUdpLoopbackTest.h"

namespace NATF {
namespace Tests {

    // Constructor
    TcpUdpLoopback::TcpUdpLoopback(const char* pTestName, const nn::util::Uuid& netProfile, uint32_t timeoutMs, uint32_t portNumber, const char* protocolTypeName, uint32_t durationSec, uint32_t connectionCount, const char* clientXferPattern, const char* serverXferPattern) NN_NOEXCEPT :
        BaseTest(pTestName, false, Utils::InitApiFlags::InitApiFlags_Socket, netProfile),
        m_timeoutMs(timeoutMs),
        m_portNumber(portNumber),
        m_durationSec(durationSec),
        m_connectionCount(connectionCount),
        m_clientXferPattern(nullptr),
        m_serverXferPattern(nullptr)
    {
        NN_STATIC_ASSERT(static_cast<void *>(0) == nullptr);
        memset(m_pServers, 0, sizeof(m_pServers));
        memset(m_pClients, 0, sizeof(m_pClients));

        NETTEST_COPY_BUFFER_STRING_ARRAY(m_protocolTypeName, protocolTypeName);
        if( nullptr != clientXferPattern )
        {
            m_clientXferPattern = (char*)malloc(Modules::SoTestDataXfer::MaxBufSize);
            NETTEST_COPY_BUFFER_BYTES_POINTER(m_clientXferPattern, clientXferPattern, Modules::SoTestDataXfer::MaxBufSize);
        }
        if( nullptr != serverXferPattern )
        {
            m_serverXferPattern = (char*)malloc(Modules::SoTestDataXfer::MaxBufSize);
            NETTEST_COPY_BUFFER_BYTES_POINTER(m_serverXferPattern, serverXferPattern, Modules::SoTestDataXfer::MaxBufSize);
        }
    }

    // Destructor
    TcpUdpLoopback::~TcpUdpLoopback() NN_NOEXCEPT
    {
        for( size_t i = 0; i < MaxConnectionCount; i++)
        {
            if( nullptr != m_pServers[i] )
            {
                delete m_pServers[i];
                m_pServers[i] = nullptr;
            }
            if( nullptr != m_pClients[i] )
            {
                delete m_pClients[i];
                m_pClients[i] = nullptr;
            }
        }
        if( nullptr != m_clientXferPattern )
        {
            free(m_clientXferPattern);
            m_clientXferPattern = nullptr;
        }
        if( nullptr != m_serverXferPattern )
        {
            free(m_serverXferPattern);
            m_serverXferPattern = nullptr;
        }
    }

    // Cleanup
    bool TcpUdpLoopback::Cleanup() NN_NOEXCEPT
    {
        for( size_t i = 0; i < MaxConnectionCount; i++)
        {
            if( nullptr != m_pServers[i] )
            {
                delete m_pServers[i];
                m_pServers[i] = nullptr;
            }
            if( nullptr != m_pClients[i] )
            {
                delete m_pClients[i];
                m_pClients[i] = nullptr;
            }
        }
        if( nullptr != m_clientXferPattern )
        {
            free(m_clientXferPattern);
            m_clientXferPattern = nullptr;
        }
        if( nullptr != m_serverXferPattern )
        {
            free(m_serverXferPattern);
            m_serverXferPattern = nullptr;
        }
        return true;
    }

    // Config
    bool TcpUdpLoopback::Config() NN_NOEXCEPT
    {
        if( MaxConnectionCount < m_connectionCount )
        {
            LogError(" * Invalid command line arguments: Connections = %d (max: %d)\n\n",
                m_connectionCount, MaxConnectionCount);
            return false;
        }

        for( size_t i = 0; i < m_connectionCount; ++i )
        {
            char serverThreadName[] = "Server_#";
            size_t nNameLength = strlen(serverThreadName);
            serverThreadName[nNameLength - 1] = '0' + static_cast<char>(i);

            TestThread* pServerThread = CreateTestThread(serverThreadName, m_timeoutMs);
            if( nullptr == pServerThread )
            {
                return false;
            }

            m_pServers[i] = Modules::SoTestDataXfer::CreateTestNode(pServerThread,
                "Server", "127.0.0.1", static_cast<short>(m_portNumber + i), m_protocolTypeName,
                m_durationSec, m_serverXferPattern, true);
            if( nullptr == m_pServers[i] )
            {
                LogError(" * Failed to allocate node module!\n\n");
                return false;
            }

            pServerThread->AddModule(*m_pServers[i]);

            char clientThreadName[] = "Client_#";
            nNameLength = strlen(clientThreadName);
            clientThreadName[nNameLength - 1] = '0' + static_cast<char>(i);

            TestThread* pClientThread = CreateTestThread(clientThreadName, m_timeoutMs);
            if( nullptr == pClientThread )
            {
                return false;
            }

            m_pClients[i] = Modules::SoTestDataXfer::CreateTestNode(pClientThread,
                "Client", "127.0.0.1", static_cast<short>(m_portNumber + i), m_protocolTypeName,
                m_durationSec, m_clientXferPattern, true);
            if( nullptr == m_pClients[i] )
            {
                LogError(" * Failed to allocate node module!\n\n");
                return false;
            }

            pClientThread->AddModule(*m_pClients[i]);
        }

        return true;
    }

} // namespace Tests
} // namespace NATF
