﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>


// Anonymous Name Space
namespace {

/////////////////////////
//  C O N S T A N T S
/////////////////////////

const size_t        kMaxThreads = 32;
const size_t        kThreadStackSize = 65536;

NN_OS_ALIGNAS_THREAD_STACK uint8_t threadStack[kMaxThreads + 1][kThreadStackSize];


/////////////////////////
// ThreadController
/////////////////////////

class ThreadedController
{
     private:

        nn::os::ThreadType              threads[kMaxThreads];

        int                             threadCount;

    public:

        ThreadedController()
        {
           threadCount = 0;
        }

        ~ThreadedController()
        {
           threadCount = 0;
        }

        int Initialize( int     numThreads)
        {
            if ( numThreads > kMaxThreads ) {
                NN_LOG( "This object only supports %d threads - Please set kMaxtshreads appropriately\n",
                            kMaxThreads );
                return( -1 );
            }

            // Start at first slot
            threadCount = 0;

            return( 0 );
        }


        int CreateThread( void          (*threadFunct) (void *),
                          void *        threadFunctArg,
                          int           threadPriority )
       {
            nn::Result  rval;
            int idx = 0;
            int threadState = 0;

            // Generate a unique Thread ID
            if ( ( threadCount + 1 ) <= kMaxThreads )
            {
                // Allocate a (new) Thread
                threadCount++;

                // Create a Thread for use
                rval = nn::os::CreateThread( &threads[threadCount],
                                             threadFunct,
                                             threadFunctArg,
                                             threadStack[threadCount],
                                             sizeof( threadStack[threadCount] ),
                                             threadPriority );
                if ( rval.IsFailure() )
                {
                    NN_LOG( "Failed to CreateThread - a normal thread\n" );
                    return( -1 );
                }

                // Start the Thread
                nn::os::StartThread( &threads[threadCount] );

               // Return Thread Id to caller
               return( threadCount );
           }

           for( idx = 1; idx <= kMaxThreads; idx++ )
           {
                threadState = threads[idx]._state;

                if ( threadState == nn::os::ThreadType::State_Exited )
                {
                    nn::os::WaitThread(  &threads[ idx ] );
                    nn::os::DestroyThread( &threads[ idx ] );

                    // Create a Thread for use
                    rval = nn::os::CreateThread( &threads[ idx ],
                                                 threadFunct,
                                                 threadFunctArg,
                                                 threadStack[ idx ],
                                                 sizeof( threadStack[ idx ] ),
                                                 threadPriority );

                    if ( rval.IsFailure() )
                    {
                        NN_LOG( "Failed to CreateThread - a reused thread\n" );
                        return( -1 );
                    }

                    // Start the Thread
                    nn::os::StartThread( &threads[ idx ] );

                    // Return Thread Id to caller
                    return( idx );
                }
            }

            NN_LOG( "All avialable threads are execusting! Max Threads = %d\n", kMaxThreads );
            return( -1 );
        }

        int DestroyThread( int       threadId )
        {
             nn::os::DestroyThread( &threads[threadId] );

             return( 0 );
        }

        int WaitThread( int       threadId )
        {
             nn::os::WaitThread( &threads[threadId] );

             return( 0 );
        }

        int WaitAllThreads()
        {
            int   idx;
            int   threadState = -1;

            for( idx = 1; idx <= threadCount; idx++ )
            {
                threadState = threads[idx]._state;

                if( ( threadState == nn::os::ThreadType::State_Initialized ) ||
                    ( threadState == nn::os::ThreadType::State_Started ) ||
                    ( threadState == nn::os::ThreadType::State_Exited ) )
                {
                    nn::os::WaitThread( &threads[idx] );
                }
            }

            return( 0 );
        }

        int DestroyAllThreads()
        {
            int   idx;
            int   threadState = -1;

            for( idx = 1; idx <= threadCount; idx++ )
            {
                threadState = threads[idx]._state;

                if( ( threadState == nn::os::ThreadType::State_Initialized ) ||
                    ( threadState == nn::os::ThreadType::State_Started ) ||
                    ( threadState == nn::os::ThreadType::State_Exited ) )
                {
                    nn::os::DestroyThread( &threads[idx] );
                }
            }

            threadCount = 0;

            return( 0 );
        }
};

}

