﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SensorData.h"
#include <random>

namespace nerd {


PreparedSensorData<float> LoadDataNerd(char const* path, int downsample)
{
    NERD_ASSERT(path);
    HostSensorCapture capture;
    FILE* file = fopen(path, "r");
    NERD_ASSERT(file);
    if (!file) return nerd::PrepareRawData<float>(capture, downsample);
    char buf[1024];
    while (fgets(buf, 1024, file))
    {
#define PARSEINT  atoll(strtok(NULL, ";"))
#define PARSESHORT  s16(atoi(strtok(NULL, ";")))
#define PARSEFLOAT  float(atof(strtok(NULL, ";")))
        CSVRawSensorSample sample;
        char* tok = strtok(buf, ";\n");
        if (!tok) continue; // skip first column and check for empty line
        sample.timestamp = PARSEINT;
        sample.Rotation.x() = PARSESHORT;
        sample.Rotation.y() = PARSESHORT;
        sample.Rotation.z() = PARSESHORT;
        sample.Acceleration.x() = PARSESHORT;
        sample.Acceleration.y() = PARSESHORT;
        sample.Acceleration.z() = PARSESHORT;
        capture.Data.EmplaceBack(sample);
    }
    fclose(file);
    return nerd::PrepareRawData<float>(capture, downsample);
}

PreparedSensorData<float> LoadDataProcessed(char const* path, int downsample)
{
    PreparedSensorData<float> result;

    NERD_ASSERT(path);
    HostSensorCapture capture;
    FILE* file = fopen(path, "r");
    NERD_ASSERT(file);
    if (!file) return result;
    char buf[1024];
    int64_t t = 0;
    std::mt19937 gen((int(3456)));

    while (fgets(buf, 1024, file))
    {
        char* tok = strtok(buf, ";\n");
        if (!tok) continue; // skip first column and check for empty line
        int64_t ts = atoll(tok);
        PARSEINT;
        PARSEFLOAT;
        float3 acc = { PARSEFLOAT, PARSEFLOAT, PARSEFLOAT };
        float3 omega = { PARSEFLOAT, PARSEFLOAT, PARSEFLOAT };
        if (t == 0) result.dts.PushBack(0.001f);
        else result.dts.PushBack((ts - t) * 1e-6f);
        result.localAccels.PushBack(9.81f * acc);
        std::normal_distribution<float> noise;
        result.omegas.PushBack(omega * 2.0f * constants::PI + 0.005f * float3(noise(gen), noise(gen), noise(gen)));
        t = ts;
    }
    fclose(file);
    return result;
}

PreparedSensorData<float> LoadData(char const* path, int downsample)
{
    HostSensorCapture capture;
    FILE* file = fopen(path, "r");
    NERD_ASSERT(file);
    if (!file) return nerd::PrepareRawData<float>(capture, downsample);
    char buf[1024];
    if (!fgets(buf, 1024, file)) return nerd::PrepareRawData<float>(capture, downsample);
    fclose(file);
    int countsemicolon = 0;
    for (char* p = buf;*p;++p)
        countsemicolon += *p == ';';
    if (countsemicolon == 18) return LoadDataProcessed(path, downsample);
    else
        if (countsemicolon >= 8) return LoadDataNerd(path, downsample);
    return nerd::PrepareRawData<float>(capture, downsample);
}

template<typename scal>
PreparedSensorData<scal> PrepareRawData(HostSensorCapture const& capture, u32 const downSample)
{
    //******************************************************************************
    const scal cAccelerometerFSRTable[] = {
        2.0f,
        4.0f,
        8.0f,
        16.0f
    };
    const scal cGyroscopeFSRTable[] = {
        125.0f,
        245.0f,
        500.0f,
        1000.0f,
        2000.0f
    };
    const scal cAccelerometerSensitivityTable[] = {
        0.061f,// ±2 G
        0.122f,// ±4 G
        0.244f,// ±8 G
        0.488f// ±16 G
    };
    const scal cGyroscopeSensitivityTable[] = {
        4.375f,// ±125 deg/s
        8.75f,// ±245 deg/s
        17.50f,// ±500 deg/s
        35.0f,// ±1000 deg/s
        70.0f// ±2000 deg/s
    };
    typedef NumericalVector<scal, 3> v3;

    auto const dataFullSampling = capture.Data.ConstView();
    //auto const dataFullSampling = capture.Data.ConstSubView(550*151, capture.Data.Size()-151*550);
    NERD_ASSERT(downSample & 1); //must be odd
    s32 const downSampleHalf = downSample / 2;
    ContiguousStorageContainer<TemplateSensorSample<scal>> data;
    for (s32 midSample = downSampleHalf; midSample < s32(dataFullSampling.Size()) - downSampleHalf; midSample += downSample)
    {
        size_t const start = midSample - downSampleHalf;
        size_t const end = midSample + downSampleHalf + 1;
        v3 accAvg(0);
        v3 omegaAvg(0);
        forrangest(i, start, end)
        {
            auto const& di = dataFullSampling[i];
            v3 const diAShuff = (v3(-1, 1, -1) * di.Acceleration).yxz();
            v3 const diRShuf(-scal(di.Rotation._[1]), scal(di.Rotation._[0]), scal(di.Rotation._[2]));
            accAvg += diAShuff;
            omegaAvg += diRShuf;
        }
        scal const normalizer = 1.f / (end - start);
        accAvg *= normalizer;
        omegaAvg *= normalizer;
        auto& newItem = data.EmplaceBack();
        newItem.Acceleration = accAvg;
        newItem.Rotation = omegaAvg;
        newItem.timestamp = dataFullSampling[midSample].timestamp;
    }

    //auto const data = capture.Data.ConstView();

    //Acquisition en 2G (accelero FSR)
    const int cAccelerometerCalibrationFsr = 2; //8G
    const scal cAccelerometerCalibrationBase = 1.0f;
    const v3 cAccelerometerDefaultCalibrationOrigin(0.0f);
    const v3 cAccelerometerDefaultCalibrationSensitivity = v3(cAccelerometerCalibrationBase * 1000.0f / cAccelerometerSensitivityTable[cAccelerometerCalibrationFsr]);

    // Gyroscope FSR = nnd::lsm6ds3::GyroscopeFsr_2000dps; == 4
    const int cGyroscopeFsr = 3; // ±1000 deg/s
    const scal cGyroscopeCalibrationBase = 936.0f;
    const v3 cGyroscopeDefaultCalibrationOrigin(0.0f);
    const v3 cGyroscopeDefaultCalibrationSensitivity = v3(cGyroscopeCalibrationBase * 1000.0f / cGyroscopeSensitivityTable[cGyroscopeFsr]);

    const scal fsrScaleAccel = cAccelerometerFSRTable[cAccelerometerCalibrationFsr] / cAccelerometerFSRTable[cAccelerometerCalibrationFsr];
    v3 const mAccelerometerScale = cAccelerometerCalibrationBase / cAccelerometerDefaultCalibrationSensitivity * fsrScaleAccel;

    const scal fsrScaleGyro = cGyroscopeFSRTable[cGyroscopeFsr] / cGyroscopeFSRTable[cGyroscopeFsr];
    const scal convertScale = constants::PI / scal(180);
    v3 const mGyroscopeScale = cGyroscopeCalibrationBase / cGyroscopeDefaultCalibrationSensitivity * convertScale * fsrScaleGyro;

    ContiguousStorageContainer<v3> omegas;
    ContiguousStorageContainer<scal> dts;
    ContiguousStorageContainer<v3> localAccels;
    omegas.Reserve(data.Size());
    dts.Reserve(data.Size());
    localAccels.Reserve(data.Size());

    //scal const timestampToSec = 1.f / 19200000.f;
    scal const timestampToSec = 1e-6f;
    forrange0st(i, data.Size())
    {
        size_t const im = i > 0 ? i - 1 : i;
        size_t const ip = i + 1 < data.Size() ? i + 1 : i;
        auto const& dm = data[im];
        auto const& di = data[i];
        auto const& dp = data[ip];
        scal const dtmp = (dp.timestamp - dm.timestamp) * timestampToSec;
        scal const dt = dtmp / (ip - im);
        v3 const v = di.Rotation * mGyroscopeScale;
        omegas.PushBackAssumeEnoughRoom(v);
        dts.PushBackAssumeEnoughRoom(dt);
        localAccels.PushBackAssumeEnoughRoom(di.Acceleration * mAccelerometerScale * 9.81f);
    }
    PreparedSensorData<scal> result;
    result.dts = dts;
    result.omegas = omegas;
    result.localAccels = localAccels;
    return result;
}

template PreparedSensorData<float> PrepareRawData<float>(HostSensorCapture const& capture, u32 const downSample);
template PreparedSensorData<double> PrepareRawData<double>(HostSensorCapture const& capture, u32 const downSample);


}
