﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

#include <nn/ndd.h>

#define TEST_LOG(format, ...) NN_LOG("[testNdd_Auto]" format, ##__VA_ARGS__)

//[todo]構造化

namespace {
nn::os::SystemEventType g_SendDataUpdateEvent;
nn::os::SystemEventType g_ReceiveDataEvent;
nn::os::SystemEventType g_DeviceScanEvent;
static const int RddCountMax = 100;
nn::ndd::ReceiveDataDescription g_Rdd[RddCountMax];

static const uint32_t RingBufferCountMax = 50;//内部実装に依存する値
}

namespace testUtil {

//スレッド
const size_t ThreadStackSize = 4096;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ThreadStackSize];
nn::os::ThreadType  g_DummyDeviceThread;

enum NetworkType
{
    NetworkType_Disabled,  // AutoCommunication無効。通信は行われない
    NetworkType_Enabled,   // AutoCommunication有効。通信は、他モジュールとの奪い合いが発生する
    //NetworkType_Dedicated, // AutoCommunication有効、かつアプリレベルのNetworkRequestを発行。通信はnddが占有する
};

void Init()
{
    TEST_LOG("Initialize Test\n");
    nn::ndd::Initialize();
    nn::ndd::AcquireSendDataUpdateEvent(&g_SendDataUpdateEvent);
    nn::ndd::AcquireReceiveDataEvent(&g_ReceiveDataEvent);
    nn::ndd::AcquireDeviceScanEvent(&g_DeviceScanEvent);
}

//スタック消費注意
class DataGenerator
{
public:
    static nn::ndd::SendDataDescription GenerateSendData(int mode, uint8_t data);
    static nn::ndd::ReceiveDataDescription GenerateReceiveData(int mode, uint8_t data);
    static bool IsSameSendData(nn::ndd::SendDataDescription& sdd, int mode, uint8_t data);
    static bool IsSameReceiveData(nn::ndd::ReceiveDataDescription& rdd, int mode, uint8_t data);
    static bool IsSameReceiveDataFromSendData(nn::ndd::ReceiveDataDescription& rdd, int originalMode, uint8_t originalData);
};

nn::ndd::SendDataDescription DataGenerator::GenerateSendData(int mode, uint8_t data)
{
    if(mode == 0)
    {
        //データ無し, DataIdは0x00埋め
        nn::ndd::SendDataDescription sdd;
        sdd.dataSize = 0;
        memset(&sdd.dataId.raw[0], 0x00, nn::ndd::DataIdSize);
        return sdd;
    }
    else if(mode == 1)
    {
        //データ最大長、DataIdは0xFF埋め
        nn::ndd::SendDataDescription sdd;
        memset(&sdd.data[0], data, nn::ndd::SendDataSizeMax);
        sdd.dataSize = nn::ndd::SendDataSizeMax;
        memset(&sdd.dataId.raw[0], 0xAA, nn::ndd::DataIdSize);
        return sdd;
    }
    else
    {
        NN_ABORT("faulty design\n");
    }
}

nn::ndd::ReceiveDataDescription DataGenerator::GenerateReceiveData(int mode, uint8_t data)
{
    if(mode == 0)
    {
        //データ長1、受信時刻0
        nn::ndd::ReceiveDataDescription rdd;
        memset(&rdd.data[0], data, 1);
        rdd.dataSize = 1;
        rdd.receiveTimePoint.value = 0;
        return rdd;
    }
    else if(mode == 1)
    {
        //データ最大長、受信時刻1
        nn::ndd::ReceiveDataDescription rdd;
        memset(&rdd.data[0], data, nn::ndd::ReceiveDataSizeMax);
        rdd.dataSize = nn::ndd::ReceiveDataSizeMax;
        rdd.receiveTimePoint.value = 1;
        return rdd;
    }
    else
    {
        NN_ABORT("faulty design\n");
    }
}

bool DataGenerator::IsSameSendData(nn::ndd::SendDataDescription& sdd, int mode, uint8_t data)
{
    auto originalData = GenerateSendData(mode, data);
    if(memcmp(&sdd.dataId.raw[0], &originalData.dataId.raw[0], nn::ndd::DataIdSize) != 0)
    {
        return false;
    }
    if(sdd.dataSize != originalData.dataSize)
    {
        return false;
    }
    if(memcmp(&sdd.data[0], &originalData.data[0], sdd.dataSize) != 0)
    {
        return false;
    }
    return true;
}

bool DataGenerator::IsSameReceiveData(nn::ndd::ReceiveDataDescription& rdd, int mode, uint8_t data)
{
    auto originalData = GenerateReceiveData(mode, data);
    if(rdd.receiveTimePoint.value != originalData.receiveTimePoint.value)
    {
        return false;
    }
    if(rdd.dataSize != originalData.dataSize)
    {
        return false;
    }
    if(memcmp(&rdd.data[0], &originalData.data[0], rdd.dataSize) != 0)
    {
        return false;
    }
    return true;
}

void Reset(NetworkType networkType)
{
    if(networkType == NetworkType_Disabled)
    {
        nn::ndd::DisableAutoCommunication();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));//サーバ側の動作完了まで少し待つ
        NN_ABORT_UNLESS(nn::ndd::IsAutoCommunicationEnabled() == false);
        //NN_ABORT_UNLESS(nn::ndd::IsNetworkActive() == false);//現状、AutoCommunicationが無効でも、NetworkRequestはする仕様のため、チェックしない
        //[todo]
    }
    else if(networkType == NetworkType_Enabled)
    {
        nn::ndd::EnableAutoCommunication();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));//サーバ側の動作完了まで少し待つ
        NN_ABORT_UNLESS(nn::ndd::IsAutoCommunicationEnabled() == true);
        //NN_ABORT_UNLESS(nn::ndd::IsNetworkActive() == true);//他モジュールの状態に依存するため、評価しない
    }
    else
    {
        NN_ABORT("faulty design\n");
    }
    nn::ndd::ClearSendData();
    nn::ndd::ClearReceiveData();
    nn::ndd::CancelDeviceScan();

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));//サーバ側の動作完了まで少し待つ
    nn::os::ClearSystemEvent(&g_SendDataUpdateEvent);
    nn::os::ClearSystemEvent(&g_ReceiveDataEvent);
    nn::os::ClearSystemEvent(&g_DeviceScanEvent);

    nn::ndd::SendDataDescription sddResult;
    NN_ABORT_UNLESS(nn::ndd::GetSendData(&sddResult, 0, 1) == 0);
    NN_ABORT_UNLESS(nn::ndd::GetCurrentReceiveDataCounter() == 0);
}

void DummyDeviceThread(void* arg)
{
    const int SendIntervalMs = 100;
    const int SendCountMax  = 100;
    for(int i=0;i<SendCountMax;++i)
    {
        nn::ndd::AddReceiveData(DataGenerator::GenerateReceiveData(1, i));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SendIntervalMs));
    }
}

void CreateDummyDeviceThread()
{
    auto result = nn::os::CreateThread( &g_DummyDeviceThread, testUtil::DummyDeviceThread, nullptr, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread(&g_DummyDeviceThread);
}

void WaitDummyDeviceThreadFinished()
{
    WaitThread(&g_DummyDeviceThread);
    DestroyThread(&g_DummyDeviceThread);
}

void ValidateCounter(uint32_t receivedCount)
{
    uint32_t counter;
    counter = nn::ndd::GetOldestReceiveDataCounter();
    if(receivedCount <= RingBufferCountMax)
    {
        //回りこんでいない限りは常に0
        NN_ABORT_UNLESS(counter == 0);
    }
    else
    {
        NN_ABORT_UNLESS(counter == receivedCount - RingBufferCountMax);
    }

    counter = nn::ndd::GetCurrentReceiveDataCounter();
    if(receivedCount == 0)
    {
        NN_ABORT_UNLESS(counter ==  0);
    }
    else if(receivedCount == 1)
    {
        //index0に一件保存されるため、Currentは0
        NN_ABORT_UNLESS(counter ==  0);
    }
    else
    {
        NN_ABORT_UNLESS(counter ==  receivedCount - 1);
    }

    counter = nn::ndd::GetNextReceiveDataCounter();
    NN_ABORT_UNLESS(counter == receivedCount);

    //0件指定時はNextと同等
    counter = nn::ndd::GetRecentReceiveDataCounter(0, 0);
    NN_ABORT_UNLESS(counter == nn::ndd::GetNextReceiveDataCounter());

    //全てを対象として、1件を取得する場合はCurrentと同等
    counter = nn::ndd::GetRecentReceiveDataCounter(0, 1);
    NN_ABORT_UNLESS(counter == nn::ndd::GetCurrentReceiveDataCounter());

    //全てを対象として、最大件取得する場合はOldestと同等
    counter = nn::ndd::GetRecentReceiveDataCounter(0, RingBufferCountMax + 1);//[todo] +1必要？
    NN_ABORT_UNLESS(counter == nn::ndd::GetOldestReceiveDataCounter());

    //Current以降を対象として、1件を取得する場合はCurrentと同等
    counter = nn::ndd::GetRecentReceiveDataCounter(nn::ndd::GetCurrentReceiveDataCounter(), 1);
    NN_ABORT_UNLESS(counter == nn::ndd::GetCurrentReceiveDataCounter());

    //Current以降を対象として、10件を取得する場合はCurrentと同等
    counter = nn::ndd::GetRecentReceiveDataCounter(nn::ndd::GetCurrentReceiveDataCounter(), 10);
    NN_ABORT_UNLESS(counter == nn::ndd::GetCurrentReceiveDataCounter());

    //Next以降指定時はNextと同等
    counter = nn::ndd::GetRecentReceiveDataCounter(nn::ndd::GetNextReceiveDataCounter(), 1);
    NN_ABORT_UNLESS(counter == nn::ndd::GetNextReceiveDataCounter());

    auto count = nn::ndd::GetAvailableReceiveDataCount(0);
    if(receivedCount < RingBufferCountMax)
    {
        NN_ABORT_UNLESS(count == receivedCount);
    }
    else
    {
        NN_ABORT_UNLESS(count == RingBufferCountMax);
    }
}

}

namespace testSuite {

void Padding()
{
    TEST_LOG("%s start\n", __FUNCTION__);
    //バイトオーダー
    {
        int test = 1;
        for(int i=0;i<4;++i)
        {
            TEST_LOG("order[%d] = %d\n", i, *(reinterpret_cast<uint8_t*>(&test) + i));
            NN_UNUSED(test);
        }
        NN_ABORT_UNLESS(*(reinterpret_cast<uint8_t*>(&test)) == 1);//リトルエンディアン
    }

    //構造体配列のサイズ
    {
        nn::ndd::ReceiveDataDescription receiveData[2];
        TEST_LOG("receiveData[2] = %d\n", sizeof(receiveData));
        NN_ABORT_UNLESS(sizeof(receiveData) == sizeof(nn::ndd::ReceiveDataDescription) * 2);
        NN_UNUSED(receiveData);
    }

    //アラインメント
    {
        TEST_LOG("alignof SendDataDescription = %d\n", alignof(nn::ndd::SendDataDescription));
        TEST_LOG("alignof ReceiveDataDescription = %d\n", alignof(nn::ndd::ReceiveDataDescription));
    }
    TEST_LOG("%s finish\n\n", __FUNCTION__);
}

void Init()
{
    TEST_LOG("%s start\n", __FUNCTION__);
    //多重初期化
    for(int i=0;i<10;++i)
    {
        nn::ndd::Initialize();
    }

    //多重登録
    {
        static int EventCount = 5;
        nn::os::SystemEventType sduEvent[EventCount];
        nn::os::SystemEventType rdEvent[EventCount];
        for(int i=0;i<EventCount;i++)
        {
            nn::ndd::AcquireSendDataUpdateEvent(&sduEvent[i]);
            nn::ndd::AcquireSendDataUpdateEvent(&rdEvent[i]);
        }
    }
    TEST_LOG("%s finish\n\n", __FUNCTION__);
}

void Ndd()
{
    TEST_LOG("%s start\n", __FUNCTION__);
    //自律すれちがいの切替
    {
        nn::ndd::EnableAutoCommunication();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));//同期手段が無いので少し待つ
        bool isEnabled;
        isEnabled = nn::ndd::IsAutoCommunicationEnabled();
        NN_ABORT_UNLESS(isEnabled == true);

        nn::ndd::DisableAutoCommunication();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));//同期手段が無いので少し待つ
        isEnabled = nn::ndd::IsAutoCommunicationEnabled();
        NN_ABORT_UNLESS(isEnabled == false);

        nn::ndd::EnableAutoCommunication();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));//同期手段が無いので少し待つ
        isEnabled = nn::ndd::IsAutoCommunicationEnabled();
        NN_ABORT_UNLESS(isEnabled == true);
    }

    //ネットワーク動作の確認
    {
        for(;;)
        {
            //ずっと変化しないケースは現状無いため、変わるまで待つ
            auto isEnabled = nn::ndd::IsNetworkActive();
            if(isEnabled)
            {
                break;
            }
            else
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        }
    }
    TEST_LOG("%s finish\n\n", __FUNCTION__);
}

void SendData()
{
    TEST_LOG("%s start\n", __FUNCTION__);
    //送信データの登録と確認
    {
        //登録
        nn::ndd::AddSendData(testUtil::DataGenerator::GenerateSendData(0, 0));
        nn::os::WaitSystemEvent(&g_SendDataUpdateEvent);

        //再登録
        nn::ndd::AddSendData(testUtil::DataGenerator::GenerateSendData(1, 0xFF));
        nn::os::WaitSystemEvent(&g_SendDataUpdateEvent);

        //取得
        nn::ndd::SendDataDescription sddResult;
        int count;
        count = nn::ndd::GetSendData(&sddResult, 0, 1);
        NN_ABORT_UNLESS(count == 1);

        //確認
        testUtil::DataGenerator::IsSameSendData(sddResult, 1, 0xFF);
    }

    //送信データの削除と確認
    {
        //削除
        nn::ndd::ClearSendData();
        nn::os::WaitSystemEvent(&g_SendDataUpdateEvent);

        //確認
        nn::ndd::SendDataDescription sddResult;
        int count;
        count = nn::ndd::GetSendData(&sddResult, 0, 1);
        NN_ABORT_UNLESS(count == 0);
    }
    TEST_LOG("%s finish\n\n", __FUNCTION__);
}

void ReceiveDataEmpty()
{
    TEST_LOG("%s start\n", __FUNCTION__);
    //空の状態での各種取得
    {
        testUtil::ValidateCounter(0);
        uint32_t nextCounter;
        auto resultCount = nn::ndd::GetReceiveData(&g_Rdd[0], &nextCounter, 0, RddCountMax);
        NN_ABORT_UNLESS(resultCount == 0);
        NN_ABORT_UNLESS(nextCounter == 0);
    }
    //空の状態での各種削除
    {
        nn::ndd::ClearReceiveData();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));//同期手段が無いので少し待つ
        testUtil::ValidateCounter(0);
        nn::ndd::ClearDataIdFilter();//確認手段無し
    }
    TEST_LOG("%s finish\n\n", __FUNCTION__);
}

void ReceiveData(int seed, uint32_t testCount)
{
    TEST_LOG("%s start\n", __FUNCTION__);

    int dataMode;
    if(seed == 0)
    {
        dataMode = 0;
    }
    else if(seed == 1)
    {
        dataMode = 1;
    }
    else
    {
        NN_ABORT("faulty design\n");
    }

    int overwriteCount;
    if(testCount > RingBufferCountMax)
    {
        overwriteCount = testCount - RingBufferCountMax;
    }
    else
    {
        overwriteCount = 0;
    }

    //受信データ追加
    for(int i=0;i<testCount;++i)
    {
        //受信データ追加
        {
            nn::os::ClearSystemEvent(&g_ReceiveDataEvent);
            nn::ndd::AddReceiveData(testUtil::DataGenerator::GenerateReceiveData(dataMode, i));
            nn::os::WaitSystemEvent(&g_ReceiveDataEvent);
        }
        //更新確認
        {
            testUtil::ValidateCounter(i + 1);
        }
        //受信データ取得
        {
            nn::ndd::ReceiveDataDescription rdd;
            uint32_t nextCounter;
            auto resultCount = nn::ndd::GetReceiveData(&rdd, &nextCounter, i, 1);

            NN_ABORT_UNLESS(resultCount == 1);
            testUtil::DataGenerator::IsSameReceiveData(rdd, dataMode, i);
            NN_ABORT_UNLESS(nextCounter == i + 1);
        }
    }
    //1件づつ受信データ取得
    {
        uint32_t counter = nn::ndd::GetOldestReceiveDataCounter();
        NN_ABORT_UNLESS(counter == overwriteCount);
        const auto ExpectedCount = nn::ndd::GetAvailableReceiveDataCount(counter);
        NN_ABORT_UNLESS(ExpectedCount == std::min(testCount, RingBufferCountMax));
        for(int i=0;;++i)
        {
            //TEST_LOG("%d\n",i);
            nn::ndd::ReceiveDataDescription rdd;
            auto resultCount = nn::ndd::GetReceiveData(&rdd, &counter, counter, 1);
            if(resultCount == 1)
            {
                testUtil::DataGenerator::IsSameReceiveData(rdd, dataMode, i);
                NN_ABORT_UNLESS(counter == overwriteCount + i + 1);
            }
            else if(resultCount == 0)
            {
                NN_ABORT_UNLESS(ExpectedCount == i);
                NN_ABORT_UNLESS(counter == nn::ndd::GetNextReceiveDataCounter());
                break;
            }
            else
            {
                NN_ABORT("faulty design\n");
            }
        }
    }
    //一括での受信データ取得
    {
        uint32_t counter = nn::ndd::GetOldestReceiveDataCounter();
        NN_ABORT_UNLESS(counter == overwriteCount);
        const auto ExpectedCount = nn::ndd::GetAvailableReceiveDataCount(counter);
        NN_ABORT_UNLESS(ExpectedCount == RingBufferCountMax);
        uint32_t nextCounter;
        auto resultCount = nn::ndd::GetReceiveData(&g_Rdd[0], &nextCounter, counter, RddCountMax);
        NN_ABORT_UNLESS(resultCount == ExpectedCount);
        NN_ABORT_UNLESS(nextCounter == nn::ndd::GetNextReceiveDataCounter());
        for(int i=0;i<resultCount;++i)
        {
            testUtil::DataGenerator::IsSameReceiveData(g_Rdd[i], dataMode, i);
        }
    }
    //貯まった状態での、新しいデータの指定件数の受信
    {
        uint32_t counter;
        const int Count = 20;
        counter = nn::ndd::GetRecentReceiveDataCounter(0, Count);
        NN_ABORT_UNLESS(nn::ndd::GetAvailableReceiveDataCount(counter) == Count);
        for(int i=0;;++i)
        {
            nn::ndd::ReceiveDataDescription rdd;
            auto resultCount = nn::ndd::GetReceiveData(&rdd, &counter, counter, 1);
            if(resultCount == 0)
            {
                NN_ABORT_UNLESS(i == Count);
                NN_ABORT_UNLESS(counter == nn::ndd::GetNextReceiveDataCounter());
                break;
            }
            testUtil::DataGenerator::IsSameReceiveData(rdd, dataMode, i);
            NN_ABORT_UNLESS(counter == RingBufferCountMax + overwriteCount - Count + 1 + i);
        }
    }
    //各種削除
    {
        nn::ndd::ClearReceiveData();
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));//同期手段が無いので少し待つ
        testUtil::ValidateCounter(0);
        nn::ndd::ClearDataIdFilter();//確認手段無し
    }
    TEST_LOG("%s finish\n\n", __FUNCTION__);
}//NOLINT(impl/function_size)

void ReceiveDataSimultaneously()
{
    TEST_LOG("%s start\n", __FUNCTION__);

    testUtil::CreateDummyDeviceThread();

    //受信しながらの受信データ取得
    //API単位の取得値として正しければよい。ので、fatalやデッドロックしないことをメインとし、値は目視確認
    for(int i=0;i<1000;++i)
    {
        TEST_LOG("Oldest = %d, Current = %d, Next = %d, Available = %d, Recent10 = %d\n",
                nn::ndd::GetOldestReceiveDataCounter(),
                nn::ndd::GetCurrentReceiveDataCounter(),
                nn::ndd::GetNextReceiveDataCounter(),
                nn::ndd::GetAvailableReceiveDataCount(nn::ndd::GetOldestReceiveDataCounter()),
                nn::ndd::GetRecentReceiveDataCounter(nn::ndd::GetOldestReceiveDataCounter(), 10));
        auto counter = nn::ndd::GetOldestReceiveDataCounter();
        auto count = nn::ndd::GetReceiveData(&g_Rdd[0], &counter, counter, RddCountMax);
        TEST_LOG("GetResult: Count = %d, Next = %d\n", count, counter);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(11));//[todo]揺らぎ足す
    }

    testUtil::WaitDummyDeviceThreadFinished();

    TEST_LOG("%s finish\n\n", __FUNCTION__);
}

void Scan()
{
    TEST_LOG("%s start\n", __FUNCTION__);

    //Scan結果0件を期待するScan
    {
        nn::ndd::StartDeviceScan();
        nn::os::WaitSystemEvent(&g_DeviceScanEvent);
        auto resultCount = nn::ndd::GetDeviceScanResult(&g_Rdd[0], RddCountMax);
        NN_ABORT_UNLESS(resultCount == 0);
    }

    //別スレッドで10秒間、100ms毎にデータを挿入
    testUtil::CreateDummyDeviceThread();

    //Scan
    {
        nn::ndd::StartDeviceScan();
        nn::os::WaitSystemEvent(&g_DeviceScanEvent);
        auto resultCount = nn::ndd::GetDeviceScanResult(&g_Rdd[0], RddCountMax);
        TEST_LOG("resultCount = %d\n", resultCount);
        NN_ABORT_UNLESS(resultCount != 0);
        for(int i=0;i<resultCount;++i)
        {
            //[todo]確認内容の拡充
            NN_ABORT_UNLESS(g_Rdd[i].dataSize == nn::ndd::SendDataSizeMax);
        }
    }

    //[todo]キャンセル時間の評価
    //Scanのキャンセル
    {
        nn::ndd::StartDeviceScan();
        nn::ndd::CancelDeviceScan();
        nn::os::WaitSystemEvent(&g_DeviceScanEvent);
        auto resultCount = nn::ndd::GetDeviceScanResult(&g_Rdd[0], RddCountMax);
        TEST_LOG("resultCount = %d\n", resultCount);
        for(int i=0;i<resultCount;++i)
        {
            //[todo]確認内容の拡充
            NN_ABORT_UNLESS(g_Rdd[i].dataSize == nn::ndd::SendDataSizeMax);
        }
    }

    testUtil::WaitDummyDeviceThreadFinished();

    TEST_LOG("%s finish\n\n", __FUNCTION__);
}

void QueueFull()
{
    TEST_LOG("%s start\n", __FUNCTION__);

    nn::ndd::ReceiveDataDescription rdd;
    rdd.data[0] = 0;
    rdd.dataSize = 1;
    for(int i=0;i<100;++i)
    {
        nn::ndd::AddReceiveData(rdd);
    }

    TEST_LOG("%s finish\n\n", __FUNCTION__);
}

//TEST_LOG("%s start\n", __FUNCTION__);
//TEST_LOG("%s finish\n\n", __FUNCTION__);


}

extern "C" void nnMain()
{
    testUtil::Init();

    //単機能のテスト
    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::Padding();

    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::Init();

    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::Ndd();

    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::SendData();

    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::ReceiveDataEmpty();

    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::ReceiveData(0, RingBufferCountMax);

    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::ReceiveData(1, RingBufferCountMax + 10);

    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::ReceiveDataSimultaneously();

    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::Scan();

    //複数デバイスを用いた単機能のテスト
    //[todo]DataIdFilterの動作確認。およびClear

    //ステート系のテスト
    //連続試行、エージング
    //異常系。サイズ境界等
    testUtil::Reset(testUtil::NetworkType_Disabled);
    testSuite::QueueFull();

    testUtil::Reset(testUtil::NetworkType_Enabled);
}
