﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto.h>
#include <nn/nn_SdkLog.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fssystem/fs_DeltaHeader.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_DeltaApplier.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

namespace {

/**
 * @brief   ApplyDeltaPerformanceTest で利用するテストフィクスチャです。
 */
class ApplyDeltaPerformanceTest : public ::testing::Test
{
protected:

    /**
    * @brief テスト開始時に毎回呼び出される関数です。
    */
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::ncm::Initialize();
    }

    /**
    * @brief テスト終了時に毎回呼び出される関数です。
    */
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::ncm::Finalize();
    }
};

}

TEST_F(ApplyDeltaPerformanceTest, CheckApplyDeltaCharacteristic)
{
    const int TryCountSmallSize = 1000;
    const int TryCountLargeSize = 250;
    const int LargeSizeThreshold = 256 * 1024;
    const int WriteSizeMin = 4;
    const int WriteSizeMax = 8 * 1024 * 1024;
    const int BufferSize = 128 * 1024;

    int64_t contentSize = 0;
    int64_t deltaSize = 0;
    for (int length = WriteSizeMin; length <= WriteSizeMax; length *= 2)
    {
        auto tryCount = length >= LargeSizeThreshold ? TryCountLargeSize : TryCountSmallSize;

        contentSize += length * tryCount;

        auto commandSize = nn::fssystem::DeltaCommandWrite::GetCommandSize(0, length) + length;
        deltaSize += commandSize * tryCount;
    }

    nn::ncm::ContentId contentId;
    {
        char mac[nn::crypto::HmacSha256Generator::MacSize];
        nn::crypto::GenerateHmacSha256Mac(
            mac, sizeof(mac),
            "testtesttest", 12,
            "testtesttest", 12
        );
        std::memcpy(&contentId, mac, sizeof(contentId));
    }

    nn::ncm::ContentStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, nn::ncm::StorageId::BuildInUser));

    auto placeHolder = storage.GeneratePlaceHolderId();
    NNT_ASSERT_RESULT_SUCCESS(storage.CreatePlaceHolder(placeHolder, contentId, contentSize));
    NN_UTIL_SCOPE_EXIT
    {
        storage.DeletePlaceHolder(placeHolder);
    };

    std::unique_ptr<char[]> buffer(new char[BufferSize]);
    GTEST_ASSERT_NE(nullptr, buffer.get());

    std::unique_ptr<nn::ncm::DeltaApplier> pApplier(new nn::ncm::DeltaApplier());
    GTEST_ASSERT_NE(nullptr, pApplier.get());

    char status[nn::ncm::DeltaApplier::StatusSize];
    int64_t offset;
    NNT_ASSERT_RESULT_SUCCESS(pApplier->Initialize(&offset, &storage, placeHolder, status, 0));

    {
        nn::fssystem::DeltaHeader header;
        std::memset(&header, 0, sizeof(header));
        header.signature = static_cast<uint32_t>(nn::fssystem::DeltaHeader::Signature::V0);
        header.offsetBody = nn::fssystem::DeltaHeader::Size;
        header.sizeSource = contentSize;
        header.sizeDestination = contentSize;
        header.sizeBody = deltaSize;

        size_t sizeProcessed;
        NNT_ASSERT_RESULT_SUCCESS(pApplier->ApplyDelta(&sizeProcessed, status, sizeof(status), &header, sizeof(header)));
        ASSERT_EQ(sizeProcessed, sizeof(header));
        NNT_ASSERT_RESULT_SUCCESS(pApplier->ApplyDelta(&sizeProcessed, status, sizeof(status), buffer.get(), header.offsetBody - sizeof(header)));
        ASSERT_EQ(sizeProcessed, header.offsetBody - sizeof(header));
    }

    NN_SDK_LOG("size (byte),time (millisec),count\n");

    for (int length = WriteSizeMin; length <= WriteSizeMax; length *= 2)
    {
        auto tryCount = length >= LargeSizeThreshold ? TryCountLargeSize : TryCountSmallSize;

        nn::fssystem::DeltaCommandWrite::Create(buffer.get(), BufferSize, 0, length);
        auto commandHeaderSize = nn::fssystem::DeltaCommandWrite::GetCommandSize(0, length);
        auto commandSize = commandHeaderSize + length;

        int commandCount = std::min(BufferSize / commandSize, tryCount);
        for (int i = 1; i < commandCount; ++i)
        {
            std::memcpy(buffer.get() + (i * commandSize), buffer.get(), commandHeaderSize);
        }

        nn::os::Tick tick = nn::os::GetSystemTick();

        if (commandCount == 0)
        {
            int tryIndex = 0;
            int offset = 0;
            while (tryIndex < tryCount)
            {
                int currentSize = std::min(commandSize - offset, BufferSize);
                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(pApplier->ApplyDelta(&sizeProcessed, status, sizeof(status), buffer.get(), currentSize));
                offset += sizeProcessed;
                if (offset == commandSize)
                {
                    offset = 0;
                    ++tryIndex;
                }
            }
        }
        else
        {
            int tryIndex = 0;
            while (tryIndex < tryCount)
            {
                int currentTryCount = std::min(tryCount - tryIndex, commandCount);
                size_t sizeProcessed;
                NNT_ASSERT_RESULT_SUCCESS(pApplier->ApplyDelta(&sizeProcessed, status, sizeof(status), buffer.get(), commandSize * currentTryCount));
                tryIndex += currentTryCount;
            }
        }

        auto timeMilliSeconds = (nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds();
        NN_SDK_LOG("%d,%lld,%d\n", length, timeMilliSeconds, tryCount);
        NN_UNUSED(timeMilliSeconds);
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    static const size_t HeapSize = 32 * 1024 * 1024;
    void* pHeapStack = ::malloc(HeapSize);

    nnt::fs::util::InitializeTestLibraryHeap(pHeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    ::testing::InitGoogleTest(&argc, argv);

    int result = RUN_ALL_TESTS();

    ::free(pHeapStack);

    nnt::Exit(result);
}
