﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/nn_Abort.h>

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return std::malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

class Buffer
{
public:
    explicit Buffer(size_t size) :
        m_data(nullptr),
        m_size(0)
    {
        m_size = size;
        m_data = (size > 0? std::malloc(m_size): nullptr);
    }
    ~Buffer()
    {
        if (m_data != nullptr)
        {
            std::free(m_data);
            m_data = nullptr;
        }
    }
    void *GetDataPtr() const
    {
        return m_data;
    }
    size_t GetDataSize() const
    {
        return m_size;
    }

private:
    void *m_data;
    size_t m_size;
};

class FileData
{
public:
    explicit FileData(const char *path) :
        m_data(nullptr),
        m_size(0)
    {
        /*int64_t fileSize = 0;
        NvOsStatType        stat;
        size_t BytesRead;
        NvOsFileHandle fpInputBuffer;
        NN_ABORT_UNLESS(NvSuccess == NvOsFopen(path, NVOS_OPEN_READ, &fpInputBuffer));
        NvOsFstat(fpInputBuffer, &stat);
        fileSize = (int64_t)stat.size;
        m_size = static_cast<size_t>(fileSize);
        m_data = std::malloc(m_size);
        NN_ABORT_UNLESS(nullptr != m_data);
        NvOsFread(fpInputBuffer, m_data, fileSize, &BytesRead);
        NN_ABORT_UNLESS(BytesRead);
        NvOsFclose(fpInputBuffer);*/
        nn::fs::FileHandle fileHandle;
        int64_t fileSize = 0;
        NN_ABORT_UNLESS(
            nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read).IsSuccess() && // ファイルのオープンに失敗
            nn::fs::GetFileSize(&fileSize, fileHandle).IsSuccess() && // ファイルのサイズ取得に失敗
            fileSize <= 0xFFFFFFFF); // ファイルが大きすぎる。
        m_size = static_cast<size_t>(fileSize);
        m_data = std::malloc(m_size);
        NN_ABORT_UNLESS(
            m_data != nullptr &&
            nn::fs::ReadFile(fileHandle, 0, m_data, m_size).IsSuccess());
        nn::fs::CloseFile(fileHandle);

    }
    ~FileData()
    {
        if (m_data != nullptr)
        {
            std::free(m_data);
            m_data = nullptr;
        }
    }
    const void *GetDataPtr() const
    {
        return m_data;
    }
    size_t GetDataSize() const
    {
        return m_size;
    }

private:
    void *m_data;
    size_t m_size;
};

NN_FORCEINLINE void LogIfNotNull(const char *label, const char *data)
{
    if (data != nullptr)
    {
        NN_LOG("%s: \"%s\"\n", label, data);
    }
}
