﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//#define DEBUG_LOG

//#define IMAGE_DUMP
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/init.h>
#include <cstdio>
#include <nn/image/image_ExifExtractor.h>
#include <nn/image/image_JpegDecoder.h>
#include "AVIExtractor.h"
#include "Utilities.h"


int parse_riff(FILE* in, char* outFilePath)
{
    char chunk_id[5];
    int chunk_size;
    char chunk_type[5];
    int end_of_chunk,end_of_subchunk;
    struct avi_header_t avi_header;
    struct stream_header_t stream_header;
    struct stream_format_t stream_format = {0};
    long offset=ftell(in);

    read_chars(in,chunk_id,4);
    chunk_size=read_long(in);
    read_chars(in,chunk_type,4);

    NN_LOG("RIFF Chunk (id=%s size=%d type=%s offset=0x%lx)\n",chunk_id,chunk_size,chunk_type, offset);
    NN_LOG("{\n");

    if (strcasecmp("RIFF",chunk_id)!=0)
    {
        NN_LOG("Not a RIFF file.\n");
        return 1;
    }
        else
    if (strcasecmp("AVI ",chunk_type)!=0)
    {
        NN_LOG("Not an AVI file.\n");
        return 1;
    }

    end_of_chunk=ftell(in) + chunk_size - 4;
    while (ftell(in)<end_of_chunk)
    {
        read_chars(in,chunk_id,4);
        chunk_size=read_long(in);
        end_of_subchunk=ftell(in) + chunk_size;

        if (strcasecmp("JUNK",chunk_id)==0 || strcasecmp("PAD ",chunk_id)==0)
        {
            chunk_type[0]=0; }
        else
        {
            read_chars(in,chunk_type,4);
        }
#ifdef DEBUG_LOG
        long offset=ftell(in);
        NN_LOG("   New Chunk (id=%s size=%d type=%s offset=0x%lx)\n",chunk_id,chunk_size,chunk_type,offset);
        NN_LOG("   {\n");
#endif
        fflush(stdout);

        if (strcasecmp("JUNK",chunk_id)==0 || strcasecmp("PAD ",chunk_id)==0)
        {
            if ((chunk_size % 4)!=0)
            {
                chunk_size=chunk_size + (4 - (chunk_size % 4));
            }

            //hex_dump_chunk(in, chunk_size);
        }
        else
        if (strcasecmp("INFO",chunk_type)==0)
        {
            if ((chunk_size % 4)!=0)
            {
                chunk_size=chunk_size + (4 - (chunk_size % 4));
            }
            //hex_dump_chunk(in, chunk_size);
        }
        else
        if (strcasecmp("hdrl",chunk_type)==0)
        {
            parse_hdrl(in,&avi_header,&stream_header,&stream_format, chunk_size);
            /* skip_chunk(in); */
        }
        else
        if (strcasecmp("movi",chunk_type)==0)
        {
            NN_LOG("\n*********Parse movi data**********\n");
            parse_movi_list(&avi_header, in, outFilePath);
        }

        else
        if (strcasecmp("idx1",chunk_id)==0)
        {
            fseek(in,ftell(in) - 4,SEEK_SET);
            //parse_idx1(in,chunk_size);
        }
        else
        {
            NN_LOG("Unknown chunk at %d (%4s)\n",(int)ftell(in) - 8,chunk_type);
            if (chunk_size==0) break;
        }

        fseek(in,end_of_subchunk,SEEK_SET);
#ifdef DEBUG_LOG
        NN_LOG("   }\n");
#endif

    }

    if (stream_format.palette!=0)
    {
        free(stream_format.palette);
    }

    NN_LOG("}\n");
    return 0;
}

int OutFile(int fn,FILE** fp, char* outFilePath)
{
    char buf[30];
    char buf1[15];
    strcpy(buf, outFilePath);
#ifndef IMAGE_DUMP
    sprintf(buf1,"\\avi.%04d.rgb",fn);
#else
    sprintf(buf1,"\\avi.%04d.jpg",fn);
#endif
    strcat(buf, buf1);
    if (!(*fp = fopen(buf, "wb")))
    {
        NN_LOG("Output file fail\n\n");
        return 0;
    }
    return 0;
}


int parse_movi_list(struct avi_header_t *avi_header, FILE *in, char* outFilePath)
{
    char chunk_id[5];
    int chunk_size;
    //int stream_type=0;     // 0=video 1=sound
    FILE *ofp ;
    int i =0;

    for(i= 0; i<avi_header->TotalNumberOfFrames; i++){
        read_chars(in,chunk_id,4);
        chunk_size = read_long(in);
        NN_LOG("AVI Header LIST (id=%s size=%d offset=0x%lx)\n",chunk_id,chunk_size,ftell(in));
        if ((chunk_size % 2)!=0)
        {
            chunk_size = chunk_size + 1;
        }
#ifdef IMAGE_DUMP
        OutFile(i, &ofp, outFilePath);
        for (n=0; n<chunk_size; n++){
            putc(getc(in), ofp);
        }
        fclose(ofp);
#else
        Buffer pixBuf(1920 * 1080 * sizeof(nn::Bit32));
        static const int AlignmentSize = 4;
        nn::image::Dimension dim;
        if (!DecodeJpegData(&dim, &pixBuf, AlignmentSize, in, chunk_size)){
            NN_LOG("[Failed to decode JPEG ]\n");
        }
        if (outFilePath){
            OutFile(i, &ofp, outFilePath);
            fwrite(pixBuf.GetDataPtr(), 1, dim.width * dim.height * 4, ofp);
            fclose(ofp);
        }
#endif
        NN_LOG("\n*****Decoded image count ::%d******\n",i);

    }
    return 0;
}




int hex_dump_chunk(FILE *in, int chunk_len)
{
    char chars[17];
    int ch,n;

    chars[16]=0;

    for (n=0; n<chunk_len; n++)
    {
    if ((n % 16)==0)
    {
        if (n!=0) NN_LOG("%s\n", chars);
        NN_LOG("      ");
        memset(chars, ' ', 16);
    }
    ch=getc(in);
    if (ch==EOF) break;
    NN_LOG("%02x ", ch);
    if (ch>=' ' && ch<=126){
        chars[n % 16]=ch; }
    else{
        chars[n % 16]='.'; }
    }

    if ((n % 16)!=0){
        for (ch=n % 16; ch<16; ch++) {
            NN_LOG("   ");
        }
    }
    NN_LOG("%s\n", chars);
    return 0;
}

int parse_idx1(FILE *in, int chunk_len)
{
    struct index_entry_t index_entry;
    int t;

    NN_LOG("      IDX1\n");
    NN_LOG("      -------------------------------\n");
    NN_LOG("      ckid   dwFlags         dwChunkOffset        dwChunkLength\n");

    for (t=0; t<chunk_len / 16; t++)
    {
        read_chars(in,index_entry.ckid,4);
        index_entry.dwFlags=read_long(in);
        index_entry.dwChunkOffset=read_long(in);
        index_entry.dwChunkLength=read_long(in);

        NN_LOG("      %s   0x%08x      0x%08x           0x%08x\n",
                index_entry.ckid,
                index_entry.dwFlags,
                index_entry.dwChunkOffset,
                index_entry.dwChunkLength);
    }

    NN_LOG("\n");

    return 0;
}

int read_avi_header(FILE *in,struct avi_header_t *avi_header)
{
    avi_header->TimeBetweenFrames=read_long(in);
    avi_header->MaximumDataRate=read_long(in);
    avi_header->PaddingGranularity=read_long(in);
    avi_header->Flags=read_long(in);
    avi_header->TotalNumberOfFrames=read_long(in);
    avi_header->NumberOfInitialFrames=read_long(in);
    avi_header->NumberOfStreams=read_long(in);
    avi_header->SuggestedBufferSize=read_long(in);
    avi_header->Width=read_long(in);
    avi_header->Height=read_long(in);
    avi_header->TimeScale=read_long(in);
    avi_header->DataRate=read_long(in);
    avi_header->StartTime=read_long(in);
    avi_header->DataLength=read_long(in);
#ifdef DEBUG_LOG
    NN_LOG("         offset=0x%lx\n",offset);
    NN_LOG("             TimeBetweenFrames: %d\n",avi_header->TimeBetweenFrames);
    NN_LOG("               MaximumDataRate: %d\n",avi_header->MaximumDataRate);
    NN_LOG("            PaddingGranularity: %d\n",avi_header->PaddingGranularity);
    NN_LOG("                         Flags: %d\n",avi_header->Flags);
    NN_LOG("           TotalNumberOfFrames: %d\n",avi_header->TotalNumberOfFrames);
    NN_LOG("         NumberOfInitialFrames: %d\n",avi_header->NumberOfInitialFrames);
    NN_LOG("               NumberOfStreams: %d\n",avi_header->NumberOfStreams);
    NN_LOG("           SuggestedBufferSize: %d\n",avi_header->SuggestedBufferSize);
    NN_LOG("                         Width: %d\n",avi_header->Width);
    NN_LOG("                        Height: %d\n",avi_header->Height);
    NN_LOG("                     TimeScale: %d\n",avi_header->TimeScale);
    NN_LOG("                      DataRate: %d\n",avi_header->DataRate);
    NN_LOG("                     StartTime: %d\n",avi_header->StartTime);
    NN_LOG("                    DataLength: %d\n",avi_header->DataLength);
#endif
    return 0;
}

void print_data_handler(unsigned char *handler)
{
    int t;

    for (t=0; t<4; t++)
    {
        if ((handler[t]>='a' && handler[t]<='z') ||
            (handler[t]>='A' && handler[t]<='Z') ||
            (handler[t]>='0' && handler[t]<='9'))
        {
        NN_LOG("%c",handler[t]);
        }
        else
        {
        NN_LOG("[0x%02x]",handler[t]);
        }
    }
}

int read_stream_header(FILE *in,struct stream_header_t *stream_header)
{
    read_chars(in,stream_header->DataType,4);
    read_chars(in,stream_header->DataHandler,4);
    stream_header->Flags=read_long(in);
    stream_header->Priority=read_long(in);
    stream_header->InitialFrames=read_long(in);
    stream_header->TimeScale=read_long(in);
    stream_header->DataRate=read_long(in);
    stream_header->StartTime=read_long(in);
    stream_header->DataLength=read_long(in);
    stream_header->SuggestedBufferSize=read_long(in);
    stream_header->Quality=read_long(in);
    stream_header->SampleSize=read_long(in);
#ifdef DEBUG_LOG
    NN_LOG("            offset=0x%lx\n",offset);
    NN_LOG("                      DataType: %s\n",stream_header->DataType);
    NN_LOG("                   DataHandler: ");
    print_data_handler((unsigned char *)stream_header->DataHandler);
    NN_LOG("\n");
    NN_LOG("                         Flags: %d\n",stream_header->Flags);
    NN_LOG("                      Priority: %d\n",stream_header->Priority);
    NN_LOG("                 InitialFrames: %d\n",stream_header->InitialFrames);
    NN_LOG("                     TimeScale: %d\n",stream_header->TimeScale);
    NN_LOG("                      DataRate: %d\n",stream_header->DataRate);
    NN_LOG("                     StartTime: %d\n",stream_header->StartTime);
    NN_LOG("                    DataLength: %d\n",stream_header->DataLength);
    NN_LOG("           SuggestedBufferSize: %d\n",stream_header->SuggestedBufferSize);
    NN_LOG("                       Quality: %d\n",stream_header->Quality);
    NN_LOG("                    SampleSize: %d\n",stream_header->SampleSize);
#endif
    return 0;
}

int read_stream_format(FILE *in,struct stream_format_t *stream_format)
{
    int t,r,g,b;

    stream_format->header_size=read_long(in);
    stream_format->image_width=read_long(in);
    stream_format->image_height=read_long(in);
    stream_format->number_of_planes=read_word(in);
    stream_format->bits_per_pixel=read_word(in);
    stream_format->compression_type=read_long(in);
    stream_format->image_size_in_bytes=read_long(in);
    stream_format->x_pels_per_meter=read_long(in);
    stream_format->y_pels_per_meter=read_long(in);
    stream_format->colors_used=read_long(in);
    stream_format->colors_important=read_long(in);
    stream_format->palette=0;

    if (stream_format->colors_important!=0)
    {
        stream_format->palette = (int *)malloc(stream_format -> colors_important * sizeof(int));
        for (t=0; t<stream_format->colors_important; t++)
        {
        b=getc(in);
        g=getc(in);
        r=getc(in);
        stream_format->palette[t]=(r<<16) + (g<<8) + b;
        }
    }
#ifdef DEBUG_LOG
    NN_LOG("            offset=0x%lx\n",offset);
    NN_LOG("                   header_size: %d\n",stream_format->header_size);
    NN_LOG("                   image_width: %d\n",stream_format->image_width);
    NN_LOG("                  image_height: %d\n",stream_format->image_height);
    NN_LOG("              number_of_planes: %d\n",stream_format->number_of_planes);
    NN_LOG("                bits_per_pixel: %d\n",stream_format->bits_per_pixel);
    NN_LOG("              compression_type: %04x (%c%c%c%c)\n",stream_format->compression_type,
                            ((stream_format->compression_type)&255),
                            ((stream_format->compression_type>>8)&255),
                            ((stream_format->compression_type>>16)&255),
                            ((stream_format->compression_type>>24)&255));
    NN_LOG("           image_size_in_bytes: %d\n",stream_format->image_size_in_bytes);
    NN_LOG("              x_pels_per_meter: %d\n",stream_format->x_pels_per_meter);
    NN_LOG("              y_pels_per_meter: %d\n",stream_format->y_pels_per_meter);
    NN_LOG("                   colors_used: %d\n",stream_format->colors_used);
    NN_LOG("              colors_important: %d\n",stream_format->colors_important);
#endif
    return 0;
}

int read_stream_format_auds(FILE *in, struct stream_format_auds_t *stream_format)
{
    stream_format->format=read_word(in);
    stream_format->channels=read_word(in);
    stream_format->samples_per_second=read_long(in);
    stream_format->bytes_per_second=read_long(in);
    stream_format->block_size_of_data=read_word(in);
    stream_format->bits_per_sample=read_word(in);
    //stream_format->extended_size=read_word(in);
#ifdef DEBUG_LOG
    long offset=ftell(in);
    int block_align=read_word(in);
    NN_LOG("            offset=0x%lx\n",offset);
    NN_LOG("                        format: %d\n",stream_format->format);
    NN_LOG("                      channels: %d\n",stream_format->channels);
    NN_LOG("            samples_per_second: %d\n",stream_format->samples_per_second);
    NN_LOG("              bytes_per_second: %d\n",stream_format->bytes_per_second);
    NN_LOG("                   block_align: %d\n",block_align);
    NN_LOG("            block_size_of_data: %d\n",stream_format->block_size_of_data);
    NN_LOG("               bits_per_sample: %d\n",stream_format->bits_per_sample);
#endif

    return 0;
}

int parse_hdrl_list(FILE *in,struct avi_header_t *avi_header, struct stream_header_t *stream_header, struct stream_format_t *stream_format)
{
    struct stream_format_auds_t stream_format_auds;
    struct stream_header_t stream_header_auds;
    char chunk_id[5];
    int chunk_size;
    char chunk_type[5];
    int end_of_chunk;
    int next_chunk;
    int stream_type=0;     // 0=video 1=sound

    read_chars(in,chunk_id,4);
    chunk_size=read_long(in);
    read_chars(in,chunk_type,4);
#ifdef DEBUG_LOG
    long offset=ftell(in);
    NN_LOG("      AVI Header LIST (id=%s size=%d type=%s offset=0x%lx)\n",chunk_id,chunk_size,chunk_type,offset);
    NN_LOG("      {\n");
#endif
    end_of_chunk=ftell(in) + chunk_size - 4;
    if ((end_of_chunk % 4)!=0)
    {
    //NN_LOG("Adjusting end of chunk %d\n", end_of_chunk);
        //end_of_chunk=end_of_chunk+(4-(end_of_chunk%4));
    //NN_LOG("Adjusting end of chunk %d\n", end_of_chunk);
    }

    if (strcmp(chunk_id,"JUNK")==0)
    {
        fseek(in,end_of_chunk,SEEK_SET);
#ifdef DEBUG_LOG
        NN_LOG("      }\n");
#endif
        return 0;
    }

    while (ftell(in)<end_of_chunk)
    {
        read_chars(in,chunk_type,4);
        chunk_size=read_long(in);
        next_chunk=ftell(in) + chunk_size;
        if ((chunk_size % 4)!=0)
        {
        //NN_LOG("Chunk size not a multiple of 4?\n");
        //chunk_size=chunk_size+(4-(chunk_size%4));
        }
#ifdef DEBUG_LOG
        long offset=ftell(in);
        NN_LOG("         %.4s (size=%d offset=0x%lx)\n",chunk_type,chunk_size,offset);
        NN_LOG("         {\n");
#endif
        if (strcasecmp("strh",chunk_type)==0)
        {
        long marker=ftell(in);
        char buffer[5];
        read_chars(in,buffer,4);
        fseek(in,marker,SEEK_SET);

        if (strcmp(buffer, "vids")==0)
        {
            stream_type=0;
            read_stream_header(in,stream_header);
        }
            else
        if (strcmp(buffer, "auds")==0)
        {
            stream_type=1;
            read_stream_header(in,&stream_header_auds);
        }
            else
        {
            NN_LOG("Unknown stream type %s\n", buffer);
            return -1;
        }
        }
        else
        if (strcasecmp("strf",chunk_type)==0)
        {
        if (stream_type==0)
        {
            read_stream_format(in,stream_format);
        }
        else
        {
            read_stream_format_auds(in,&stream_format_auds);
        }
        }
        else
        if (strcasecmp("strd",chunk_type)==0)
        {

        }
        else
        {
#ifdef DEBUG_LOG
        NN_LOG("            Unknown chunk type: %s\n",chunk_type);
#endif
        // skip_chunk(in);
        }
#ifdef DEBUG_LOG
        NN_LOG("         }\n");
#endif
        fseek(in,next_chunk,SEEK_SET);
    }

    //NN_LOG("@@@@ %ld %d\n", ftell(in), end_of_chunk);
#ifdef DEBUG_LOG
    NN_LOG("      }\n");
#endif
    fseek(in,end_of_chunk,SEEK_SET);

    return 0;
}

int parse_hdrl(FILE *in,struct avi_header_t *avi_header, struct stream_header_t *stream_header, struct stream_format_t *stream_format, unsigned int size)
{
    char chunk_id[5];
    int chunk_size;
    int end_of_chunk;
    long offset=ftell(in);

    read_chars(in,chunk_id,4);
    chunk_size=read_long(in);
#ifdef DEBUG_LOG
    NN_LOG("      AVI Header Chunk (id=%s size=%d offset=0x%lx)\n",chunk_id,chunk_size,offset);
    NN_LOG("      {\n");
#endif
    end_of_chunk=ftell(in) + chunk_size;
    if ((end_of_chunk % 4)!=0)
    {
        end_of_chunk=end_of_chunk + (4 - (end_of_chunk % 4));
    }

    read_avi_header(in,avi_header);
#ifdef DEBUG_LOG
    NN_LOG("      }\n");
#endif

    while(ftell(in)<offset + size - 4)
    {
        //NN_LOG("Should end at 0x%lx  0x%lx\n",offset+size,ftell(in));
        parse_hdrl_list(in,avi_header,stream_header,stream_format);
    }

    return 0;
}





int read_long(FILE *in)
{
    int c;

    c=getc(in);
    c=c + (getc(in)<<8);
    c=c + (getc(in)<<16);
    c=c + (getc(in)<<24);

    return c;
}

int read_word(FILE *in)
{
    int c;

    c=getc(in);
    c=c + (getc(in)<<8);

    return c;
}

int read_chars(FILE *in, char *s, int count)
{
    int t;

    for (t=0; t<count; t++)
    {
        s[t]=getc(in);
    }

    s[t]=0;

    return 0;
}

bool DecodeJpegData(
    nn::image::Dimension *pOutDim,
    Buffer *pOutBuf,
    const int alignment,
    FILE *in, int chunkSize){

    /* ----------------------------------------------------------------------------------------
        デコード設定フェーズ
     */
    int n;
    char* mData;

    mData = (char*)malloc(chunkSize);
    for (n=0; n<chunkSize; n++){
        mData[n] = getc(in);
    }

    #if 1
    // デコーダのインスタンス
    nn::image::JpegDecoder decoder;

    // JPEG データを設定します。
    //decoder.SetImageData(jpegData.GetDataPtr(), jpegData.GetDataSize());
    decoder.SetImageData(mData, chunkSize);

    // [入力例]
    // 縦横をそれぞれ 1/2 倍してデコードする場合、分母 (2) を指定します。等倍の場合は省略可能です。
    decoder.SetResolutionDenominator(1);

    /* ----------------------------------------------------------------------------------------
        JPEG データ解析フェーズ
     */
    nn::image::JpegStatus jpegStatus = decoder.Analyze();
    switch (jpegStatus)
    {
    case nn::image::JpegStatus_WrongFormat:
        NN_LOG("[この JPEG データは破損しています。]\n");
        return false;
    case nn::image::JpegStatus_UnsupportedFormat:
        NN_LOG("[この JPEG データはサポートしていない形式です。]\n");
        return false;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
    }

    // 画像の幅と高さを取得
    *pOutDim = decoder.GetAnalyzedDimension();
    NN_LOG("出力画像サイズ: 幅 %u px, 高さ %u px\n", pOutDim->width, pOutDim->height);

    /* ----------------------------------------------------------------------------------------
        Exif 情報解析フェーズ
        (Exif 情報を取り扱わない場合はこのセクションは読み飛ばしてください。)
     */
    // Exif データの場所を取得
    size_t exifSize;
    const void *exifData = decoder.GetAnalyzedExifData(&exifSize);
    if (exifData != nullptr)
    {
        NN_LOG("Exif 情報が見つかりました。\n");

        // Exif 情報の内容を取得し内容を表示
        MetaData metaData;
        ExtractExifData(&metaData, exifData, exifSize);
        PrintMetaData(metaData);

        if (metaData.thumbnail != nullptr)
        {
            // [入力例]
            // サムネイル画像があれば、デコード対象を置き換えます。
            NN_LOG("→ デコード対象をサムネイルに変更します。\n");
            decoder.SetImageData(metaData.thumbnail, metaData.thumbnailSize);

            // [入力例]
            // サムネイルは縮小せずにデコードします。 (↑上で 2 に設定した値を 1 に戻す。)
            decoder.SetResolutionDenominator(1);

            nn::image::JpegStatus jpegStatus = decoder.Analyze();
            switch (jpegStatus)
            {
            case nn::image::JpegStatus_WrongFormat:
                NN_LOG("[このサムネイルは破損しています。]\n");
                return false;
            case nn::image::JpegStatus_UnsupportedFormat:
                NN_LOG("[このサムネイルはサポートしていない形式です。]\n");
                return false;
            default:
                NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
            }

            // 画像の幅と高さを取得
            *pOutDim = decoder.GetAnalyzedDimension();
            NN_LOG("出力画像サイズ: 幅 %u px, 高さ %u px\n", pOutDim->width, pOutDim->height);
        }
    }

    /* ----------------------------------------------------------------------------------------
        JPEG データ展開フェーズ
     */
    NN_LOG("JPEG データをデコードします。\n");

    // 展開に必要なメモリ量を取得
    Buffer workBuf(decoder.GetAnalyzedWorkBufferSize());
    NN_LOG(" - 必要な作業メモリ量: %zu bytes\n", workBuf.GetDataSize());

    // 出力サイズを計算
    const size_t AlignedWidth = pOutDim->width % alignment == 0? pOutDim->width: pOutDim->width - pOutDim->width % alignment + alignment;
    if (AlignedWidth * pOutDim->height * sizeof(nn::Bit32) > pOutBuf->GetDataSize())
    {
        NN_LOG(
            "[出力データ(%zu バイト)がバッファの大きさ(%zu バイト)を超過しました。]\n",
            AlignedWidth * pOutDim->height * sizeof(nn::Bit32),
            pOutBuf->GetDataSize());
        return false;
    }

    // デコード処理
    jpegStatus = decoder.Decode(
        pOutBuf->GetDataPtr(),
        pOutBuf->GetDataSize(),
        alignment,
        workBuf.GetDataPtr(),
        workBuf.GetDataSize());
    switch (jpegStatus)
    {
    case nn::image::JpegStatus_WrongFormat:
        NN_LOG("[この JPEG データは破損しています。]\n");
        return false;
    case nn::image::JpegStatus_UnsupportedFormat:
        NN_LOG("[この JPEG データはサポートしていない形式です。]\n");
        return false;
    case nn::image::JpegStatus_OutOfMemory:
        NN_LOG("[デコードに必要なワークメモリが不足しました。未知の形式のデータが入力された可能性があります。]\n");
        return false;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
    }

    // この時点で、decoder オブジェクトは不要になります。
    #endif
    return true;
}

void PrintMetaData(const MetaData &metaData)
{
    LogIfNotNull(" - メーカー名", metaData.maker);
    LogIfNotNull(" - 撮影機器名", metaData.model);
    if (metaData.hasOrientation)
    {
        const char *orientationStr;
        switch (metaData.orientation)
        {
        case nn::image::ExifOrientation_Normal:
            orientationStr = "正立";
            break;
        case nn::image::ExifOrientation_FlipHorizontal:
            orientationStr = "水平方向に反転";
            break;
        case nn::image::ExifOrientation_Rotate180:
            orientationStr = "時計回りに 180 度回転";
            break;
        case nn::image::ExifOrientation_FlipVertical:
            orientationStr = "垂直方向に反転";
            break;
        case nn::image::ExifOrientation_FlipTopRightToLeftBottom:
            orientationStr = "右下がりの対角線を軸に、上辺右端が左辺下端に来るように反転";
            break;
        case nn::image::ExifOrientation_Rotate270:
            orientationStr = "時計回りに 270 度回転";
            break;
        case nn::image::ExifOrientation_FlipTopLeftToRightBottom:
            orientationStr = "左下がりの対角線を軸に、上辺左端が右辺下端に来るように反転";
            break;
        case nn::image::ExifOrientation_Rotate90:
            orientationStr = "時計回りに 90 度回転";
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        LogIfNotNull(" - 画像方向", orientationStr);
    }
    LogIfNotNull(" - ソフトウェア名", metaData.software);
    LogIfNotNull(" - 撮影日時", metaData.dateTime);

    if (metaData.makerNote != nullptr)
    {
        NN_LOG(" - メーカーノート: %zu バイト\n", metaData.makerNoteSize);
    }
    if (metaData.hasEffectiveDimension)
    {
        NN_LOG(" - 実効画像サイズ: 幅 %zu px, 高さ %zu px\n", metaData.effectiveDim.width, metaData.effectiveDim.height);
    }
    LogIfNotNull(" - ユニークID", metaData.uniqueId);

    if (metaData.thumbnail != nullptr)
    {
        NN_LOG(" - サムネイル: %zu バイト\n", metaData.thumbnailSize);
    }
}

/// @brief 与えられたバイト列を解析し Exif 情報を抽出します。
void ExtractExifData(MetaData *pOutMetaData, const void *exifData, size_t exifSize)
{
    // Exif 情報解析クラスのインスタンス
    Buffer workBuf(nn::image::ExifExtractor::GetWorkBufferSize());
    nn::image::ExifExtractor extractor(workBuf.GetDataPtr(), workBuf.GetDataSize());

    /* ----------------------------------------------------------------------------------------
        解析フェーズ
     */
    // Exif データを設定します。
    extractor.SetExifData(exifData, exifSize);

    // 解析します。
    nn::image::JpegStatus jpegStatus = extractor.Analyze();
    switch (jpegStatus)
    {
    case nn::image::JpegStatus_WrongFormat:
        NN_LOG("[この Exif 情報は破損しています。]\n");
        // Exif 情報が存在しないとして処理します。
        return;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
    }

    /* ----------------------------------------------------------------------------------------
        メタデータ展開フェーズ
     */
    // 0th TIFF IFD の情報を取得します。
    pOutMetaData->maker = extractor.ExtractMaker(&pOutMetaData->makerSize);
    pOutMetaData->model = extractor.ExtractModel(&pOutMetaData->modelSize);
    pOutMetaData->hasOrientation = extractor.ExtractOrientation(&pOutMetaData->orientation);
    pOutMetaData->software = extractor.ExtractSoftware(&pOutMetaData->softwareSize);
    // 「撮影日時」の文字列は Exif 情報の仕様上、終端文字を含め 20 バイトです。
    size_t dateTimeSize;
    pOutMetaData->dateTime = extractor.ExtractDateTime(&dateTimeSize);
    NN_ASSERT(pOutMetaData->dateTime == nullptr || dateTimeSize == 20);

    // 0th Exif IFD
    pOutMetaData->makerNote = extractor.ExtractMakerNote(&pOutMetaData->makerNoteSize);
    pOutMetaData->hasEffectiveDimension = extractor.ExtractEffectiveDimension(&pOutMetaData->effectiveDim);
    // 「ユニーク ID」の文字列は Exif 情報の仕様上、終端文字を含め 33 バイトです。
    size_t uniqueIdSize;
    pOutMetaData->uniqueId = extractor.ExtractUniqueId(&uniqueIdSize);
    NN_ASSERT(pOutMetaData->uniqueId == nullptr || uniqueIdSize == 33);

    // 1st TIFF IFD
    pOutMetaData->thumbnail = extractor.ExtractThumbnail(&pOutMetaData->thumbnailSize);
}


