﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/os.h>

#include <movie/Utils.h>
#include <movie/Decoder.h>

#include "NvnRenderer.h"

#include <vector>

void VideoRendererThreadFunction(void *arg);

class VideoRenderer
{
public:
    VideoRenderer();

    ~VideoRenderer();

    movie::Status Initialize();

    movie::Status Finalize();

    movie::Status Open(movie::Decoder* videoDecoder, int32_t width, int32_t height, double videoFrameRate);

    movie::Status Start(int32_t width, int32_t height);

    movie::Status UpdateVideoInformation(int32_t width, int32_t height);

    movie::Status Render();

    void Stop();

    void Close();

    friend void VideoRendererThreadFunction(void *arg);

    void OnOutputAvailable(int index, int64_t presentationTimeUs, uint32_t flags);

    bool DoneProcessing();

    int32_t GetRenderInterval();

    bool m_Started;

    void InitNvn(int32_t width, int32_t height);
    void DestroyNvn();
    void DrawVideoNvn(int32_t width, int32_t height);
    void ResizeTextures(int32_t width, int32_t height);

protected:
    movie::Decoder* m_VideoDecoder;
    bool m_ThreadDone;
    bool m_RenderThreadCreated;
    int m_FrameSize;
    int64_t m_PresentationTimeUs;
    double m_VideoFrameRate;

    size_t m_VideoRendererThreadstacksize;
    char* m_VideoRendererThreadstack;
    nn::os::ThreadType  m_VideoRendererThreadType;
    size_t m_SampleSize;
    static const int numBuffers = 3;
    void *m_OutBuffers[numBuffers];

    struct VideoData
    {
        int index;
        int64_t presentationTimeUs;
        uint32_t flags;
    };

    std::vector<VideoData> m_VideoBuffers;
    int32_t m_RenderIntervalMs;
    size_t m_BufferSize;
    nn::os::MutexType m_VideoMutex;
    size_t m_YuvBufferSize;
    char* m_YuvBuffer;
    int32_t m_Width;
    int32_t m_Height;
    bool m_InitNvn;

private:

    nn::mem::StandardAllocator m_allocator;
    void *m_pAllocatorMemory;

    NvnWindow m_window;
    NvnDevice m_device;
    NvnQueue m_queue;

    void *m_pShaderPoolMemory;
    void *m_pVertexPoolMemory;
    void *m_pCommandMemory;
    void *m_pTextureMemory;
    void *m_pBufferMemory;
    void *m_pShaderScratchMemory;
    NvnMemoryPool m_shaderPool;
    NvnMemoryPool m_vertexPool;
    NvnMemoryPool m_commandPool;
    NvnMemoryPool m_textureMemoryPool;
    NvnMemoryPool m_bufferMemory;
    NvnMemoryPool m_shaderScratchPool;

    NvnProgram m_program;
    NvnShaderSerialize m_vertexShaderFile;
    NvnShaderSerialize m_fragmentShaderFile;

    NvnCommandBuffer m_drawCommands;
    NvnCommandBuffer m_renderTargetCommand;

    NvnTextureSamplerPool m_texturePool;
    NvnTexture m_videoTextureY;
    NvnTexture m_videoTextureUV;
    NvnSampler m_videoSampler;
    NVNtextureHandle m_videoTextureHandleY;
    NVNtextureHandle m_videoTextureHandleUV;
    unsigned m_videoTextureIdY;
    unsigned m_videoTextureIdUV;
    NvnBuffer m_videoBufferY;
    NvnBuffer m_videoBufferUV;
    NvnBuffer m_vertexDataBuffer;
    NvnBuffer m_fragmentDataBuffer;
    NvnBuffer m_squareMesh;

    NVNblendState m_blendState;
    NVNchannelMaskState m_channelMaskState;
    NVNcolorState m_colorState;
    NVNmultisampleState m_multisampleState;
    NVNpolygonState m_polygonState;
    NVNdepthStencilState m_depthStencilState;

    DataSection m_vertexDataSection;
    DataSection m_vertexControlSection;
    DataSection m_fragmentDataSection;
    DataSection m_fragmentControlSection;

    NVNcommandHandle m_commandHandleDraw;
    bool m_frameReady;
};

