﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//#define FILEDUMP 1

#include <cstddef>
#include <mm_MemoryManagement.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nnt/nntest.h>
#include <nn/audio.h>
#include <nn/mem.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#include "MediaDecoder.h"

const size_t bufferSize = nn::audio::AudioOutBuffer::SizeGranularity;
const int bufferCount = 3;

#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "MediaCodecDecoder"

namespace{

    const int FsHeapSize = 512 * 1024;
    const int bytesPerShort = sizeof(int16_t) / sizeof(int8_t);

    uint8_t              g_FsHeapBuffer[FsHeapSize];
    nn::lmem::HeapHandle g_FsHeap;

    void FsInitHeap()
    {
        g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
    }

    void* FsAllocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
    }

    void FsDeallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
            return nn::lmem::FreeToExpHeap(g_FsHeap, p);
    }

}

const char* GetSampleFormatName(nn::audio::SampleFormat format)
{
    switch (format)
    {
    case nn::audio::SampleFormat_Invalid:
        return "Invalid";
    case nn::audio::SampleFormat_PcmInt8:
        return "PcmInt8";
    case nn::audio::SampleFormat_PcmInt16:
        return "PcmInt16";
    case nn::audio::SampleFormat_PcmInt24:
        return "PcmInt24";
    case nn::audio::SampleFormat_PcmInt32:
        return "PcmInt32";
    case nn::audio::SampleFormat_PcmFloat:
        return "PcmFloat";
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

namespace // 'unnamed'
{
nn::mem::StandardAllocator g_MultimediaAllocator;
}

void MMCreateAllocator()
{
    const int allocatorSize = 512 << 20;
    static char s_MMAllocatorBuffer[allocatorSize];
    g_MultimediaAllocator.Initialize(s_MMAllocatorBuffer, sizeof(s_MMAllocatorBuffer));
}

void MMDestroyAllocator()
{
    g_MultimediaAllocator.Finalize();
}

void *MMAlloc(size_t size, size_t alignment, void*)
{
    return g_MultimediaAllocator.Allocate(size, alignment);
}

void MMFree(void *p, void*)
{
    g_MultimediaAllocator.Free(p);
}

void *MMRealloc(void *p, size_t newSize, void*)
{
    return g_MultimediaAllocator.Reallocate(p, newSize);
}

extern "C" void nninitStartup()
{
    const size_t heapSize = 128 << 20;
    const size_t mallocSize = 32 << 20;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
    NN_LOG("\ninit done: \n");
    NN_LOG("nninitStartup:%p\n", (void *)nninitStartup);
}

size_t LoadAudioOutBuffer(nn::audio::AudioOutBuffer* audioOutBuffer, nn::audio::AudioOut* audioOut, int16_t* sampleBufferStart, int16_t* sampleBufferEnd, int channels)
{
    size_t copySize = 0;
    void* tempOutBuffer = nn::audio::GetAudioOutBufferDataPointer(audioOutBuffer);

    // Copy audio sample to outbuffer
    if (sampleBufferStart + bufferSize < sampleBufferEnd)
    {
        copySize = bufferSize;
    }
    else
    {
        copySize = sampleBufferEnd - sampleBufferStart;
    }

    int16_t* outputBuffer = static_cast<int16_t*>(tempOutBuffer);

    if (channels == 1) {
        size_t copySamples = (copySize * sizeof(uint8_t) / 2) / sizeof(int16_t);

        for (size_t sample = 0; sample < copySamples; sample++)
        {
            for (int ch = 0; ch < 2; ch++)
            {
                outputBuffer[sample * 2 + ch] = sampleBufferStart[sample];
            }
        }
    }
    else
    {
        memcpy(outputBuffer, sampleBufferStart, copySize);
    }

    nn::audio::SetAudioOutBufferInfo(audioOutBuffer, tempOutBuffer, bufferSize, copySize);

    nn::audio::AppendAudioOutBuffer(audioOut, audioOutBuffer);
    return (channels==1) ? copySize / 2 : copySize;
}

// Sample rate must be 48kHz
void PlayAudio(void* sampleBuffer, size_t sampleBufferSize, int channels)
{
    if ((sampleBufferSize == 0) || (sampleBuffer == nullptr) || (channels < 1))
    {
        NN_LOG("Incorrect parameters\n");
        return;
    }

    int16_t* currentLocation = static_cast<int16_t*>(sampleBuffer);
    int16_t* endLocation = currentLocation + (sampleBufferSize / bytesPerShort);

    char bufferHeap[1024 * 64];

    nn::mem::StandardAllocator allocator(bufferHeap, sizeof(bufferHeap));

    nn::audio::AudioOut audioOut;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);

    NN_ABORT_UNLESS(
        nn::audio::OpenDefaultAudioOut(&audioOut, parameter).IsSuccess(),
        "Failed to open AudioOut."
        );
    NN_LOG("AudioOut is opened\n");

    int channelCount = nn::audio::GetAudioOutChannelCount(&audioOut);
    int audioOutSampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    NN_LOG("  Name: %s\n", nn::audio::GetAudioOutName(&audioOut));
    NN_LOG("  ChannelCount: %d\n", channelCount);
    NN_LOG("  SampleRate: %d\n", audioOutSampleRate);
    NN_LOG("  SampleFormat: %s\n", GetSampleFormatName(sampleFormat));

    NN_ASSERT(sampleFormat == nn::audio::SampleFormat_PcmInt16);

    // TODO: Add code to verify parameters are compatible with AudioOut -> Issue warnings otherwise

    // Audio buffers used by audio out
    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount]; // AudioOutBuffer objects
    void* outBuffer[bufferCount];                          // Buffers to hold audio samples
    bool doneFeeding = false;

    for (int i = 0; i < bufferCount; ++i)
    {
        if (!doneFeeding) {
            outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
            NN_ASSERT(outBuffer[i]);


            // Copy audio sample to outbuffer
            NN_LOG("Initializing AudioOutBuffer %d\n",i);

            if (currentLocation + bufferSize / bytesPerShort  >= endLocation)
            {
                doneFeeding = true;
            }
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
            nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize);
#pragma clang diagnostic pop
            currentLocation += LoadAudioOutBuffer(&audioOutBuffer[i], &audioOut, currentLocation, endLocation, channels) / bytesPerShort;
        }
    }

    NN_ABORT_UNLESS(
        nn::audio::StartAudioOut(&audioOut).IsSuccess(),
        "Failed to start playback."
        );
    NN_LOG("AudioOut is started\n");

    nn::audio::AudioOutBuffer* tempAudioOutBuffer = nullptr;
    while (!doneFeeding)
    {
        // This is non-blocking, so we sleep and continuously poll
        tempAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);

        if (tempAudioOutBuffer)
        {
            if (currentLocation + bufferSize / bytesPerShort >= endLocation)
            {
                doneFeeding = true;
            }

            currentLocation += LoadAudioOutBuffer(tempAudioOutBuffer, &audioOut, currentLocation, endLocation, channels) / bytesPerShort;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
    }

    // Wait for all the buffers to be removed
    for (int i = 0; i < bufferCount; ++i)
    {
        NN_LOG("Done loading, waiting for playback to end\n");
        while (nn::audio::ContainsAudioOutBuffer(&audioOut, &audioOutBuffer[i]))
        {
            tempAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(100));
        }
    }

    NN_LOG("Stop audio playback\n");

    // 再生を停止します。
    nn::audio::StopAudioOut(&audioOut);
    NN_LOG("AudioOut is closed\n");
    nn::audio::CloseAudioOut(&audioOut);
}


TEST(MediaCodecAudioPlayer, AudioPlayback)
{
    //using namespace android;
    NN_LOG("\n--- Testing %d bit build ---\n\n", sizeof(size_t) * 8);

    int64_t fileLength;
    void* fileBuffer = nullptr;
    MediaDecoder* decoder = nullptr;


    MMCreateAllocator();
    NN_SDK_LOG("Calling nv::mm::SetAllocator\n");
    nv::mm::SetAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    NN_SDK_LOG("Calling nv::SetGraphicsAllocator\n");
    nv::SetGraphicsAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    nv::SetGraphicsServiceName("nvdrv:t");
    NN_SDK_LOG("Calling nv::SetGraphicsDevtoolsAllocator\n");
    nv::SetGraphicsDevtoolsAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    const int mmFirmwareMemorySize = 8 << 20;
    NN_ALIGNAS(4096) static char s_mmFirmwareMemory[mmFirmwareMemorySize];
    NN_SDK_LOG("Calling nv::InitializeGraphics\n");
    nv::InitializeGraphics(s_mmFirmwareMemory, sizeof(s_mmFirmwareMemory));
    int argc =  nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    if (argc <= 1)
    {
        NN_LOG("Invalid arguments\n");
        NN_LOG("Argument count = %d\n", argc);
        return;
    }

    char* inputFileName = inputFileName = argv[1];
    NN_LOG("Opening input file %s\n", inputFileName);
    nn::Result resultHostMount = nn::fs::MountHostRoot();
    if (resultHostMount.IsFailure())
    {
        NN_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
            resultHostMount.GetModule(),
            resultHostMount.GetDescription());
        FAIL() << "nn::fs::Houst root mount failure\n";
        return;
    }

    /* Check whether media file exists */
    nn::fs::FileHandle fileHandle;
    fileHandle.handle = NULL;
    nn::Result result = nn::fs::OpenFile(&fileHandle, inputFileName, nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_LOG("\n Failed to open %s \n \n Exiting MediaPlayerSample Test \n", inputFileName);
        FAIL() << "Failed to open test file\n";
        return;
    }
    else
    {
        // Read the file into a buffer
        nn::fs::GetFileSize(&fileLength, fileHandle);
        NN_LOG("Allocating memory buffer for file\n");

        fileBuffer = new char[fileLength];

        NN_LOG("Reading input file %s\n", inputFileName);
        nn::fs::ReadFile(fileHandle, 0, fileBuffer, fileLength);
        nn::fs::CloseFile(fileHandle);

        NN_LOG("Decoding file\n");

        int sampleRate = -1;
        int bitsPerSample = -1;
        int channels = -1;
        int endian = -1;

        decoder = MediaDecoder::Create();
        int decodedLength = decoder->DecodeAudioData(fileBuffer, fileLength, nullptr, 0, &sampleRate, &bitsPerSample, &channels, &endian);
        NN_LOG("Decoded length = %d\n", decodedLength);

        if(decodedLength <= 0)
        {
            FAIL() << "Decode fail\n";
            return;
        }

        NN_LOG("Copying decoded bits to buffer\n");
        void* decodedBuffer = new int8_t[decodedLength];
        decodedLength = decoder->DecodeAudioData(fileBuffer, fileLength, decodedBuffer, decodedLength, &sampleRate, &bitsPerSample, &channels, &endian);

#ifdef FILEDUMP
        /* Write to output to disk */
        fileHandle.handle = NULL;
        char* outputFileName = "c:\\temp\\AudioPlayer_out.wav";
        nn::fs::DeleteFile(outputFileName);
        nn::fs::CreateFile(outputFileName, 0);
        result = nn::fs::OpenFile(&fileHandle, outputFileName, nn::fs::OpenMode_AllowAppend | nn::fs::OpenMode_Write);
        if (result.IsFailure())
        {
            NN_LOG("\n Failed to open %s for writing\nSkipping write\n", outputFileName);
        }
        else
        {
            // Write the decoded data to file
            NN_LOG("Writing output file %s\n", outputFileName);
            nn::fs::WriteFile(fileHandle, 0, decodedBuffer, decodedLength, nn::fs::WriteOption());
            nn::fs::FlushFile(fileHandle);
            nn::fs::CloseFile(fileHandle);
        }
#endif

        NN_LOG("Playing audio %d bytes with %d channels\n", decodedLength, channels);

        PlayAudio(decodedBuffer, decodedLength, channels);

        delete(decoder);
        delete(static_cast<char*>(fileBuffer));
        delete(static_cast<char*>(decodedBuffer));

        MMDestroyAllocator();
        nn::lmem::DestroyExpHeap(g_FsHeap);
    }
}
