﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <sys/types.h>

#include <stdint.h>
#include <string>

#include "NXMSESource.h"

//#define VERBOSE_LOGGING

movie::Status NXMSESource::AddSourceBuffer(movie::DataSource* dataSource)
{
    m_dataSourceList.push_back(dataSource);
    return movie::Status::Status_Success;
}

ssize_t NXMSESource::GetSourceBufferCount()
{
    return m_dataSourceList.size();
}

movie::DataSource* NXMSESource::GetSourceBuffer(size_t bufferIndex)
{
    return m_dataSourceList[bufferIndex];
}

movie::Status NXMSESource::RemoveSourceBuffer(size_t bufferIndex)
{
    m_dataSourceList.erase(m_dataSourceList.begin() + bufferIndex);
    return movie::Status::Status_Success;
}

movie::Status NXMSESource::AdvanceSourceBuffer(size_t bufferIndex)
{
    NXMSESourceBuffer* bufferPtr = static_cast<NXMSESourceBuffer*>(m_dataSourceList[bufferIndex]);
    return bufferPtr->advanceDataSegment();
}

bool NXMSESource::IsEndOfStreamReached()
{
    bool isEOSReached = true;

    for (int i = 0; i < m_dataSourceList.size(); ++i)
    {
        isEOSReached = isEOSReached && static_cast<NXMSESourceBuffer*>(m_dataSourceList[i])->getEndOfData();
    }

    return isEOSReached;
}

NXMSESourceBuffer::NXMSESourceBuffer()
    :
    mOffset(0),
    mLength(-1)       ///< The file's size, negative if unknown
{
    mMutex = new nn::os::Mutex(true);

    mSegmentBuffers = nullptr;       // head of list
    mCurrentInitializationSegment = nullptr;
    mCurrentDataSegment = nullptr;
    mLastAppendedInitSegment = nullptr;
    mLastAppendedSegment = nullptr;
    mEOFReached = false;
    mSegmentIndex = 0;
    mTimeOffsetUs = 0;

    movie::PlayerExtractor::Create(&mExtractor);
}

// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------

NXMSESourceBuffer::~NXMSESourceBuffer()
{
    DataSegment* iter = mSegmentBuffers;
    DataSegment* temp = nullptr;
    while (iter != nullptr)
    {
        temp = iter->next;
        delete iter;
        iter = temp;
    }

    movie::PlayerExtractor::Destroy(mExtractor);
    mExtractor = nullptr;

    delete mMutex;
}

// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------

bool NXMSESourceBuffer::initCheck()
{
    return (mOffset >= 0 && mLength >= 0) ? true : false;
}

// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------

ssize_t NXMSESourceBuffer::readAt(off64_t offset, void *data, size_t size)
{
    mMutex->Lock();
#ifdef VERBOSE_LOGGING
    NN_SDK_LOG("NXMSESourceBuffer::readAt offset=%d, size=%d, %d, %d\n", offset, size,  mLength, mOffset);
#endif /* VERBOSE_LOGGING */

    // seeking
    bool seeking = false;
    if (mOffset != offset)
    {
        if (offset + size >= mLength)
        {
            NN_SDK_LOG("MSESource: Error: bailing out: offset: %d, length:%d \n", offset, mLength);
            //mMutex->Unlock();
            //return 0;
        }

        seeking = true;
        mOffset = offset;
    }

    if (seeking)
    {
        DataSegment* iter = mSegmentBuffers;
        bool found = false;
        while (!found && iter != nullptr)
        {
            if (mOffset >= iter->mOffset
                && mOffset < (iter->mOffset + iter->mLength - 1))
            {
                mCurrentDataSegment = iter;
                found = true;
#ifdef VERBOSE_LOGGING
                NN_SDK_LOG("MSESource: Seek to segment: index %d\n", mCurrentDataSegment->mSegmentIndex);
#endif /* VERBOSE_LOGGING */
            }
            else if (mOffset >= iter->mOffset && iter->type == MSEDataType::END_OF_FILE_SEGMENT)
            {
                mCurrentDataSegment = iter;
                found = true;
#ifdef VERBOSE_LOGGING
                NN_SDK_LOG("MSESource: Seek to segment: index %d\n", mCurrentDataSegment->mSegmentIndex);
#endif /* VERBOSE_LOGGING */
            }
            iter = iter->next;
        }

        if (mCurrentDataSegment == nullptr)
        {
            NN_SDK_LOG("MSESource: Error: Seek to invalid segment: bad offset: %d, a valid range: 0 - %d \n", offset, mLength);
            mMutex->Unlock();
            return 0;
        }
        else
        {
            if (mCurrentDataSegment->type == MSEDataType::END_OF_FILE_SEGMENT)
            {
                mEOFReached = true;
                mCurrentDataSegment = nullptr;
            }

        }
    }
    else
    {
        if (mOffset >= mCurrentDataSegment->mOffset + mCurrentDataSegment->mLength)
        {
#ifdef VERBOSE_LOGGING
            NN_SDK_LOG("MSESource: mOffset: %d, mCurrentDataSegment->mOffset: %d, mCurrentDataSegment->mLength: %d\n", mOffset, mCurrentDataSegment->mOffset, mCurrentDataSegment->mLength);
#endif /* VERBOSE_LOGGING */
            if (mCurrentDataSegment->next != nullptr)
            {
                mCurrentDataSegment = mCurrentDataSegment->next;
#ifdef VERBOSE_LOGGING
                NN_SDK_LOG("MSESource: Go to next segment: index %d\n", mCurrentDataSegment->mSegmentIndex);
#endif /* VERBOSE_LOGGING */
            }
            else if (mCurrentInitializationSegment->next != nullptr)
            {
                mCurrentDataSegment = mCurrentInitializationSegment->nextInitialization;
#ifdef VERBOSE_LOGGING
                NN_SDK_LOG("MSESource: Go to next initializationsegment: index %d\n", mCurrentDataSegment->mSegmentIndex);
#endif /* VERBOSE_LOGGING */
            }
            else
            {
#ifdef VERBOSE_LOGGING
                NN_SDK_LOG("MSESource: We have reached the last segment\n");
#endif /* VERBOSE_LOGGING */
                //We have reached the last segment
                if (mCurrentDataSegment->type == MSEDataType::END_OF_FILE_SEGMENT)
                {
                    mEOFReached = true;
                }
                mCurrentDataSegment = nullptr;
            }
        }
    }

    off64_t actual = 0;
    off64_t segmentOffset = 0;
    if (mCurrentDataSegment && mCurrentDataSegment->mData && mCurrentDataSegment->mLength > 0)
    {
        segmentOffset = offset - mCurrentDataSegment->mDataBegin;
        actual = size;
        if ((actual + segmentOffset) > mCurrentDataSegment->mLength)
        {
            memcpy(data, mCurrentDataSegment->mData + segmentOffset, mCurrentDataSegment->mLength - segmentOffset);
            off64_t bytesRemaining = size - mCurrentDataSegment->mLength - segmentOffset;
            while (mCurrentDataSegment->next != NULL && bytesRemaining > 0)
            {
                mCurrentDataSegment = mCurrentDataSegment->next;
                if (bytesRemaining < mCurrentDataSegment->mLength)
                {
                    memcpy((char*)data + (size - bytesRemaining), mCurrentDataSegment->mData, bytesRemaining);
                    bytesRemaining = 0;
                }
                else
                {
                    memcpy((char*)data + (size - bytesRemaining), mCurrentDataSegment->mData, mCurrentDataSegment->mLength);
                    bytesRemaining -= mCurrentDataSegment->mLength;
                }
            }

            if (bytesRemaining > 0)
            {
                actual = size - bytesRemaining;
            }
        }
        else
        {
            memcpy(data, mCurrentDataSegment->mData + segmentOffset, actual);
        }
#ifdef VERBOSE_LOGGING
        NN_SDK_LOG("NXMSESourceBuffer::readAt segmentOffset=%lld, offset=%d, size=%d, actual=%d, %d, %d, %lld, segmentIndex: %d\n", segmentOffset, offset, size, actual, mLength, mOffset, mCurrentDataSegment->mDataBegin, mCurrentDataSegment->mSegmentIndex);
#endif
    }

    mMutex->Unlock();
    mOffset += actual;
#ifdef VERBOSE_LOGGING
    NN_SDK_LOG("NXMSESourceBuffer::readAt segmentOffset=%d, offset=%d, size=%d, actual=%d, %d, %d\n", segmentOffset, offset, size, actual, mLength, mOffset);
#endif /* VERBOSE_LOGGING */
    return actual;
}//NOLINT(impl/function_size)

off64_t NXMSESourceBuffer::getSize()
{
    return (off64_t)mLength;
}

////////////////////////////////////////////////////////////////////////////////
// List Management

int64_t NXMSESourceBuffer::appendData(void* data, size_t size, MSEDataType type)
{
    mMutex->Lock();
    // Check input parameters: data, size, type
    if ((type == MSEDataType::DATA_SEGMENT || type == MSEDataType::END_OF_FILE_SEGMENT)
        && mLastAppendedInitSegment == nullptr)
    {
        //return ERROR because you can't append data segments without an initialization segment
        mMutex->Unlock();
        return -1;
    }

    if (mLastAppendedSegment && mLastAppendedSegment->type == MSEDataType::END_OF_FILE_SEGMENT)
    {
        //return ERROR because you can't append after signaling end of file segment
        mMutex->Unlock();
        return -1;
    }

    if (type == MSEDataType::DATA_SEGMENT && size == 0)
    {
        //Return ERROR because you can't append a data segment of size 0
        mMutex->Unlock();
        return -1;
    }

    if (mLength < 0)
    {
        mLength = 0;
    }

    DataSegment* segment = new DataSegment(size);
    segment->mOffset = (mLastAppendedSegment != nullptr) ? mLastAppendedSegment->mOffset + mLastAppendedSegment->mLength : 0;
    segment->mLength = size;
    segment->type = type;
    segment->mCachedDurationUs = -1;
    segment->mSegmentIndex = mSegmentIndex;
    segment->mDataBegin = mLength;
    segment->mDataEnd = segment->mDataBegin + size;
    segment->mTimeOffsetUs = mTimeOffsetUs;
    segment->next = NULL;
    mSegmentIndex++;
    if (size > 0)
    {
        memcpy(segment->mData, data, size);
    }

    mLength += size;

    {
        switch (type)
        {
        case MSEDataType::INITIALIZATION_SEGMENT:
        {
            // Signify the next data segment is different in some other way
            // (different bit rate, video dimensions, etc.)

            // If first initialization segment, append to the current buffer
            // If not first initialization segment, add to buffer list.
            // Then add to initialization segment list
            if (mSegmentBuffers == nullptr)
            {
                mSegmentBuffers = segment;
                mCurrentInitializationSegment = segment;
                mCurrentDataSegment = segment;
            }
            else
            {
                mLastAppendedInitSegment->nextInitialization = segment;
                mLastAppendedSegment->next = segment;
            }

            mLastAppendedSegment = segment;
            mLastAppendedInitSegment = segment;

            segment->mStartTimeUs = -1;
            bool eos;

            if (mExtractor)
            {
                movie::ExtractorConfig config;
                //config.isMSE = true;
                movie::PlayerExtractor::Destroy(mExtractor);
                mExtractor = nullptr;
                movie::PlayerExtractor::Create(&mExtractor, &config);
            }



            movie::Status result = mExtractor->SetDataSource(data, size);

            NN_SDK_LOG("MSESource: InitSegment: SetDataSource: %d\n", result);
            //This should only succeed if its a single file broken up into internal segments
            //if (result == movie::Status_Success)
            {
                int trackCount;
                result = mExtractor->CountTracks(&trackCount);
                NN_SDK_LOG("MSESource: InitSegment: CountTracks: %d, %d\n", trackCount, result);

                result = mExtractor->GetCachedDuration(&(segment->mCachedDurationUs), &eos);
                NN_SDK_LOG("MSESource: InitSegment: GetCachedDuration: %lld\n", segment->mCachedDurationUs);

                result = mExtractor->GetSampleTime((&(segment->mStartTimeUs)));
                NN_SDK_LOG("MSESource: InitSegment: GetSampleTime: %lld\n", segment->mStartTimeUs);
            }

        }
        break;
        case MSEDataType::DATA_SEGMENT:
        {
            if (mLastAppendedSegment != nullptr)
            {
                mLastAppendedSegment = segment;
            }

            // Data Segment just appends to the current buffer
            bool eos;

            // Create a temp buffer for the data segment in order to caculate the duration
            // of the segment and then store the duration in DataSegment::mCachedDurationUs
            char* temp = nullptr;
            int dataSourceSize = mLastAppendedInitSegment->mLength + size;
            temp = new char[mLastAppendedInitSegment->mLength + size];
            memcpy(temp, (void*)(mLastAppendedInitSegment->mData), mLastAppendedInitSegment->mLength);
            memcpy(temp + mLastAppendedInitSegment->mLength, data, size);

            // Must re-initialize mExtractor because Extractor can ONLY set one DataSource.
            if (mExtractor)
            {
                movie::ExtractorConfig config;
                //config.isMSE = true;
                movie::PlayerExtractor::Destroy(mExtractor);
                mExtractor = nullptr;
                movie::PlayerExtractor::Create(&mExtractor, &config);
            }

            movie::Status result = mExtractor->SetDataSource(temp, dataSourceSize);


            if (result == movie::Status_Success)
            {
                int trackCount;
                result = mExtractor->CountTracks(&trackCount);
                NN_SDK_LOG("MSESource: CountTracks: %d\n", trackCount);
                if (result != movie::Status_Success)
                {
#ifdef VERBOSE_LOGGING
                    NN_SDK_LOG("MSESource: Failed to get track count\n");
#endif /* VERBOSE_LOGGING */
                }

                result = mExtractor->SelectTrack(0);

                result = mExtractor->GetFragmentIndex(&(segment->mSegmentIndex));
                NN_SDK_LOG("MSESource: fragmentIndex: %d\n", segment->mSegmentIndex);
                if (result != movie::Status_Success)
                {
#ifdef VERBOSE_LOGGING
                    NN_SDK_LOG("MSESource: Failed to get fragment index\n");
#endif /* VERBOSE_LOGGING */
                }

                if (segment->mSegmentIndex < 0)
                {
                    segment->mSegmentIndex = mSegmentIndex;
                }

                result = mExtractor->GetCachedDuration(&(segment->mCachedDurationUs), &eos);
                NN_SDK_LOG("MSESource: GetCachedDuration: %lld\n", segment->mCachedDurationUs);
                if (result != movie::Status_Success)
                {
#ifdef VERBOSE_LOGGING
                    NN_SDK_LOG("MSESource: Failed to get cached duration\n");
#endif /* VERBOSE_LOGGING */
                }

                result = mExtractor->GetSampleTime((&(segment->mStartTimeUs)));
                NN_SDK_LOG("MSESource: GetSampleTime: %lld\n", segment->mStartTimeUs);
                if (result != movie::Status_Success)
                {
#ifdef VERBOSE_LOGGING
                    NN_SDK_LOG("MSESource: Failed to get start time\n");
#endif /* VERBOSE_LOGGING */
                }

                DataSegment* iter = mSegmentBuffers;
                DataSegment* prev = NULL;
                DataSegment* next = NULL;

                bool found = false;
                while ((!found) && iter != nullptr)
                {
                    if (segment->mSegmentIndex >= iter->mSegmentIndex)
                    {
                        if (iter->next)
                        {
                            if (iter->next->mSegmentIndex > segment->mSegmentIndex)
                            {
                                //we found two segments to wedge between
                                prev = iter;
                                next = iter->next;
                                iter->next = segment;
                                segment->next = next;
                                found = true;
                            }
                        }
                        else
                        {
                            //append the new last segment
                            iter->next = segment;
                            segment->next = NULL;
                            found = true;
                        }
                    }
                    iter = iter->next;
                }

            }
            else
            {
#ifdef VERBOSE_LOGGING
                NN_SDK_LOG("MSESource: Failed to set datasource\n");
#endif /* VERBOSE_LOGGING */
            }

            if (segment->mCachedDurationUs >= 0)
            {
                mTimeOffsetUs += segment->mCachedDurationUs;
            }
            delete[] temp;
        }
        break;
        case MSEDataType::END_OF_FILE_SEGMENT:
        {
            // Reached the end of the file and not just the end of the data buffer
            if (mLastAppendedSegment != nullptr)
            {
                mLastAppendedSegment = segment;
            }
            DataSegment* iter = mSegmentBuffers;
            while (iter->next != NULL)
            {
                iter = iter->next;
            }
            iter->next = segment;
            segment->next = NULL;

            segment->mStartTimeUs = -1;
            // Data Segment just appends to the current buffer
        }
        break;

        default:
        {
            break;
        }

        }
        DataSegment* iter = mSegmentBuffers;
        int64_t offsetTotal = 0;
        while (iter != NULL)
        {
            iter->mOffset = offsetTotal;
            iter->mDataBegin = iter->mOffset;
            iter->mDataEnd = iter->mOffset + iter->mLength;
            offsetTotal += iter->mLength;
            iter = iter->next;
        }
        mMutex->Unlock();
        return segment->mSegmentIndex;
    }
    mMutex->Unlock();
    return segment->mSegmentIndex; // There is an error
}//NOLINT(impl/function_size)

int64_t NXMSESourceBuffer::removeDataSegment(int64_t dataIndex)
{
    mMutex->Lock();
    DataSegment* iter = mSegmentBuffers;
    DataSegment* prevIter = NULL;
    while (iter != nullptr)
    {
        if (iter->mSegmentIndex == dataIndex)
        {
            if (prevIter)
                prevIter->next = iter->next;
            if (mSegmentBuffers == iter)
                mSegmentBuffers = iter->next;
            delete iter;
            mMutex->Unlock();
            return dataIndex;
        }
        prevIter = iter;
        iter = iter->next;
    }
    mMutex->Unlock();
    return -1;
}

movie::Status NXMSESourceBuffer::advanceDataSegment()
{
    if (mCurrentDataSegment)
    {
        char buff;
        readAt(mCurrentDataSegment->mOffset + mCurrentDataSegment->mLength, &buff, 1);
        return movie::Status::Status_Success;
    }
    return movie::Status::Status_UnknownError;
}

char* NXMSESourceBuffer::newBuffer(void* data, size_t size)
{
    char *buffer = allocateBuffer(size);
    if (buffer == nullptr)
    {
        return nullptr;
    }

    memcpy(buffer, data, size);

    return buffer;
}

char* NXMSESourceBuffer::allocateBuffer(size_t size)
{
    return new char[size];
}

// End List Management
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Segment Changes
DataSegment NXMSESourceBuffer::getNextInitializationSegment()
{
    // If all the initialization segments are stored in a list and there is
    // a pointer to the current Initialization Segment.
    if (mCurrentInitializationSegment && mCurrentInitializationSegment->nextInitialization)
    {
        return *mCurrentInitializationSegment->nextInitialization;
    }
    //Throw an error
    return DataSegment(0);
}

DataSegment NXMSESourceBuffer::getCurrentInitializationSegment()
{
    // If all the initialization segments are stored in a list and there is
    // a pointer to the current Initialization Segment.
    if (mCurrentInitializationSegment)
    {
        return *mCurrentInitializationSegment;
    }
    //Throw an error
    return DataSegment(0);
}

DataSegment NXMSESourceBuffer::getCurrentDataSegment()
{
    // Keep track of the current data segment based on what was the last read
    if (mCurrentDataSegment)
    {
        return *mCurrentDataSegment;
    }
    //Throw an error
    return DataSegment(0);
}

// Skip Current Segment
int NXMSESourceBuffer::cancelCurrentSegment()
{
    if (mCurrentDataSegment == nullptr)
    {
        // Error: There is no data segment to cancel
        return 1;
    }
    else if (mCurrentDataSegment->next == nullptr)
    {
        // Cancelled the last segment, signal end of stream
        return 1;
    }

    // Skip to the next segment
    mCurrentDataSegment = mCurrentDataSegment->next;

    // The new current segment is an initialization segment
    if (mCurrentInitializationSegment && mCurrentInitializationSegment->nextInitialization == mCurrentDataSegment)
    {
        mCurrentInitializationSegment = mCurrentDataSegment;
    }

    return 0;
}

// End Segment Changes
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Timing Management

int64_t NXMSESourceBuffer::getSegmentDuration(int64_t dataIndex)
{
    DataSegment* iter = mSegmentBuffers;
    while (iter != nullptr)
    {
        if (iter->mSegmentIndex == dataIndex)
        {
            return iter->mCachedDurationUs;
        }
        iter = iter->next;
    }
    return -1;
}

int64_t NXMSESourceBuffer::setSegmentStartTime(int64_t dataIndex, int64_t startTime, bool addTimeToFollowingSegments)
{
    DataSegment* iter = mSegmentBuffers;
    bool continueToAddTime = false;
    int64_t prevTime = 0;
    while (iter != nullptr)
    {
        if (iter->mSegmentIndex == dataIndex)
        {
            iter->mTimeOffsetUs = startTime;
            if (!addTimeToFollowingSegments)
            {
                return 0;
            }
            else
            {
                continueToAddTime = true;
                prevTime = iter->mTimeOffsetUs + iter->mCachedDurationUs;
            }
        }
        else if (continueToAddTime)
        {
            iter->mTimeOffsetUs = prevTime;
            prevTime = iter->mTimeOffsetUs + iter->mCachedDurationUs;
        }
        iter = iter->next;
    }
    return -1.0;
}

int64_t NXMSESourceBuffer::getSegmentStartTime(int64_t dataIndex)
{
    DataSegment* iter = mSegmentBuffers;
    while (iter != nullptr)
    {
        if (iter->mSegmentIndex == dataIndex)
        {
            return iter->mTimeOffsetUs;
        }
        iter = iter->next;
    }
    return -1.0;
}

int64_t NXMSESourceBuffer::getSegmentEndTime(int64_t dataIndex)
{
    DataSegment* iter = mSegmentBuffers;
    while (iter != nullptr)
    {
        if (iter->mSegmentIndex == dataIndex)
        {
            return iter->mTimeOffsetUs + iter->mCachedDurationUs;
        }
        iter = iter->next;
    }
    return -1.0;
}


// End Timing Management
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
// Query Management

bool NXMSESourceBuffer::getEndOfData()
{
    return mEOFReached;
}

int NXMSESourceBuffer::getTotalDataBuffered()
{
    return mLength;
}

bool NXMSESourceBuffer::isTypeSupported(const char* in_mimetype, const char* in_codec, const char* in_system, const char* in_url)
{
    return true;
}

// End Query Management
////////////////////////////////////////////////////////////////////////////////

// Static property management functions
int64_t NXMSESourceBuffer::getSegmentDuration(void* data, size_t size)
{
    movie::PlayerExtractor* extractor;
    movie::PlayerExtractor::Create(&extractor);

    int64_t durationUs = 0;
    bool eos;
    extractor->SetDataSource(data, size);
    extractor->GetCachedDuration(&durationUs, &eos);

    movie::PlayerExtractor::Destroy(extractor);
    extractor = nullptr;
    return durationUs;
}

MSESegmentStatus NXMSESourceBuffer::getSegmentStatus(void* data, size_t size)
{
    movie::PlayerExtractor* extractor;
    movie::PlayerExtractor::Create(&extractor);

    MSESegmentStatus segmentStatus = INVALID_SEGMENT;

    int64_t durationUs = 0;
    bool eos;
    movie::Status extractorStatus = extractor->SetDataSource(data, size);

    if (extractorStatus == movie::Status_Success)
    {
        extractorStatus = extractor->GetCachedDuration(&durationUs, &eos);
        if (extractorStatus == movie::Status_Success)
        {
            if (durationUs == 0)
            {
                segmentStatus = MSE_INITIALIZATION_SEGMENT;
            }
            else if (durationUs > 0)
            {
                segmentStatus = MEDIA_SEGMENT;
            }
        }
        else
        {
            int trackCount = 0;
            extractorStatus = extractor->CountTracks(&trackCount);
            if (!trackCount)
            {
                segmentStatus = MSE_INITIALIZATION_SEGMENT_INCOMPLETE;
            }
        }
    }
    else
    {
        segmentStatus = MEDIA_SEGMENT_INCOMPLETE;
    }
    movie::PlayerExtractor::Destroy(extractor);
    extractor = nullptr;


    return INVALID_SEGMENT;
}
