﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_log.h>
#include "BlitVicSession.h"

VicSession::VicSession(NvRmDeviceHandle device, std::uint32_t width, std::uint32_t height) NN_NOEXCEPT
    : m_Device(device)
    , m_pSession(nullptr)
{
    InitializeTargetSurface(width, height);

    NvDdkVicCreateSession(device, nullptr, &m_pSession);
}

VicSession::~VicSession() NN_NOEXCEPT
{
    NvRmSurfaceFree(&m_Target);

    if ( m_pSession != nullptr )
    {
        NvDdkVicFreeSession(m_pSession);
    }
}

void VicSession::InitializeTargetSurface(std::uint32_t width, std::uint32_t height) NN_NOEXCEPT
{
    NN_LOG("VicSession::InitializeTargetSurface: width = %u, height = %u\n", width, height);
    NvRmSurfaceSetup(&m_Target, width, height, NvColorFormat_A8R8G8B8, nullptr);

    std::uint32_t alignment = NvRmSurfaceComputeAlignment(m_Device, &m_Target);
    std::uint32_t size = NvRmSurfaceComputeSize(&m_Target);

    NVRM_DEFINE_MEM_HANDLE_ATTR(memAttributes);
    NVRM_MEM_HANDLE_SET_ATTR(memAttributes,
                             alignment,
                             NvOsMemAttribute_WriteCombined,
                             size,
                             NVRM_MEM_TAG_DDK2D_MISC);
    NVRM_MEM_HANDLE_SET_KIND_ATTR(memAttributes, m_Target.Kind);

    NvRmMemHandleAllocAttr(m_Device, &memAttributes, &m_Target.hMem);
}

NvError VicSession::Execute(NvRmSurface* pSurfaces, int length, const NvDdkVicFloatColor& background) NN_NOEXCEPT
{
    NvDdkVicConfigParameters config;
    memset(&config, 0, sizeof(NvDdkVicConfigParameters));
    config.BackgroundColor = background;

    // setting up VIC to use a particular output surface
    NvDdkVicConfigureTargetSurface(m_pSession, &config, &m_Target, 1, nullptr);

    // setup input surfaces
    for ( int i = 0; i < length; ++i )
    {
        // map each slot to a particular surface
        // Note: Last two parameters are rectangles, but the coordinate systems aren't documented...
        //       See 16.3.5.1 for details.
        NvDdkVicConfigureSourceSurface(m_pSession, &config, i, pSurfaces + i, 1, nullptr, nullptr);
    }

    NvDdkVicConfigure(m_pSession, &config);

    NvDdkVicExecParameters parameters;
    memset(&parameters, 0, sizeof(parameters));
    parameters.OutputSurface = &m_Target;

    for ( int i = 0; i < length; ++i )
    {
        // other surface slots are not needed for RGB surfaces
        parameters.InputSurfaces[i][0] = pSurfaces + i;
    }

    NvRmFence fence;
    NvError error = NvDdkVicExecute(m_pSession, &parameters, nullptr, 0, &fence);

    if ( error == NvSuccess )
    {
        NvRmFenceWait(m_Device, &fence, NV_WAIT_INFINITE);
    }

    return error;
}

NvRmSurface* VicSession::GetOutput() NN_NOEXCEPT
{
    return &m_Target;
}
