﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/mii/mii_PrivateCommon.h>
#include <nn/mii/detail/mii_Debug.h>
#include <nn/mii/detail/mii_UtilityImpl.h>
#include <nn/mii/mii_Database.h>
#include <nn/mii/detail/mii_DetailCommon.h>
#include <nn/mii/detail/mii_StoreDataRaw.h>
#include <nn/mii/detail/mii_DatabaseFileManager.h>
#include <nn/mii/mii_StoreDataAccessor.h>
#include <nn/mii/mii_StoreDataContext.h>
#include <nn/mii/mii_Result.h>
#include <nn/mii/mii_PrivateResult.h>
//#include "Utility.h"
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include "utility/utility_ScopedPrivateDatabase.h"
#include "utility/utility_ScopedDatabaseImpl.h"
#include "utility/utility_DatabaseUtility.h"
#include "utility/utility_FileUtility.h"

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

namespace
{
    void SetMiiDatabaseTestModeEnabled(bool isEnabled) NN_NOEXCEPT
    {
#if defined (NN_BUILD_CONFIG_OS_HORIZON)
        nn::settings::fwdbg::SetSettingsItemValue("mii", "is_db_test_mode_enabled", &isEnabled, sizeof(isEnabled));
#else
        // do nothing
        NN_UNUSED(isEnabled);
#endif
    }

    // nn::mii::PrivateDatabase::Import(),DestroyFile() などを扱うのでテストモード有効にしてテスト実行
    class PrivateDatabaseForUserTest : public ::testing::Test
    {
    protected:
        static void SetUpTestCase()
        {
            SetMiiDatabaseTestModeEnabled(true);
        }
        static void TearDownTestCase()
        {

        }
    };
}

// mii:u OK, mii:e OK
TEST_F(PrivateDatabaseForUserTest, IsUpdated)
{
    utility::ScopedPrivateDatabase database;
    database.Get().Format();
    EXPECT_FALSE(database.Get().IsUpdated(nn::mii::SourceFlag_Default));
}
// mii:u OK, mii:e OK
TEST_F(PrivateDatabaseForUserTest, IsFullUpdated)
{
    utility::ScopedPrivateDatabase database;
    database.Get().Format();
    ASSERT_FALSE(database.Get().IsFullDatabase());
}
// mii:u OK, mii:e OK
TEST_F(PrivateDatabaseForUserTest, GetCount)
{
    utility::ScopedPrivateDatabase database;
    database.Get().Format();

    ASSERT_EQ(database.Get().GetCount(nn::mii::SourceFlag_Default),nn::mii::DefaultMiiCount);
    ASSERT_EQ(database.Get().GetCount(nn::mii::SourceFlag_Database),0);

    {
        // Mii追加(Export, Import を使用)
        nn::mii::detail::ImportFile file;
        nn::mii::detail::DatabaseFile databaseFile;
        databaseFile.Initialize();
        for (int i = 0; i < 2; ++i)
        {
            nn::mii::StoreData storeData;
            nn::mii::StoreDataContext context;
            nn::mii::StoreDataAccessor(&storeData).BuildDefault(&context, i);
            databaseFile.AddOrReplace(nn::mii::detail::StoreDataRaw::Cast(storeData));
        }
        databaseFile.UpdateCrc();
        databaseFile.Export(&file);
        NNT_ASSERT_RESULT_SUCCESS(database.Get().Import(&file, sizeof(file)));
    }

    ASSERT_EQ(database.Get().GetCount(nn::mii::SourceFlag_Database),2);
}
// mii:u NG, mii:e OK
TEST_F(PrivateDatabaseForUserTest, GetStoreDataElement)
{
    utility::ScopedPrivateDatabase database;

    database.Get().Format();

    const int ElementCount = nn::mii::DatabaseMiiCount + nn::mii::DefaultMiiCount;
    nn::mii::StoreDataElement elements[ElementCount];

    int count;
    NNT_ASSERT_RESULT_FAILURE(
        nn::mii::ResultPermissionDenied,
        database.Get().Get(&count,elements,NN_DETAIL_MII_ARRAY_NUM(elements),nn::mii::SourceFlag_All));
}

// mii:u NG, mii:e OK
TEST_F(PrivateDatabaseForUserTest, GetStoreData)
{
    utility::ScopedPrivateDatabase database;

    database.Get().Format();

    const int ElementCount = nn::mii::DatabaseMiiCount + nn::mii::DefaultMiiCount;
    nn::mii::StoreData elements[ElementCount];

    int count;
    NNT_ASSERT_RESULT_FAILURE(
        nn::mii::ResultPermissionDenied,
        database.Get().Get(&count,elements,NN_DETAIL_MII_ARRAY_NUM(elements),nn::mii::SourceFlag_All));
}

// mii:u NG, mii:e OK
TEST_F(PrivateDatabaseForUserTest, UpdateLatest)
{
    utility::ScopedPrivateDatabase database;
    database.Get().Format();

    /// 該当Miiを取得
    nn::mii::StoreData storeDataOld;
    {
        int count;
        NNT_ASSERT_RESULT_FAILURE(
            nn::mii::ResultPermissionDenied,
            database.Get().Get(&count, &storeDataOld, 1, nn::mii::SourceFlag_Database));
    }

    nn::mii::StoreData storeDataNew = storeDataOld;
    nn::mii::detail::StoreDataRaw::Cast(storeDataNew).SetBuild(nn::mii::MiiBuild_Min);
    nn::mii::detail::StoreDataRaw::Cast(storeDataNew).UpdateCrcAll();

    NNT_ASSERT_RESULT_FAILURE(
        nn::mii::ResultPermissionDenied,
        database.Get().UpdateLatest(&storeDataNew, nn::mii::SourceFlag_Database));
}
// mii:u NG, mii:e OK
TEST_F(PrivateDatabaseForUserTest, Move)
{
    utility::ScopedPrivateDatabase database;
    database.Get().Format();
    nn::mii::StoreData storeData[2];

    int count;
    NNT_ASSERT_RESULT_FAILURE(
        nn::mii::ResultPermissionDenied,
        database.Get().Get(&count, storeData, 2, nn::mii::SourceFlag_Database));
}
// mii:u NG, mii:e OK
TEST_F(PrivateDatabaseForUserTest, AddOrReplace)
{
    utility::ScopedPrivateDatabase database;
    database.Get().Format();

    nn::mii::StoreData storeData;
    nn::mii::StoreDataContext context;

    /// Mii作成
    nn::mii::StoreDataAccessor(&storeData).BuildDefault(&context, 0);

    NNT_ASSERT_RESULT_FAILURE(
        nn::mii::ResultPermissionDenied,
        database.Get().AddOrReplace(storeData));
}
// mii:u NG, mii:e OK
TEST_F(PrivateDatabaseForUserTest, Delete)
{
    utility::ScopedPrivateDatabase database;
    database.Get().Format();

    nn::mii::StoreData storeData;
    {
    int count;
    NNT_ASSERT_RESULT_FAILURE(
        nn::mii::ResultPermissionDenied,
        database.Get().Get(&count, &storeData, 1, nn::mii::SourceFlag_Database));
    }

    nn::mii::CreateId id = nn::mii::detail::StoreDataRaw::Cast(storeData).GetCreateId();

    NNT_ASSERT_RESULT_FAILURE(
        nn::mii::ResultPermissionDenied,
        database.Get().Delete(id));
}
// mii:u NG, mii:e OK
TEST_F(PrivateDatabaseForUserTest, IsBrokenDatabaseWithClearFlag)
{
    /// 破損させる(DestroyFileのフラグで扱う)
    {
    utility::ScopedPrivateDatabase database;
    database.Get().DestroyFile();
    }

    /// 破損状態をロードする(Load時のフラグをチェックする)
    {
    utility::ScopedPrivateDatabase database;
    }

    // mii:u では必ず false を返す
    ASSERT_FALSE(nn::mii::PrivateDatabase::IsBrokenDatabaseWithClearFlag());
}

