﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/migration/migration_UserMigrationApi.h>
#include <nn/migration/migration_UserMigrationClient.h>
#include <nn/migration/migration_UserMigrationServer.h>

#include <nn/nn_Abort.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/util/util_FormatString.h>

#include "nntMigration_Account.h"
#include "nntMigration_Assertion.h"
#include "testUserMigrationSequence_Util.h"

// ---------------------------------------------------------------------------------------------
// Server

inline bool RunServerImpl(char* buffer, size_t bufferSize)
{
    NN_ABORT_UNLESS_EQUAL(reinterpret_cast<uintptr_t>(buffer) % nn::os::MemoryPageSize, 0u);
    NN_ABORT_UNLESS_EQUAL(bufferSize % nn::os::MemoryPageSize, 0u);

    nn::account::Uid uid = nnt::migration::SelectUserAccount();
    NN_ABORT_UNLESS(uid);
    LOG("[Server] - User account selected: %016llx_%016llx\n", uid._data[0], uid._data[1]);

    NN_LOG("[Server] - Started\n");
    nn::migration::UserMigrationServerProfile profile = {};
    nn::util::SNPrintf(reinterpret_cast<char*>(profile.data), sizeof(profile.data), "Hello, client");
    auto server = nn::migration::CreateUserMigrationServer(uid, profile,  buffer, bufferSize);

    while (NN_STATIC_CONDITION(true))
    {
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(server.PrepareAsync()));
        NNT_MIGRATION_RETURN_FALSE_UNLESS(server.IsConnectionRequired());

        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(server.WaitConnectionAsync()));
        nn::migration::UserMigrationClientProfile clientProfile;
        server.GetClientProfile(&clientProfile);
        NN_LOG(" - Client - \"%s\"\n", clientProfile.data);

        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(server.AcceptConnectionAsync()));

        auto r = WaitAsyncDone(server.ProcessTransferAsync());
        if (r.IsSuccess())
        {
            break;
        }

        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_INCLUDED(nn::migration::ResultSuspendedByExternal, r);
        NN_LOG("[Server] - Suspended\n");

        server = decltype(server)();
        server = nn::migration::ResumeUserMigrationServer(buffer, bufferSize);
    }
    NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(server.CompleteAsync()));
    NN_LOG("[Server] - Completed\n");
    return true;
}
inline bool RunServer(char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return RunServerImpl(buffer, bufferSize);
}

// ---------------------------------------------------------------------------------------------
// Client

inline bool RunClientImpl(const char* id, const char* password, bool testSuspend, char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_EQUAL(reinterpret_cast<uintptr_t>(buffer) % nn::os::MemoryPageSize, 0u);
    NN_ABORT_UNLESS_EQUAL(bufferSize % nn::os::MemoryPageSize, 0u);

    NN_LOG("[Client] - Started\n");
    nn::migration::UserMigrationClientProfile profile = {};
    nn::util::SNPrintf(reinterpret_cast<char*>(profile.data), sizeof(profile.data), "Hello, server");
    auto client = nn::migration::CreateUserMigrationClient(profile, buffer, bufferSize);

    nn::account::SessionId sessionId = client.CreateLoginSession();
    NN_ABORT_UNLESS(nnt::migration::LoginNetworkServiceAccount(sessionId, id, password));

    int suspendCount = testSuspend ? 3 : 0; // TORIAEZU
    while (NN_STATIC_CONDITION(true))
    {
        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(client.PrepareAsync()));
        NNT_MIGRATION_RETURN_FALSE_UNLESS(client.IsConnectionRequired());

        nn::migration::UserMigrationServerInfo server;
        const int ScanCountLimit = 1000;
        bool found = false;
        for (auto scanCount = 0; !found && scanCount < ScanCountLimit; ++ scanCount)
        {
            NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(client.ScanServersAsync()));

            nn::migration::UserMigrationServerInfo servers[4];
            auto actual = client.ListServers(servers, std::extent<decltype(servers)>::value);
            for (auto i = 0u; i < actual; ++ i)
            {
                char str[32];
                servers[i].id.ToString(str, sizeof(str));
                NN_LOG(" - Server[%u] %s - \"%s\"\n", i, str, servers[i].profile.data);
            }

            if (actual > 0)
            {
                // サーバーは自動選択
                server = servers[0];
                found = true;
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
        NNT_MIGRATION_RETURN_FALSE_UNLESS(found);

        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(client.ConnectAsync(server.id)));
        auto shortfall = client.GetStorageShortfall();
        NN_LOG("[Client] - User region shortfall: %zu\n", shortfall);
        NN_ABORT_UNLESS(shortfall <= 0);

        if (suspendCount == 0)
        {
            auto total = client.GetTotalTransferInfo();
            nn::migration::TransferInfo current;
            while ((current = client.GetCurrentTransferInfo()).count < total.count)
            {
                LOG("[Client] - Transferred: %u of %u\n", current.count, total.count);
                NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(client.TransferNextAsync()));
            }
            LOG("[Client] - Transferred: %u of %u\n", current.count, total.count);
            break;
        }

        NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(client.SuspendAsync()));
        NN_LOG("[Client] - Suspended\n");
        -- suspendCount;

        client = decltype(client)();
        client = nn::migration::ResumeUserMigrationClient(buffer, bufferSize);
    }
    NNT_MIGRATION_RETURN_FALSE_UNLESS_RESULT_SUCCESS(WaitAsyncDone(client.CompleteAsync()));
    NN_LOG("[Client] - Completed\n");
    return true;
}
inline bool RunClient(const char* id, const char* password, char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    return RunClientImpl(id, password, true, buffer, bufferSize);
}
