﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <random>
#include <chrono>
#include <set>
#include <tuple>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/fs/fs_ContentStorage.h>

#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_ContentLocationResolverImpl.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/ncm/ncm_ContentStorage.h>

#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaDatabaseImpl.h>
#include <nn/ncm/ncm_IContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

#include <nn/fs/fs_Bis.h>

#include <nn/sf/sf_StdAllocationPolicy.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <new>

namespace {
    typedef std::string String;
    typedef nn::sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> ContentStorageFactory;
    typedef nn::sf::SharedPointer<nn::ncm::IContentStorage> ContentStorageShared;
    typedef nn::sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> ContentMetaDatabaseFactory;
    typedef nn::sf::SharedPointer<nn::ncm::IContentMetaDatabase> ContentMetaDatabaseShared;

    struct ContentData
    {
        nn::ncm::ContentId      id;
        size_t                  size;
        std::vector<int64_t>    data;
    };

    struct ApplicationContentData
    {
        ContentData meta;
        ContentData program;
        ContentData control;
    };

    class RegisteredLocationResolverTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {

        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS(nn::fs::MountHostRoot().IsSuccess());

#if defined(NN_BUILD_CONFIG_OS_WIN32)
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::System, nullptr));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
#else
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::System));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::User));
#endif
            nn::ncm::Initialize();
            nn::lr::Initialize();
        }

        static void TearDownTestCase()
        {
            nn::ncm::Finalize();
            nn::lr::Finalize();
            nn::fs::UnmountHostRoot();
        }


        nn::Result RegisterRandomContent(ContentData* outValue, nn::ncm::ContentStorage* storage)
        {
            auto placeHolderId = storage->GeneratePlaceHolderId();

            auto size = GetRandomSize();
            auto data = GetRandomData(size);
            auto contentId = GetRandomContentId();
            NN_RESULT_DO(storage->CreatePlaceHolder(placeHolderId, contentId, size));
            NN_RESULT_DO(storage->WritePlaceHolder(placeHolderId, 0, data.data(), size));
            NN_RESULT_DO(storage->Register(placeHolderId, contentId));

            ContentData ret = { contentId, size, data };
            *outValue = ret;

            NN_RESULT_SUCCESS;
        }

        nn::ncm::PackagedContentInfo MakePackagedContentInfo(nn::ncm::ContentId contentId, size_t size, nn::ncm::ContentType contentType)
        {
            nn::ncm::PackagedContentInfo packagedContentInfo = {};
            packagedContentInfo.info = nn::ncm::ContentInfo::Make(contentId, size, contentType, 0);
            return packagedContentInfo;
        }

        std::unique_ptr<char[]> ConvertToContentMeta(
            size_t *outSize,
            const ContentData& metaContent,
            const void* packagedContentMeta,
            size_t packagedContentMetaSize)
        {
            nn::ncm::PackagedContentMetaReader pr(packagedContentMeta, packagedContentMetaSize);
            size_t contentMetaSize = pr.CalculateConvertContentMetaSize();
            std::unique_ptr<char[]> contentMeta(new char[contentMetaSize]);

            auto contentInfo = nn::ncm::ContentInfo::Make(metaContent.id, metaContent.size, nn::ncm::ContentType::Meta, 0);
            pr.ConvertToContentMeta(contentMeta.get(), contentMetaSize, contentInfo);

            *outSize = contentMetaSize;
            return contentMeta;
        }

        nn::Result InstallRandomContentApplication(ApplicationContentData* outValue, nn::ncm::ApplicationId id, uint16_t version, nn::ncm::ContentMetaDatabase* metaDatabase, nn::ncm::ContentStorage* storage)
        {
            ContentData meta;
            ContentData program;
            ContentData control;

            NN_RESULT_DO(RegisterRandomContent(&meta, storage));
            NN_RESULT_DO(RegisterRandomContent(&program, storage));
            NN_RESULT_DO(RegisterRandomContent(&control, storage));

            auto metaSize = nn::ncm::PackagedContentMetaWriter::CalculateSize(nn::ncm::ContentMetaType::Application, 2, 0, 0);
            std::unique_ptr<char[]> packagedContentMeta(new char[metaSize]);
            nn::ncm::PackagedContentMetaWriter writer(packagedContentMeta.get(), metaSize);
            writer.WriteHeader(id.value, version, nn::ncm::ContentMetaType::Application, nn::ncm::ContentMetaAttribute_None, 0, 2, 0);
            writer.WriteContentInfo(MakePackagedContentInfo(program.id, program.size, nn::ncm::ContentType::Program), 0);
            writer.WriteContentInfo(MakePackagedContentInfo(control.id, control.size, nn::ncm::ContentType::Control), 1);

            NN_RESULT_DO(metaDatabase->Set(nn::ncm::ContentMetaKey::Make(id, version), writer.GetData(), writer.GetSize()));

            ApplicationContentData appData = { meta, program, control };
            *outValue = appData;

            NN_RESULT_SUCCESS;
        }

        nn::ncm::ContentId GetRandomContentId()
        {
            nn::ncm::ContentId contentId = {};
            auto low = m_RandomEngine();
            auto high = m_RandomEngine();
            std::memcpy(&contentId.data[0], &low, sizeof(low));
            std::memcpy(&contentId.data[8], &high, sizeof(high));

            return contentId;
        }

        size_t GetRandomSize()
        {
            return std::uniform_int_distribution<>(1, 1024 * 1024)(m_RandomEngine);
        }

        void ExpectContent(const nn::lr::Path& path, const ContentData& contentData)
        {
            nn::fs::FileHandle fileHandle;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path.string, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

            int64_t fileSize64;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize64, fileHandle));
            size_t fileSize = static_cast<size_t>(fileSize64);
            EXPECT_EQ(contentData.size, fileSize);

            std::unique_ptr<char[]> buffer(new char[fileSize]);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, buffer.get(), fileSize));
            EXPECT_TRUE(memcmp(contentData.data.data(), buffer.get(), contentData.size) == 0);
        }

        std::vector<int64_t> GetRandomData(size_t size)
        {
            size_t count = nn::util::align_up(size, sizeof(int64_t)) / sizeof(int64_t);
            std::vector<int64_t> data;
            data.reserve(count);
            for (size_t i = 0; i < count; i++)
            {
                data.push_back(m_RandomEngine());
            }

            return data;
        }

        nn::ncm::ApplicationId GetRandomApplicationId()
        {
            nn::ncm::ApplicationId id = { m_RandomEngine() };
            return id;
        }

        std::mt19937_64 m_RandomEngine;
    };
}

TEST_F(RegisteredLocationResolverTest, ResolveProgram)
{
    nn::lr::RegisteredLocationResolver locationResolver;
    NNT_EXPECT_RESULT_SUCCESS(nn::lr::OpenRegisteredLocationResolver(&locationResolver));

    auto id = GetRandomApplicationId();
    nn::lr::Path path;
    nn::lr::Path patchPath = { "patch" };

    // ---------------- 通常登録

    // 未登録の状態では、ProgramNotFound を返す
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultProgramNotFound, locationResolver.ResolveProgramPath(&path, id));

    // 登録は成功する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterProgramPath(id, patchPath));

    // 登録した状態で Resolve すると、登録したパスを返す
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, id));
    EXPECT_STRCASEEQ(path.string, patchPath.string);

    // 登録を解除する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.UnregisterProgramPath(id));


    // 解除した後は、再び ProgramNotFound を返す
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultProgramNotFound, locationResolver.ResolveProgramPath(&path, id));

    // ------------------ 上書き

    auto id2 = GetRandomApplicationId();
    nn::lr::Path patchPath2 = { "patch2" };

    // 登録は成功する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterProgramPath(id, patchPath));

    // 登録した状態で Resolve すると、登録したパスを返す
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, id));
    EXPECT_STRCASEEQ(path.string, patchPath.string);

    // 上書きも成功する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterProgramPath(id2, patchPath2));

    // 登録した状態で Resolve すると、登録したパスを返す
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, id2));
    EXPECT_STRCASEEQ(path.string, patchPath2.string);

    // 元の ID でとろうとすると、ProgramNotFound を返す
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultProgramNotFound, locationResolver.ResolveProgramPath(&path, id));

    // 登録を解除する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.UnregisterProgramPath(id2));

    // ------------------  リダイレクト

    nn::lr::Path redirect = { "redirect" };
    nn::lr::Path redirected;

    // Redirect の登録
    locationResolver.RedirectProgramPath(id, redirect);

    // Redirect した結果が返る
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&redirected, id));
    EXPECT_STRCASEEQ(redirected.string, redirect.string);

    // Register と Redirect を両方登録した場合、Redirect が優先される
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterProgramPath(id, patchPath));

    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, id));
    EXPECT_STRCASEEQ(path.string, redirect.string);
}

TEST_F(RegisteredLocationResolverTest, ResolveHtmlDocument)
{
    nn::lr::RegisteredLocationResolver locationResolver;
    NNT_EXPECT_RESULT_SUCCESS(nn::lr::OpenRegisteredLocationResolver(&locationResolver));

    auto id = GetRandomApplicationId();
    nn::lr::Path path;
    nn::lr::Path patchPath = { "patch" };

    // ---------------- 通常登録

    // 未登録の状態では、HtmlDocumentNotFound を返す
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultHtmlDocumentNotFound, locationResolver.ResolveHtmlDocumentPath(&path, id));

    // 登録は成功する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterHtmlDocumentPath(id, patchPath));

    // 登録した状態で Resolve すると、登録したパスを返す
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveHtmlDocumentPath(&path, id));
    EXPECT_STRCASEEQ(path.string, patchPath.string);

    // 登録を解除する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.UnregisterHtmlDocumentPath(id));


    // 解除した後は、再び HtmlDocumentNotFound を返す
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultHtmlDocumentNotFound, locationResolver.ResolveHtmlDocumentPath(&path, id));

    // ------------------ 上書き

    auto id2 = GetRandomApplicationId();
    nn::lr::Path patchPath2 = { "patch2" };

    // 登録は成功する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterHtmlDocumentPath(id, patchPath));

    // 登録した状態で Resolve すると、登録したパスを返す
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveHtmlDocumentPath(&path, id));
    EXPECT_STRCASEEQ(path.string, patchPath.string);

    // 上書きも成功する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterHtmlDocumentPath(id2, patchPath2));

    // 登録した状態で Resolve すると、登録したパスを返す
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveHtmlDocumentPath(&path, id2));
    EXPECT_STRCASEEQ(path.string, patchPath2.string);

    // 元の ID でとろうとすると、HtmlDocumentNotFound を返す
    NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultHtmlDocumentNotFound, locationResolver.ResolveHtmlDocumentPath(&path, id));

    // 登録を解除する
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.UnregisterHtmlDocumentPath(id2));

    // ------------------  リダイレクト

    nn::lr::Path redirect = { "redirect" };
    nn::lr::Path redirected;

    // Redirect の登録
    locationResolver.RedirectHtmlDocumentPath(id, redirect);

    // Redirect した結果が返る
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveHtmlDocumentPath(&redirected, id));
    EXPECT_STRCASEEQ(redirected.string, redirect.string);

    // Register と Redirect を両方登録した場合、Redirect が優先される
    NNT_EXPECT_RESULT_SUCCESS(locationResolver.RegisterHtmlDocumentPath(id, patchPath));

    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveHtmlDocumentPath(&path, id));
    EXPECT_STRCASEEQ(path.string, redirect.string);
}

