﻿using System;
using System.IO;
using System.Net.Sockets;
using System.Text;
using System.Xml;

namespace LdnTestBridge
{
    internal class TestTarget
    {
        /// <summary>
        /// ターゲットの状態です。
        /// </summary>
        public enum State
        {
            Connected,
            Server,
            Client,
            Disconnected
        }

        /// <summary>
        /// エラーコードです。
        /// </summary>
        public enum ErrorCode
        {
            SUCCESS,
            UNKNOWN,
            INVALID_STATE,
            INVALID_FORMAT,
            DUPLICATE,
            NOT_FOUND,
            FULL,
            TIMEOUT
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="client">TCP コネクションです。</param>
        /// <param name="groups">グループリストです。</param>
        public TestTarget(TcpClient client, IRepository<TestGroup> groups)
        {
            m_TcpClient = client;
            m_State = State.Connected;
            m_Groups = groups;
        }

        public void Process()
        {
            // TCP のコネクションが切断されるまで繰り返します。
            bool isRunning = true;
            while (isRunning)
            {
                // テストノードからのデータ受信まで待機します。
                string line;
                using (var stream = new StreamReader(
                    m_TcpClient.GetStream(), Encoding.ASCII, false, 4096, true))
                {
                    try
                    {
                        line = stream.ReadLine();
                        if (line.Length == 0)
                        {
                            break;
                        }
                    }
                    catch (IOException e)
                    {
                        Console.WriteLine(e);
                        break;
                    }
                }

                // 受信したデータを XML データとして解析します。
                var reader = new XmlDocument();
                try
                {
                    reader.LoadXml(line);
                }
                catch (XmlException e)
                {
                    Console.WriteLine(e);
                    continue;
                }
                var root = reader.DocumentElement;

                // リクエストに応じた処理の分岐です。
                switch (root.Name)
                {
                    case "CreateGroupRequest":
                        ProcessCreateGroupRequest(root);
                        break;
                    case "JoinGroupRequest":
                        ProcessJoinGroupRequest(root);
                        break;
                    case "SyncRequest":
                        ProcessSyncRequest(root);
                        break;
                    case "LeaveRequest":
                        ProcessLeaveRequest(root);
                        isRunning = false;
                        break;
                    default:
                        break;
                }
            }

            // コネクションを閉じます。
            m_TcpClient.Close();
        }

        private void ProcessCreateGroupRequest(XmlElement root)
        {
            // 現在の状態を確認します。
            if (m_State != State.Connected)
            {
                SendCreateGroupResponse(ErrorCode.INVALID_STATE);
                return;
            }

            // グループの名前を取得します。
            var groupNameNodes = root.GetElementsByTagName("GroupName");
            if (groupNameNodes.Count == 0)
            {
                SendCreateGroupResponse(ErrorCode.INVALID_FORMAT);
                return;
            }
            var groupName = groupNameNodes.Item(0).InnerText;
            if (groupName.Length < TestGroup.NAME_LENGTH_MIN ||
                TestGroup.NAME_LENGTH_MAX < groupName.Length)
            {
                SendCreateGroupResponse(ErrorCode.INVALID_FORMAT);
                return;
            }

            // クライアント数を取得します。
            int clientCount;
            var clientCountNodes = root.GetElementsByTagName("ClientCount");
            if (clientCountNodes.Count == 0)
            {
                SendCreateGroupResponse(ErrorCode.INVALID_FORMAT);
                return;
            }
            if (!int.TryParse(clientCountNodes.Item(0).InnerText, out clientCount) ||
                clientCount <= 0)
            {
                SendCreateGroupResponse(ErrorCode.INVALID_FORMAT);
                return;
            }

            // グループを登録します。
            var group = new TestGroup(groupName, clientCount);
            if (!m_Groups.Add(group))
            {
                SendCreateGroupResponse(ErrorCode.DUPLICATE);
                return;
            }

            // グループの作成に成功しました。
            m_State = State.Server;
            m_Group = group;
            SendCreateGroupResponse(ErrorCode.SUCCESS);
            Console.WriteLine("Create Group: {0} (1/{1})", group.Name, group.ClientCountMax + 1);
        }

        private void SendCreateGroupResponse(ErrorCode error)
        {
            var document = CreateResponse("CreateGroup", error);
            SendResponse(document);
        }

        private void ProcessJoinGroupRequest(XmlElement root)
        {
            // 現在の状態を確認します。
            if (m_State != State.Connected)
            {
                SendCreateGroupResponse(ErrorCode.INVALID_STATE);
                return;
            }

            // グループの名前を取得します。
            var groupNameNodes = root.GetElementsByTagName("GroupName");
            if (groupNameNodes.Count == 0)
            {
                SendJoinGroupResponse(ErrorCode.INVALID_FORMAT);
                return;
            }
            var groupName = groupNameNodes.Item(0).InnerText;
            if (groupName.Length < TestGroup.NAME_LENGTH_MIN ||
                TestGroup.NAME_LENGTH_MAX < groupName.Length)
            {
                SendJoinGroupResponse(ErrorCode.INVALID_FORMAT);
                return;
            }

            // 対象のグループを検索します。
            TestGroup group = m_Groups.Find((obj) => obj.Name == groupName);
            if (group == null)
            {
                SendJoinGroupResponse(ErrorCode.NOT_FOUND);
                return;
            }

            // グループに参加します。
            int clientIndex = group.Join();
            if (clientIndex <= 0)
            {
                SendJoinGroupResponse(ErrorCode.FULL);
                return;
            }

            // グループへの参加に成功しました。
            m_State = State.Client;
            m_Group = group;
            m_ClientIndex = clientIndex;
            SendJoinGroupResponse(ErrorCode.SUCCESS);
            Console.WriteLine("Join: {0} ({1}/{2})",
                group.Name, group.ClientCount + 1, group.ClientCountMax + 1);
        }

        private void SendJoinGroupResponse(ErrorCode error)
        {
            // XML 形式のレスポンスを生成します。
            var document = CreateResponse("JoinGroup", error);
            var root = document.DocumentElement;

            // 成功した場合にはレスポンスに詳細を追記します。
            if (error == ErrorCode.SUCCESS)
            {
                // レスポンスにクライアント数を追加します。
                var clientCountNode = document.CreateElement("ClientCount");
                clientCountNode.InnerText = m_Group.ClientCountMax.ToString();
                root.AppendChild(clientCountNode);

                // レスポンスにクライアントのインデックスを追加します。
                var clientIndexNode = document.CreateElement("ClientIndex");
                clientIndexNode.InnerText = m_ClientIndex.ToString();
                root.AppendChild(clientIndexNode);
            }

            // レスポンスを送信します。
            SendResponse(document);
        }

        private void ProcessSyncRequest(XmlElement root)
        {
            // 現在の状態を確認します。
            if (m_State != State.Server && m_State != State.Client)
            {
                SendSyncResponse(ErrorCode.INVALID_STATE);
                return;
            }

            // 同期に使用するキーワードを取得します。
            var keywordNodes = root.GetElementsByTagName("Keyword");
            if (keywordNodes.Count == 0)
            {
                SendSyncResponse(ErrorCode.INVALID_FORMAT);
                return;
            }
            string keyword = keywordNodes.Item(0).InnerText;
            if (keyword.Length < TestSynchronizer.KEYWORD_LENGTH_MIN ||
                TestSynchronizer.KEYWORD_LENGTH_MAX < keyword.Length)
            {
                SendSyncResponse(ErrorCode.INVALID_FORMAT);
                return;
            }

            // 同期で共有するデータを取得します。
            var dataNodes = root.GetElementsByTagName("Data");
            string data = null;
            if (0 < dataNodes.Count)
            {
                if (m_State == State.Client)
                {
                    SendSyncResponse(ErrorCode.INVALID_FORMAT);
                    return;
                }
                data = dataNodes.Item(0).InnerText;
            }

            // タイムアウト時間 (ms) を取得します。
            int timeoutMs;
            var timeoutNodes = root.GetElementsByTagName("Timeout");
            if (timeoutNodes.Count == 0)
            {
                SendSyncResponse(ErrorCode.INVALID_FORMAT);
                return;
            }
            if (!int.TryParse(timeoutNodes.Item(0).InnerText, out timeoutMs) ||
                timeoutMs < 0)
            {
                SendSyncResponse(ErrorCode.INVALID_FORMAT);
                return;
            }
            var timeout = TimeSpan.FromMilliseconds(timeoutMs);

            // 同期で共有するデータを設定します。
            var synchronizer = m_Group.GetSynchronizer(keyword);
            if (m_State == State.Server && data != null)
            {
                synchronizer.Data = data;
            }

            // 同期します。
            bool isSucceeded = synchronizer.Sync(timeout);

            // 同期に成功した場合にはデータを取得し、同期オブジェクトを削除します。
            if (isSucceeded)
            {
                if (m_State == State.Client)
                {
                    data = synchronizer.Data;
                }
                m_Group.RemoveSynchronizer(keyword);
            }
            else
            {
                Console.WriteLine("Sync Timeout: {0}", keyword);
                SendSyncResponse(ErrorCode.TIMEOUT);
                return;
            }

            // 同期に成功しました。
            SendSyncResponse(ErrorCode.SUCCESS, data);
        }

        private void SendSyncResponse(ErrorCode error, string data = null)
        {
            // XML 形式のレスポンスを生成します。
            var document = CreateResponse("Sync", error);
            var root = document.DocumentElement;

            // 成功した場合にはデータを追記します。
            if (error == ErrorCode.SUCCESS && data != null && 0 < data.Length)
            {
                var dataNode = document.CreateElement("Data");
                dataNode.InnerText = data;
                root.AppendChild(dataNode);
            }

            // レスポンスを送信します。
            SendResponse(document);
        }

        private void ProcessLeaveRequest(XmlElement root)
        {
            if (m_State == State.Server)
            {
                Console.WriteLine("Destroy Group: {0}", m_Group.Name);
                m_Groups.Remove(m_Group);
            }
            else if (m_State == State.Client)
            {
                m_Group.Leave();
                Console.WriteLine("Leave: {0} ({1}/{2})",
                    m_Group.Name, m_Group.ClientCount + 1, m_Group.ClientCountMax + 1);
            }
            m_Group = null;
            m_State = State.Disconnected;
        }

        private XmlDocument CreateResponse(string name, ErrorCode errorCode)
        {
            var document = new XmlDocument();

            // ルート要素を追加します。
            var declaration = document.CreateXmlDeclaration("1.0", "UTF-8", null);
            var root = document.CreateElement(name + "Response");
            document.AppendChild(root);

            // エラーコードを追加します。
            var errorCodeNode = document.CreateElement("Result");
            errorCodeNode.InnerText = ((int)errorCode).ToString();
            root.AppendChild(errorCodeNode);

            return document;
        }

        private void SendResponse(XmlDocument document)
        {
            var xml = document.OuterXml;
            using (var stream = new StreamWriter(
                m_TcpClient.GetStream(), Encoding.ASCII, 4096, true))
            {
                try
                {
                    stream.WriteLine(xml);
                }
                catch (IOException)
                {
                    // 送信に失敗しました。
                }
            }
        }

        private TcpClient m_TcpClient;
        private State m_State;
        private IRepository<TestGroup> m_Groups;
        private TestGroup m_Group;
        private int m_ClientIndex;
   }
}
