﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include <nnt/nnt_Argument.h>
#include "../../../../../../Net/Sources/Tests/Performance/testNet_P2P/testNet_P2P.h"
#include "../Common/Definition.h"
#include "../../Common/Util.h"

namespace
{
    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
}

NN_ALIGNAS(4096) char g_LdnThreadStack[8192];
NN_ALIGNAS(4096) char g_MonitorThreadStack[8192];

int LdnThreadPriority       = nn::os::DefaultThreadPriority + 2;
int MonitorThreadPriority   = nn::os::DefaultThreadPriority + 1;

const size_t BufferSize     = 512;

struct ApplicationResource
{
    nn::os::ThreadType          monitorThread;
    nn::os::ThreadType          p2pThread;
    nn::os::SystemEventType     stateChangeEvent;
    nn::os::EventType           cancelEvent;
    nn::os::MutexType           mutex;
    nn::ldn::NetworkInfo        networkInfo;
    nn::ldn::Ipv4Address        ipv4Address;
    nn::ldn::SubnetMask         subnetMask;
    uint32_t                    counter[nn::ldn::NodeCountMax];
    int                         socket;
    int                         argc;
    char**                      argv;
};

void MonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;
    nn::ldn::NetworkInfo networkInfo;

    // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

    while (nn::os::WaitAny(&multiWait) != &cancelHolder)
    {
        // WaitAny では必ず手動でシグナルをクリアしなければなりません。
        nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

        // ネットワークが破棄された場合には監視を終了します。
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState != nn::ldn::State_AccessPointCreated)
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 接続中のステーションを取得します。
        result = nn::ldn::GetNetworkInfo(&networkInfo);
        if (result <= nn::ldn::ResultInvalidState())
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 接続状態が変化したステーションを検索します。ノード番号 0 は AP なので無視します。
        nn::os::LockMutex(&app.mutex);
        for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
        {
            auto& before = app.networkInfo.ldn.nodes[i];
            auto& after  = networkInfo.ldn.nodes[i];
            if (!before.isConnected && after.isConnected)
            {
                NN_LOG("新しいステーションの接続です。\n");
                NN_LOG("Node ID    : %d\n", after.nodeId);
                NN_LOG("IP Address : %s\n\n", ConvertToString(after.ipv4Address));
            }
            else if (before.isConnected && !after.isConnected)
            {
                NN_LOG("ステーションが切断しました。\n");
                NN_LOG("Node ID    : %d\n", before.nodeId);
                NN_LOG("IP Address : %s\n\n", ConvertToString(before.ipv4Address));
            }
            else if (before.isConnected && after.isConnected &&
                     before.ipv4Address != after.ipv4Address)
            {
                NN_LOG("ステーションが切断しました。\n");
                NN_LOG("Node ID    : %d\n", before.nodeId);
                NN_LOG("IP Address : %s\n\n", ConvertToString(before.ipv4Address));
                NN_LOG("新しいステーションの接続です。\n");
                NN_LOG("Node Id    : %d\n", after.nodeId);
                NN_LOG("IP Address : %s\n\n", ConvertToString(after.ipv4Address));
            }
        }
        std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
        nn::os::UnlockMutex(&app.mutex);
    }

    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);
}

void P2PFunction(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    char argvAddr[512]          = "--NODES_ADDR_INFO=";
    char argvIndex[16];
    char address[32];
    nn::socket::InAddr addr;
    int nodeIndex = 0;

    for (int i = 0; i < app.networkInfo.ldn.nodeCount ; ++i)
    {
        if(app.networkInfo.ldn.nodes[i].ipv4Address.raw ==  app.ipv4Address.raw)
        {
            nodeIndex = i;
        }
        addr.S_addr = nn::socket::InetHtonl(app.networkInfo.ldn.nodes[i].ipv4Address.raw);
        sprintf(address, "%s:%d,", nn::socket::InetNtoa(addr), 8001 + i);
        strncat(argvAddr, address, BufferSize - strlen(argvAddr) - 1);
    }
    argvAddr[strlen(argvAddr) - 1] = '\0';

    sprintf(argvIndex, "--NODE_INDEX=%d", nodeIndex);

    int    argC = 3 + (app.argc - 1);

    char** arguments = (char **)std::malloc(sizeof(char *) * argC);

    arguments[0] = app.argv[0];
    arguments[1] = argvIndex;
    arguments[2] = argvAddr;

    for (int i =0; i < (app.argc - 1); ++i)
    {
        arguments[i + 3] = app.argv[i + 1];
    }

    const char** argV = (const char**)arguments;

    NN_LOG("start(argc:%d) ", argC);
    for(int i = 0; i < argC; i++)
    {
        NN_LOG("%s ", argV[i], argV[i]);
    }
    NN_LOG("\n");

    nnt::net::p2p::PacketRateTest::RunTest(argC, argV);

    std::free(arguments);
}

nn::Result CreateAccessPoint(ApplicationResource* pApp) NN_NOEXCEPT
{
    // アクセスポイントモードとして起動します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::OpenAccessPoint());

    // Advertise で配信するデータを設定します。
    const char message[] = "Welcome to LDN network!";
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::SetAdvertiseData(message, sizeof(message)));

    // 新規に構築する LDN ネットワークのパラメータを設定します。
    nn::ldn::NetworkConfig  network = { };
    nn::ldn::SecurityConfig security = { };
    nn::ldn::UserConfig user = { };
    network.intentId        = BasicSampleIntentId;
    network.channel         = nn::ldn::AutoChannel;
    network.nodeCountMax    = nn::ldn::NodeCountMax;
    security.securityMode   = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    std::strncpy(user.userName, "p2pAp", nn::ldn::UserNameBytesMax);

    // アクセスポイントとして新規にネットワークを構築します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::CreateNetwork(network, security, user));

    // ネットワークの情報を取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&pApp->networkInfo));

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask));

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, MonitorThread, pApp,
        g_MonitorThreadStack, sizeof(g_MonitorThreadStack),
        MonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void DestroyAccessPoint(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークを破棄し、 LDN ライブラリの使用を終了します。
    nn::ldn::DestroyNetwork();
    nn::ldn::CloseAccessPoint();
}

void LdnAccessPoint(void* arg) NN_NOEXCEPT
{
    nn::Result result;

    // LDN ネットワークを構築します。
    auto& app = *static_cast<ApplicationResource*>(arg);
    result = CreateAccessPoint(&app);
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("LDN ネットワークの構築に成功しました。\n");
    NN_LOG("Channel    : %d\n", app.networkInfo.common.channel);
    NN_LOG("IP Address : %s/%s\n\n",
           ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));

    // サンプルの操作方法を出力します。
    NN_LOG("A ボタンで計測を開始します。\n");

    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
    nn::os::WaitEvent(&app.cancelEvent);

    // ソケットを破棄します。
    nn::socket::Shutdown(app.socket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Close(app.socket);

    // LDN ネットワークを破棄し、データの送受信を終了します。
    DestroyAccessPoint(&app);
    NN_LOG("LDN ネットワークが破棄されました。\n\n");
}

extern "C" void nnMain()
{
    nn::Result result;

    // デバッグパッドから入力を取得するために HID ライブラリを取得します。
    nn::hid::InitializeDebugPad();

    // LDN ライブラリを初期化します。
    result = nn::ldn::Initialize();
    if (result <= nn::ldn::ResultDeviceOccupied())
    {
        // 他の通信機能を使用中のためローカル通信機能を利用できません。
        // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // socket ライブラリを初期化します。
    result = nn::socket::Initialize(
        reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);

    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // アプリケーションが使用するリソースを初期化します。
    ApplicationResource app;
    std::memset(&app, 0, sizeof(app));

    app.argc = ::nnt::GetHostArgc();
    app.argv = ::nnt::GetHostArgv();

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);

    nn::hid::DebugPadState padState;
    nn::os::ThreadType ldnThread;

    nn::os::InitializeEvent(&app.cancelEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMutex(&app.mutex, false, 0);
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &ldnThread, LdnAccessPoint, &app,
        g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
    nn::os::StartThread(&ldnThread);


    for (;;)
    {
        // デバッグパッドの入力を取得します。
        nn::hid::GetDebugPadState(&padState);

        if (padState.buttons.Test<nn::hid::DebugPadButton::A>())
        {
            NN_LOG("入力を受け付けました。計測を開始します。\n\n");

            P2PFunction(&app);

            nn::os::WaitThread(&ldnThread);
            nn::os::DestroyThread(&ldnThread);
            nn::os::FinalizeMutex(&app.mutex);
            nn::os::FinalizeEvent(&app.cancelEvent);

            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    // socket ライブラリの使用を終了します。
    result = nn::socket::Finalize();
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ライブラリの使用を終了します。
    nn::ldn::Finalize();
}
