﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkLog.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_TypesPrivate.h>
#include "IcmpTransfer.h"

static unsigned short checksum(unsigned short *buf, int bufsz) NN_NOEXCEPT
{
    unsigned long sum = 0;

    while(bufsz > 1)
    {
        sum += *buf;
        buf++;
        bufsz -= 2;
    }

    if(bufsz == 1)
    {
        sum += *(unsigned char *)buf;
    }

    sum = (sum & 0xffff) + (sum >> 16);
    sum = (sum & 0xffff) + (sum >> 16);

    return ~sum;
}

void CreateIcmpPacket(IcmpPacketData* pData)
{
    memset(&pData->hdr, 0, sizeof(pData->hdr));

    /* ICMPヘッダ */
    pData->hdr.icmp_type = nn::socket::IcmpType::Icmp_Echo;
    pData->hdr.icmp_code = static_cast<nn::socket::IcmpCode>(0);
    pData->hdr.icmp_cksum = 0;
    pData->hdr.icmp_hun.ih_idseq.icd_id = 0;
    pData->hdr.icmp_hun.ih_idseq.icd_seq = 0;

    /* Data */
    int64_t tick = nn::os::GetSystemTick().GetInt64Value();
    pData->tick[0] = tick & 0xFF;
    pData->tick[1] = tick>>8  & 0xFF;
    pData->tick[2] = tick>>16 & 0xFF;
    pData->tick[3] = tick>>24 & 0xFF;
    pData->tick[4] = tick>>32 & 0xFF;
    pData->tick[5] = tick>>40 & 0xFF;
    pData->tick[6] = tick>>48 & 0xFF;
    pData->tick[7] = tick>>56 & 0xFF;

    /* ICMPヘッダのチェックサムを計算 */
    pData->hdr.icmp_cksum = checksum((unsigned short *)&(pData->hdr), sizeof(IcmpPacketData));
}

void EchoIcmpPacket(IcmpPacketData* pData, int64_t tick)
{
    memset(&pData->hdr, 0, sizeof(pData->hdr));

    /* ICMPヘッダ */
    pData->hdr.icmp_type = nn::socket::IcmpType::Icmp_EchoReply;
    pData->hdr.icmp_code = static_cast<nn::socket::IcmpCode>(0);
    pData->hdr.icmp_cksum = 0;
    pData->hdr.icmp_hun.ih_idseq.icd_id = 0;
    pData->hdr.icmp_hun.ih_idseq.icd_seq = 0;

    /* Data */
    pData->tick[0] = tick & 0xFF;
    pData->tick[1] = tick >> 8 & 0xFF;
    pData->tick[2] = tick >> 16 & 0xFF;
    pData->tick[3] = tick >> 24 & 0xFF;
    pData->tick[4] = tick >> 32 & 0xFF;
    pData->tick[5] = tick >> 40 & 0xFF;
    pData->tick[6] = tick >> 48 & 0xFF;
    pData->tick[7] = tick >> 56 & 0xFF;

    /* ICMPヘッダのチェックサムを計算 */
    pData->hdr.icmp_cksum = checksum((unsigned short *)&(pData->hdr), sizeof(IcmpPacketData));
}
