﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/ldn.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nnt.h>
#include <nnt/ldn/testLdn_CommandLineParser.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_Log.h>
#include "Config.h"

namespace
{
    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronization* g_pSync;
    nnt::ldn::ISynchronizationClient* g_pClient;
    nnt::ldn::ISynchronizationServer* g_pServer;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
    #define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pSync->Synchronize(keyword))

    //! 通信相手毎のテスト状況です。
    struct NodeTestInfo
    {
        //! ユニキャストの送信フレーム数です。
        int32_t unicastSendCount;

        //! 最後に受信したユニキャストのカウンタです。
        int32_t unicastCurrentIndex;

        //! 最初に受信したブロードキャストのカウンタです。
        int32_t broadcastStartIndex;

        //! 最後に受信したブロードキャストのカウンタです。
        int32_t broadcastCurrentIndex;
    };

    //! 通信相手毎のテスト結果です。
    struct NodeTestResult
    {
        //! ユニキャストの受信フレーム数です。
        int32_t unicastReceiveCount;

        //! ブロードキャストの受信フレーム数です。
        int32_t broadcastReceiveCount;

        //! ユニキャストの受信損失数です。
        int32_t unicastLossCount;

        //! ブロードキャストの受信損失数です。
        int32_t broadcastLossCount;

        //! ユニキャストの連続受信損失数です。
        int32_t unicastBurstLossCount;

        //! ブロードキャストの連続受信損失数です。
        int32_t broadcastBurstLossCount;
    };

    //! スキャンに関するテスト結果の定義です。
    struct TestResult
    {
        //! 通信相手毎のテスト結果です。
        NodeTestResult nodes[nn::ldn::NodeCountMax];
    };

    //! 現在のシーンです。
    enum TestScene
    {
        TestScene_Setup,
        TestScene_CreatingNetwork,
        TestScene_NetworkCreated,
        TestScene_Scanning,
        TestScene_Connecting,
        TestScene_Connected,
        TestScene_Cleanup,
        TestScene_Finalized,
    };

    //! テストの状態です。
    struct TestStatus
    {
        int32_t scene;
        int32_t socket;
        int32_t elapsed;
        int32_t lastElapsed;
        int32_t sendCounter;
        NodeTestInfo nodes[nn::ldn::NodeCountMax];
        nn::ldn::Ipv4Address address;
        nn::ldn::SubnetMask mask;
        TestResult result;
    };

    //! データ通信で送受信するフレームです。
    struct Frame
    {
        int32_t  counter;
        int32_t  hash;
        bool     isUnicast;
        char     _reserved;
        uint16_t dataSize;
        char     data[1465];
    };

    // テストの設定値です。
    nnt::ldn::TestConfig g_TestConfig;

    // 状態変化を監視するイベントです。
    nn::os::SystemEvent g_StateChangeEvent;

    // socket のバッファです。
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // スキャンバッファです。
    nn::ldn::NetworkInfo g_ScanBuffer[nn::ldn::ScanResultCountMax];

    // 現在接続中のネットワーク情報と、ネットワーク情報を保護するミューテックスです。
    nn::ldn::NetworkInfo g_Network;
    nn::os::Mutex g_NetworkMutex(false);

    // ユニキャスト送信スレッドです。
    nn::os::ThreadType g_UnicastSendThread;
    NN_ALIGNAS(4096) char g_UnicastSendThreadStack[16 * 1024];
    const int UnicastSendThreadPriority = nn::os::DefaultThreadPriority;

    // ブロードキャスト送信スレッドです。
    nn::os::ThreadType g_BroadcastSendThread;
    NN_ALIGNAS(4096) char g_BroadcastSendThreadStack[16 * 1024];
    const int BroadcastSendThreadPriority = nn::os::DefaultThreadPriority;

    // 受信スレッドです。
    nn::os::ThreadType g_ReceiveThread;
    NN_ALIGNAS(4096) char g_ReceiveThreadStack[16 * 1024];
    const int ReceiveThreadPriority = nn::os::DefaultThreadPriority;

    void SetLdnSettings() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetWirelessControllerRestriction(
            static_cast<nn::ldn::WirelessControllerRestriction>(
                g_TestConfig.wirelessControllerRestriction)));
    }

    void PrintTestResult(const TestStatus& status) NN_NOEXCEPT
    {
        NNT_LDN_LOG_INFO_WITHOUT_PREFIX("\n");
        int hour = status.elapsed / 3600;
        int min  = (status.elapsed / 60) % 60;
        int sec  = status.elapsed % 60;
        NNT_LDN_LOG_INFO("Time         : %02d:%02d:%02d\n", hour, min, sec);
        NNT_LDN_LOG_INFO("     %-35s %-35s\n", "Unicast", "Broadcast");
        NNT_LDN_LOG_INFO("%-4s %-8s %-8s %-8s %-8s %-8s %-8s %-8s %-8s\n", "ID",
            "Recv", "Loss", "Burst", "Rate",
            "Recv", "Loss", "Burst", "Rate");
        for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
        {
            const auto& node = g_Network.ldn.nodes[i];
            const auto& result = status.result.nodes[i];
            if (node.isConnected && node.ipv4Address != status.address)
            {
                int unicastCount = result.unicastLossCount + result.unicastReceiveCount;
                float unicastLossRate = 0 < unicastCount ?
                    100.0f * result.unicastLossCount / unicastCount : 0.0f;
                int broadcastCount = result.broadcastLossCount + result.broadcastReceiveCount;
                float broadcastLossRate = 0 < broadcastCount ?
                    100.0f * result.broadcastLossCount / broadcastCount : 0.0f;
                NNT_LDN_LOG_INFO("%-4d %-8d %-8d %-8d %-8.3f %-8d %-8d %-8d %-8.3f\n", i,
                    result.unicastReceiveCount, result.unicastLossCount,
                    result.unicastBurstLossCount, unicastLossRate,
                    result.broadcastReceiveCount, result.broadcastLossCount,
                    result.broadcastBurstLossCount, broadcastLossRate);
            }
        }
    }

    uint32_t CalculateHash(const Frame& frame) NN_NOEXCEPT
    {
        uint32_t hash = UINT32_C(0x5c5c5c5c);
        for (int i = 0; i < g_TestConfig.payloadSize / 4; ++i)
        {
            hash ^= reinterpret_cast<const uint32_t*>(&frame)[i];
        }
        return hash;
    }

    bool VerifyAndRestore(void* pFrame, size_t frameSize) NN_NOEXCEPT
    {
        if (frameSize < 12)
        {
            return false;
        }
        auto& frame = *static_cast<Frame*>(pFrame);
        uint32_t hash = CalculateHash(frame);
        frame.counter = nn::socket::InetNtohl(frame.counter);
        frame.dataSize = nn::socket::InetNtohs(frame.dataSize);
        return frame.dataSize == frameSize - 12 && hash == 0;
    }

    void BroadcastSendThread(void* arg) NN_NOEXCEPT
    {
        auto& status = *static_cast<TestStatus*>(arg);

        // ブロードキャスト送信用のアドレスを生成します。
        NNT_LDN_LOG_DEBUG("broadcast send thread started\n");
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(PortNumber);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(
            nn::ldn::MakeBroadcastAddress(status.address, status.mask).raw);

        // 一定時間毎にパケットの送信を繰り返します。
        auto interval = nn::TimeSpan::FromMicroSeconds(1000000 / g_TestConfig.broadcastRate);
        const size_t dataSize = g_TestConfig.payloadSize - 12;
        while (status.scene == TestScene_Connected || status.scene == TestScene_NetworkCreated)
        {
            // 開始時刻を取得します。
            nn::os::Tick startedAt = nn::os::GetSystemTick();

            // 送信データを生成します。
            ++status.sendCounter;
            Frame frame = { };
            frame.counter = nn::socket::InetHtonl(status.sendCounter);
            frame.dataSize = nn::socket::InetHtons(static_cast<uint16_t>(dataSize));
            frame.hash = CalculateHash(frame);

            // ブロードキャストパケットを送信します。
            if (2 <= g_Network.ldn.nodeCount)
            {
                auto ret = nn::socket::SendTo(
                    status.socket, &frame, g_TestConfig.payloadSize, nn::socket::MsgFlag::Msg_None,
                    reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr));
                if (ret < 0)
                {
                    nn::socket::Errno error = nn::socket::GetLastError();
                    if (error != nn::socket::Errno::ENoBufs)
                    {
                        NNT_LDN_LOG_WARN("failed to send: errno = %d\n", error);
                        break;
                    }
                    else
                    {
                        NNT_LDN_LOG_DEBUG("failed to send: nn::socket::Errno::ENoBufs\n");
                    }
                }
            }

            // パケットレート調整のためスリープします。
            auto elapsed = (nn::os::GetSystemTick() - startedAt).ToTimeSpan();
            auto diff = interval - elapsed;
            if (0 < diff)
            {
                nn::os::SleepThread(diff);
            }
        }
        NNT_LDN_LOG_DEBUG("broadcast send thread finished\n");
    }

    void UnicastSendThread(void* arg) NN_NOEXCEPT
    {
        auto& status = *static_cast<TestStatus*>(arg);

        NNT_LDN_LOG_DEBUG("unicast send thread started\n");

        // 一定時間毎にパケットの送信を繰り返します。
        auto interval = nn::TimeSpan::FromMicroSeconds(1000000 / g_TestConfig.unicastRate);
        const size_t dataSize = g_TestConfig.payloadSize - 12;
        int dstId = 0;
        while (status.scene == TestScene_Connected || status.scene == TestScene_NetworkCreated)
        {
            // 開始時刻を取得します。
            nn::os::Tick startedAt = nn::os::GetSystemTick();

            // 次の送信先を決定します。
            nn::ldn::Ipv4Address dst = nn::ldn::ZeroIpv4Address;
            {
                std::lock_guard<nn::os::Mutex> lock(g_NetworkMutex);
                if (2 <= g_Network.ldn.nodeCount)
                {
                    nn::ldn::NodeInfo node;
                    do
                    {
                        dstId = (dstId + 1) % nn::ldn::NodeCountMax;
                        node = g_Network.ldn.nodes[dstId];
                    } while (!node.isConnected || node.ipv4Address == status.address);
                    dst = node.ipv4Address;
                }
            }

            // 自分以外にもノードが存在する場合にのみ送信します。
            if (dst != nn::ldn::ZeroIpv4Address)
            {
                auto& node = status.nodes[dstId];

                // ユニキャスト送信用のアドレスを生成します。
                nn::socket::SockAddrIn addr;
                addr.sin_family = nn::socket::Family::Af_Inet;
                addr.sin_port = nn::socket::InetHtons(PortNumber);
                addr.sin_addr.S_addr = nn::socket::InetHtonl(dst.raw);

                // 送信データを生成します。
                ++node.unicastSendCount;
                Frame frame = { };
                frame.counter = nn::socket::InetHtonl(node.unicastSendCount);
                frame.isUnicast = true;
                frame.dataSize = nn::socket::InetHtons(static_cast<uint16_t>(dataSize));
                frame.hash = CalculateHash(frame);

                // ユニャストパケットを送信します。
                auto ret = nn::socket::SendTo(
                    status.socket, &frame, g_TestConfig.payloadSize, nn::socket::MsgFlag::Msg_None,
                    reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr));
                if (ret < 0)
                {
                    nn::socket::Errno error = nn::socket::GetLastError();
                    if (error != nn::socket::Errno::ENoBufs)
                    {
                        NNT_LDN_LOG_WARN("failed to send: errno = %d\n", error);
                        break;
                    }
                    else
                    {
                        NNT_LDN_LOG_DEBUG("failed to send: nn::socket::Errno::ENoBufs\n");
                    }
                }
            }

            // パケットレート調整のためスリープします。
            auto elapsed = (nn::os::GetSystemTick() - startedAt).ToTimeSpan();
            auto diff = interval - elapsed;
            if (0 < diff)
            {
                nn::os::SleepThread(diff);
            }
        }
        NNT_LDN_LOG_DEBUG("unicast send thread finished\n");
    }

    void ReceiveThread(void* arg) NN_NOEXCEPT
    {
        auto& status = *static_cast<TestStatus*>(arg);

        NNT_LDN_LOG_DEBUG("receive thread started\n");

        // 失敗するまでパケットを受信し続けます。
        while (status.scene == TestScene_Connected || status.scene == TestScene_NetworkCreated)
        {
            // 任意のステーションからパケットを受信するためのアドレスを生成します。
            nn::socket::SockAddrIn addr;
            addr.sin_family = nn::socket::Family::Af_Inet;
            addr.sin_port = nn::socket::InetHtons(PortNumber);
            addr.sin_addr.S_addr = nn::socket::InAddr_Any;

            // パケットの受信まで待機します。
            Frame frame;
            nn::socket::SockLenT length = sizeof(addr);
            auto size = nn::socket::RecvFrom(
                status.socket, &frame, sizeof(frame), nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr), &length);
            if (size < 0)
            {
                nn::socket::Errno error = nn::socket::GetLastError();
                NNT_LDN_LOG_WARN("failed to receive: errno = %d\n", error);
                break;
            }

            // フレームの内容が正しくない場合には無視します。
            if (!VerifyAndRestore(&frame, size))
            {
                NNT_LDN_LOG_WARN("received invalid frame\n");
                continue;
            }

            // 受信元を特定します。未知の受信元の場合には無視します。
            int srcId;
            nn::ldn::Ipv4Address src = { nn::socket::InetNtohl(addr.sin_addr.S_addr) };
            {
                std::lock_guard<nn::os::Mutex> lock(g_NetworkMutex);
                for (srcId = 0; srcId < nn::ldn::NodeCountMax; ++srcId)
                {
                    const auto& node = g_Network.ldn.nodes[srcId];
                    if (node.isConnected && node.ipv4Address == src)
                    {
                        break;
                    }
                }
            }
            if (nn::ldn::NodeCountMax <= srcId)
            {
                NNT_LDN_LOG_WARN("received from unknown source: %08X\n", addr.sin_addr.S_addr);
            }

            // 受信データに基づいてテスト結果を更新します。
            auto& info = status.nodes[srcId];
            auto& result = status.result.nodes[srcId];
            if (frame.isUnicast)
            {
                int diff = frame.counter - info.unicastCurrentIndex;
                int loss = diff - 1;
                if (0 < diff)
                {
                    info.unicastCurrentIndex = frame.counter;
                    ++result.unicastReceiveCount;
                    result.unicastLossCount += loss;
                    result.unicastBurstLossCount = std::max(loss, result.unicastBurstLossCount);
                }
                else
                {
                    NNT_LDN_LOG_WARN("unicast diff = %d\n", diff);
                }
            }
            else if (info.broadcastStartIndex == 0)
            {
                info.broadcastStartIndex = frame.counter;
                info.broadcastCurrentIndex = frame.counter;
                result.broadcastReceiveCount = 1;
            }
            else
            {
                int diff = frame.counter - info.broadcastCurrentIndex;
                int loss = diff - 1;
                if (0 < diff)
                {
                    info.broadcastCurrentIndex = frame.counter;
                    ++result.broadcastReceiveCount;
                    result.broadcastLossCount += loss;
                    result.broadcastBurstLossCount = std::max(loss, result.broadcastBurstLossCount);
                }
                else
                {
                    NNT_LDN_LOG_WARN("broadcast diff = %d\n", diff);
                }
            }
        }
        NNT_LDN_LOG_DEBUG("receive thread finished\n");
    }

    void StartCommunication(TestStatus* pStatus) NN_NOEXCEPT
    {
        // UDP 通信に使用するソケットを生成します。
        pStatus->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        ASSERT_NE(nn::socket::InvalidSocket, pStatus->socket);

        // ソケットを Bind します。
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(PortNumber);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;
        ASSERT_EQ(0, nn::socket::Bind(
            pStatus->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)));

        // ブロードキャスト通信を有効化します。
        int isEnabled = 1;
        nn::socket::SetSockOpt(
            pStatus->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));

        // データ送受信を開始します。
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
            &g_ReceiveThread, ReceiveThread, pStatus,
            g_ReceiveThreadStack, sizeof(g_ReceiveThreadStack),
            ReceiveThreadPriority));
        nn::os::StartThread(&g_ReceiveThread);
        if (0 < g_TestConfig.broadcastRate)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                &g_BroadcastSendThread, BroadcastSendThread, pStatus,
                g_BroadcastSendThreadStack, sizeof(g_BroadcastSendThreadStack),
                BroadcastSendThreadPriority));
            nn::os::StartThread(&g_BroadcastSendThread);
        }
        if (0 < g_TestConfig.unicastRate)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                &g_UnicastSendThread, UnicastSendThread, pStatus,
                g_UnicastSendThreadStack, sizeof(g_UnicastSendThreadStack),
                UnicastSendThreadPriority));
            nn::os::StartThread(&g_UnicastSendThread);
        }
    }

    void StopCommunication(TestStatus* pStatus) NN_NOEXCEPT
    {
        // データの送信を終了します。
        nn::socket::Shutdown(pStatus->socket, nn::socket::ShutdownMethod::Shut_RdWr);
        nn::socket::Close(pStatus->socket);
        pStatus->socket = static_cast<int32_t>(nn::socket::InvalidSocket);

        // スレッドを停止します。
        if (0 < g_TestConfig.unicastRate)
        {
            nn::os::WaitThread(&g_UnicastSendThread);
            nn::os::DestroyThread(&g_UnicastSendThread);
        }
        if (0 < g_TestConfig.broadcastRate)
        {
            nn::os::WaitThread(&g_BroadcastSendThread);
            nn::os::DestroyThread(&g_BroadcastSendThread);
        }
        nn::os::WaitThread(&g_ReceiveThread);
        nn::os::DestroyThread(&g_ReceiveThread);
    }

    void UpdateSetup(TestStatus* pStatus) NN_NOEXCEPT
    {
        // LDN ライブラリを初期化します。
        NNT_LDN_LOG_DEBUG("initializing the ldn library...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
        SetLdnSettings();

        // socket ライブラリを初期化します。
        NNT_LDN_LOG_DEBUG("initializing the socket library...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        // 状態変化を通知するイベントを取得します。
        nn::ldn::AttachStateChangeEvent(g_StateChangeEvent.GetBase());

        // アクセスポイントとステーションの分岐です。
        if (g_TestConfig.nodeIndex == 0)
        {
            NNT_LDN_LOG_DEBUG("starting accesspoint mode...\n");
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
            pStatus->scene = TestScene_CreatingNetwork;
        }
        else
        {
            NNT_LDN_LOG_DEBUG("starting station mode...\n");
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
            pStatus->scene = TestScene_Scanning;
        }
    }

    void UpdateCreatingNetwork(TestStatus* pStatus) NN_NOEXCEPT
    {
        // ネットワークのパラメータを適当に設定します。
        nn::ldn::NetworkConfig network = { };
        network.intentId = nn::ldn::MakeIntentId(LocalCommunicationId, g_TestConfig.sceneId);
        network.nodeCountMax = static_cast<int8_t>(g_TestConfig.nodeCount);
        network.channel = g_TestConfig.channel;
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(g_TestConfig.isEncrypted ?
            nn::ldn::SecurityMode_Product : nn::ldn::SecurityMode_Debug);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

        // ネットワークを構築します。
        NNT_LDN_LOG_DEBUG("creating a network...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // ネットワークの情報を取得します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&g_Network));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&pStatus->address, &pStatus->mask));
        NNT_LDN_LOG_INFO("Channel      : %d\n", g_Network.common.channel);
        NNT_LDN_LOG_INFO("SSID         : %s\n", g_Network.common.ssid.raw);

        // 次のシーンへの遷移です。
        pStatus->scene = TestScene_NetworkCreated;

        // データフレームの送信を開始します。
        StartCommunication(pStatus);
    }

    void UpdateNetworkCreated(TestStatus* pStatus) NN_NOEXCEPT
    {
        const int interval = 1;

        // 最新のネットワーク情報を取得します。
        if (g_StateChangeEvent.TryWait())
        {
            std::lock_guard<nn::os::Mutex> lock(g_NetworkMutex);
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&g_Network));
        }

        // テストの進捗を一定時間毎に表示します。
        if (pStatus->lastElapsed + interval <= pStatus->elapsed)
        {
            pStatus->lastElapsed = pStatus->elapsed;
            PrintTestResult(*pStatus);
        }
    }

    void UpdateScanning(TestStatus* pStatus) NN_NOEXCEPT
    {
        // スキャンを開始します。
        NNT_LDN_LOG_DEBUG("scanning...\n");

        // IntentID が一致するネットワークをスキャン対象とします。
        int count;
        nn::ldn::ScanFilter filter = { };
        filter.networkId.intentId = nn::ldn::MakeIntentId(LocalCommunicationId, g_TestConfig.sceneId);
        filter.flag = static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_IntentId);

        // ネットワークをスキャンします。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
            g_ScanBuffer, &count, nn::ldn::ScanResultCountMax,
            filter, g_TestConfig.channel));

        // ネットワークを発見したら次のテストに移ります。
        if (0 < count)
        {
            g_Network = g_ScanBuffer[0];
            NNT_LDN_LOG_INFO("Channel      : %d\n", g_Network.common.channel);
            NNT_LDN_LOG_INFO("SSID         : %s\n", g_Network.common.ssid.raw);
            pStatus->scene = TestScene_Connecting;
        }
    }

    void UpdateConnecting(TestStatus* pStatus) NN_NOEXCEPT
    {
        NNT_LDN_LOG_DEBUG("connecting...\n");

        // ネットワークへの接続に必要なパラメータを生成します。
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(g_TestConfig.isEncrypted ?
            nn::ldn::SecurityMode_Product : nn::ldn::SecurityMode_Debug);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

        // ネットワークへの接続を試行します。
        nn::Result result = nn::ldn::Connect(
            g_Network, security, user, 0, nn::ldn::ConnectOption_None);
        if (result.IsSuccess())
        {
            if (nn::ldn::GetNetworkInfo(&g_Network).IsSuccess() &&
                nn::ldn::GetIpv4Address(&pStatus->address, &pStatus->mask).IsSuccess())
            {
                pStatus->scene = TestScene_Connected;
                NNT_LDN_LOG_DEBUG("connection established.\n");
                StartCommunication(pStatus);
            }
            else
            {
                NNT_LDN_LOG_WARN("invalid state\n");
            }
        }
        else
        {
            if (nn::ldn::ResultNetworkNotFound::Includes(result))
            {
                NNT_LDN_LOG_WARN("network not found\n");
            }
            else if (nn::ldn::ResultConnectionTimeout::Includes(result))
            {
                NNT_LDN_LOG_WARN("connection timeout\n");
            }
            else if (nn::ldn::ResultConnectionRejected::Includes(result))
            {
                NNT_LDN_LOG_WARN("connection rejected\n");
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }

            // スキャンからやり直します。
            pStatus->scene = TestScene_Scanning;
        }
    }

    void UpdateConnected(TestStatus* pStatus) NN_NOEXCEPT
    {
        const int interval = 1;

        // 最新のネットワーク情報を取得します。
        if (g_StateChangeEvent.TryWait())
        {
            std::lock_guard<nn::os::Mutex> lock(g_NetworkMutex);
            if (nn::ldn::GetNetworkInfo(&g_Network).IsFailure())
            {
                NNT_LDN_LOG_WARN("disconnected\n");
                pStatus->scene = TestScene_Connecting;
            }
        }

        // テストの進捗を一定時間毎に表示します。
        if (pStatus->lastElapsed + interval <= pStatus->elapsed)
        {
            pStatus->lastElapsed = pStatus->elapsed;
            PrintTestResult(*pStatus);
        }
    }

    void UpdateCleanup(TestStatus* pStatus) NN_NOEXCEPT
    {
        // LDN ライブラリを終了します。
        NNT_LDN_LOG_DEBUG("cleaning up...\n");
        nn::ldn::FinalizeSystem();

        // データ通信を終了します。
        if (pStatus->socket != nn::socket::InvalidSocket)
        {
            StopCommunication(pStatus);
        }
        pStatus->scene = TestScene_Finalized;
    }

} // namespace <unnamed>

//
// 接続・切断関係のエイジングです。
//
TEST(Integration, Udp)
{
    // テストの設定を表示します。
    NNT_LDN_LOG_INFO("Scene Id     : %d\n", g_TestConfig.sceneId);
    NNT_LDN_LOG_INFO("Node Type    : %s\n", g_TestConfig.nodeIndex == 0 ? "AP" : "STA");
    NNT_LDN_LOG_INFO("Node Index   : %d/%d\n", g_TestConfig.nodeIndex, g_TestConfig.nodeCount);
    NNT_LDN_LOG_INFO("Encryption   : %s\n", g_TestConfig.isEncrypted ? "enabled" : "disabled");
    NNT_LDN_LOG_INFO("Unicast      : %d packets/sec\n", g_TestConfig.unicastRate);
    NNT_LDN_LOG_INFO("Broadcast    : %d packets/sec\n", g_TestConfig.broadcastRate);
    NNT_LDN_LOG_INFO("Channel      : %d\n", g_TestConfig.channel);
    NNT_LDN_LOG_INFO_WITHOUT_PREFIX("\n");

    // テストの状態を初期化しておきます。
    TestStatus status;
    std::memset(&status, 0, sizeof(status));
    status.socket = static_cast<int32_t>(nn::socket::InvalidSocket);

    // 他の開発機と同期します。
    NNT_LDN_LOG_DEBUG("synchronizing...\n");
    NNT_LDN_SYNC("Aging.Start");

    // テスト開始時点の時刻を取得しておきます。
    nn::os::Tick startedAt = nn::os::GetSystemTick();

    // ネットワークの構築と破棄を繰り返します。
    while (status.scene != TestScene_Finalized)
    {
        // テストが始まってからの経過時間を取得します。
        auto now = nn::os::GetSystemTick();
        status.elapsed = static_cast<int>((now - startedAt).ToTimeSpan().GetSeconds());
        if (g_TestConfig.duration <= status.elapsed)
        {
            status.scene = TestScene_Cleanup;
        }

        // シーンに応じた処理です。
        switch (status.scene)
        {
        case TestScene_Setup:
            UpdateSetup(&status);
            break;
        case TestScene_CreatingNetwork:
            UpdateCreatingNetwork(&status);
            break;
        case TestScene_NetworkCreated:
            UpdateNetworkCreated(&status);
            break;
        case TestScene_Scanning:
            UpdateScanning(&status);
            break;
        case TestScene_Connecting:
            UpdateConnecting(&status);
            break;
        case TestScene_Connected:
            UpdateConnected(&status);
            break;
        case TestScene_Cleanup:
            UpdateCleanup(&status);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // 60FPS 程度になるようにスリープします。
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // 全員分のユニキャストおよびブロードキャストを問題無く受信できているかを確認します。
    int unicast = 0;
    int broadcast = 0;
    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
    {
        const auto& result = status.result.nodes[i];
        if (0 < result.unicastReceiveCount)
        {
            ++unicast;
        }
        if (0 < result.broadcastReceiveCount)
        {
            ++broadcast;
        }
    }
    ASSERT_EQ(g_TestConfig.nodeCount - 1, unicast);
    ASSERT_EQ(g_TestConfig.nodeCount - 1, broadcast);
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::ldn::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::ldn::CommandLineOptionFlag_NodeCount |
        nnt::ldn::CommandLineOptionFlag_NodeIndex |
        nnt::ldn::CommandLineOptionFlag_Channel |
        nnt::ldn::CommandLineOptionFlag_Encryption |
        nnt::ldn::CommandLineOptionFlag_Duration |
        nnt::ldn::CommandLineOptionFlag_UnicastRate |
        nnt::ldn::CommandLineOptionFlag_BroadcastRate |
        nnt::ldn::CommandLineOptionFlag_SceneId |
        nnt::ldn::CommandLineOptionFlag_PayloadSize |
        nnt::ldn::CommandLineOptionFlag_OperationMode |
        nnt::ldn::CommandLineOptionFlag_WirelessControllerRestriction);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nn::ldn::NodeCountMax;
    setting.defaultSceneId = SceneId;

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::ldn::Parse(&g_TestConfig, setting, argc, argv);

    // 他の開発機と同期する準備です。
    int result;
    if (g_TestConfig.nodeIndex == 0)
    {
        g_pServer = new nnt::ldn::HtcsSynchronizationServer(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        result = g_pServer->CreateServer("nn::ldn::IntegrationUdp", g_TestConfig.nodeCount - 1);
        g_pSync = g_pServer;
    }
    else
    {
        g_pClient = new nnt::ldn::HtcsSynchronizationClient(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        result = g_pClient->Connect("nn::ldn::IntegrationUdp");
        g_pSync = g_pClient;
    }

    // テストを実行します。
    int exitCode;
    if (result == nnt::ldn::SynchronizationResult_Success)
    {
        exitCode = RUN_ALL_TESTS();
    }
    else
    {
        NNT_LDN_LOG_ERROR("failed to sync: %d\n", result);
        exitCode = result;
    }
    delete g_pSync;
    nnt::Exit(exitCode);
}
