﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/ldn.h>
#include <nn/ldn/ldn_PrivateApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnt.h>
#include <nnt/ldn/testLdn_Assert.h>
#include <nnt/ldn/testLdn_CommandLineParser.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_Log.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>
#include "../Common/testLdn_Definitions.h"

namespace
{
    //! テストの設定値です。
    nnt::ldn::TestConfig g_TestConfig;

    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronizationClient* g_pClient;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
    #define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pClient->Synchronize(keyword))

    /**
    * @brief       テストケースに対応するネットワークを探索します。
    * @param[out]  pOutNetwork     見つかったネットワークです。
    * @param[in]   channel         無線チャンネルです。
    * @param[in]   timeout         タイムアウト時間です。
    * @return      処理の成否です。
    */
    nn::Result TimedScan(
        nn::ldn::NetworkInfo* pOutNetwork, int channel,
        const nn::ldn::ScanFilter& filter, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        int count;
        return nnt::ldn::TimedScan(pOutNetwork, &count, 1, filter, channel, timeout);
    }

    /**
     * @brief       テストケースに対応するネットワークを探索します。
     * @param[out]  pOutNetwork     見つかったネットワークです。
     * @param[in]   channel         無線チャンネルです。
     * @param[in]   timeout         タイムアウト時間です。
     * @return      処理の成否です。
     */
    nn::Result TimedScan(
        nn::ldn::NetworkInfo* pOutNetwork, int channel, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // IntentID をフィルタとして使用します。
        nn::ldn::ScanFilter filter = { };
        filter.networkId.intentId = nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId);
        filter.flag = static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_IntentId);

        // ネットワークを探索します。
        return TimedScan(pOutNetwork, channel, filter, timeout);
    }

    void SetLdnSettings() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetWirelessControllerRestriction(
            static_cast<nn::ldn::WirelessControllerRestriction>(
                g_TestConfig.wirelessControllerRestriction)));
    }

} // namespace <unnamed>

//
// パスフレーズが異なるステーションが接続できないことを確認します。
//
TEST(Restrict, DifferentKey)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.DifferentKey.Start");

    // このテストは 2 台以上のノードが存在しなければ実行できません。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= 1)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // パスフレーズの末尾を書き換えます。
    security.passphrase[security.passphraseSize - 1] ^= 1;

    // ネットワークを探索します。
    NNT_LDN_SYNC("Restrict.DifferentKey.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo,  g_TestConfig.channel, ScanTimeout));

    // ネットワークへの接続を試行します。
    NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultConnectionFailed, nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    NNT_LDN_SYNC("Restrict.DifferentKey.Connected");

    // ネットワークの破棄まで待機します。
    NNT_LDN_SYNC("Restrict.DifferentKey.NetworkDestroyed");
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// 接続台数制限の確認です。ノードの最大接続数を 1 ～ 7 台に制限します。
//
TEST(Restrict, NodeCount)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.NodeCount.Start");

    // 少なくとも 1 台のステーションが接続できない nodeCountMax を計算します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount) - 1;
    NN_UNUSED(nodeCountMax);

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("Restrict.NodeCount.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークへの接続を試行します。
    nn::Result result = nnt::ldn::TimedConnect(networkInfo, security, user, 0, ConnectionTimeout);
    if (nn::ldn::ResultConnectionRejected::Includes(result))
    {
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NNT_LDN_SYNC("Restrict.NodeCount.Connected");

    // 接続に成功した場合には切断まで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, DestroyNetworkTimeout);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// 接続台数制限の確認です。ノードの最大接続数は 8 台です。
//
TEST(Restrict, NodeCountMax)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.NodeCountMax.Start");

    // このテストは 9 台以上のノードが存在しなければ実行できません。
    int nodeCountMax = nn::ldn::NodeCountMax;
    if (g_TestConfig.nodeCount <= nodeCountMax)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("Restrict.NodeCountMax.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークへの接続を試行します。
    nn::Result result = nnt::ldn::TimedConnect(networkInfo, security, user, 0, ConnectionTimeout);
    if (nn::ldn::ResultConnectionRejected::Includes(result))
    {
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NNT_LDN_SYNC("Restrict.NodeCountMax.Connected");

    // 接続に成功した場合には切断まで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, DestroyNetworkTimeout);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// AcceptPolicy による接続制限です。
//
TEST(Restrict, AcceptPolicyAlwaysReject)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Start");

    // 最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークへの接続を試行します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Connected");

    // ポリシーの更新まで待機します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.AcceptPolicyUpdated");

    // アクセスポイントの指示によって切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, RejectTimeout);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Rejected");

    // 再度接続を試行します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::ldn::ResultConnectionRejected,
        nnt::ldn::TimedConnect(networkInfo, security, user, 0, ConnectionTimeout));
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.Reconnected");
    NNT_LDN_SYNC("Restrict.AcceptPolicyAlwaysReject.NetworkDestroyed");
}

//
// AcceptPolicy による接続制限です。
//
TEST(Restrict, AcceptPolicyBlackList)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.Start");

    // 最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークへの接続を試行します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.Connected");

    // ポリシーの変更まで待機します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.AcceptPolicyUpdated");

    // アクセスポイントの指示によって切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, RejectTimeout);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.Disconnected");

    // 再度接続を試行します。
    nn::Result result = nnt::ldn::TimedConnect(networkInfo, security, user, 0, ConnectionTimeout);
    if (nn::ldn::ResultConnectionRejected::Includes(result))
    {
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NNT_LDN_SYNC("Restrict.AcceptPolicyBlackList.Reconnected");

    // 接続に成功した場合、切断まで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, DestroyNetworkTimeout);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// AcceptPolicy による接続制限です。
//
TEST(Restrict, AcceptPolicyWhiteList)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.Start");

    // 最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークへの接続を試行します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.Connected");

    // ポリシーの更新まで待機します。
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.AcceptPolicyUpdated");

    // 接続に成功した場合には切断まで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, RejectTimeout);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.Disconnected");

    // 再度接続を試行します。
    nn::Result result = nnt::ldn::TimedConnect(networkInfo, security, user, 0, ConnectionTimeout);
    if (nn::ldn::ResultConnectionRejected::Includes(result))
    {
    }
    else
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NNT_LDN_SYNC("Restrict.AcceptPolicyWhiteList.Reconnected");

    // 接続に成功した場合、切断まで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, DestroyNetworkTimeout);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// アクセスポイントがネットワークを破棄した時にイベントがシグナルされることを検証します。
//
TEST(StateChangeEvent, DestroyNetwork)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続し、イベントがシグナルされていることを確認します。
    ASSERT_FALSE(stateChangeEvent.TryWait());
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.AcceptPolicyUpdated");
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

    // 他のステーションの接続時にイベントが発生することを確認します。
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    stateChangeEvent.TryWait();
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.AdvertiseReceived");

    // 切断まで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, DestroyNetworkTimeout);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// ステーションの自発的な切断時にイベントがシグナルされることを検証します。
//
TEST(StateChangeEvent, Disconect)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("StateChangeEvent.Disconect.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("StateChangeEvent.Disconect.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続し、イベントがシグナルされていることを確認します。
    ASSERT_FALSE(stateChangeEvent.TryWait());
    NNT_LDN_SYNC("StateChangeEvent.Disconect.AcceptPolicyUpdated");
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

    // 他のステーションの接続時にイベントが発生することを確認します。
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    stateChangeEvent.TryWait();
    NNT_LDN_SYNC("StateChangeEvent.Disconect.AdvertiseReceived");

    // 他のステーションが切断した際にイベントが発生することを確認します。
    // 切断はインデックスの大きいステーションから順に実行します。
    while (g_TestConfig.nodeIndex + 1 < networkInfo.ldn.nodeCount)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, DisconnectTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());

    // 自発的にアクセスポイントから切断し、イベントが発生することを確認します。
    stateChangeEvent.TryWait();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Disconnect());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    NNT_LDN_SYNC("StateChangeEvent.Disconect.DestroyReady");
}

//
// アクセスポイントから切断された時にイベントがシグナルされることを検証します。
//
TEST(StateChangeEvent, Reject)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("StateChangeEvent.Reject.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("StateChangeEvent.Reject.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続し、イベントがシグナルされていることを確認します。
    ASSERT_FALSE(stateChangeEvent.TryWait());
    NNT_LDN_SYNC("StateChangeEvent.Reject.AcceptPolicyUpdated");
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_TRUE(stateChangeEvent.TryWait());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

    // 他のステーションの接続時にイベントが発生することを確認します。
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    }
    stateChangeEvent.TryWait();
    NNT_LDN_SYNC("StateChangeEvent.DestroyNetwork.AdvertiseReceived");

    // アクセスポイントがステーションを切断した際にイベントが発生することを確認します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// 初期状態では Advertise に空のユーザデータが登録されていることを検証します。
//
TEST(Advertise, UserDataEmpty)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.UserDataEmpty.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
        nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // ネットワーク接続に使用するパラメータを生成します。
        nn::ldn::SecurityConfig security = {};
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = {};
        std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

        // ネットワークを探索します。
        NNT_LDN_SYNC("Advertise.UserDataEmpty.NetworkCreated");
        nn::ldn::NetworkInfo networkInfo;
        NNT_ASSERT_RESULT_SUCCESS(TimedScan(
            &networkInfo, g_TestConfig.channel, ScanTimeout));

        // 空のユーザデータが設定されています。
        ASSERT_EQ(0U, networkInfo.ldn.advertiseDataSize);

        // ネットワークに接続します。
        NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
            networkInfo, security, user, 0, ConnectionTimeout));
        ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
        NNT_LDN_SYNC("Advertise.UserDataEmpty.Connected");

        // アドバータイズデータの受信まで待機します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        ASSERT_EQ(0U, networkInfo.ldn.advertiseDataSize);
        NNT_LDN_SYNC("Advertise.UserDataEmpty.AdvertiseReceived");

        // ネットワークから切断されるまで待機します。
        while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
        {
            NNT_LDN_ASSERT_SIGNALED_WEAK(
                stateChangeEvent, ConnectionTimeout + AdvertisingTime);
        }
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    }
}

//
// Advertise を正しく更新できることを検証します。
//
TEST(Advertise, UserDataUpdate)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.UserDataUpdate.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
        nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // ネットワーク接続に使用するパラメータを生成します。
        nn::ldn::SecurityConfig security = {};
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = {};
        std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

        // ネットワークを探索します。
        nn::ldn::NetworkInfo networkInfo;
        NNT_LDN_SYNC("Advertise.UserDataUpdate.NetworkCreated");
        NNT_ASSERT_RESULT_SUCCESS(TimedScan(
            &networkInfo, g_TestConfig.channel, ScanTimeout));

        // 最小サイズのユーザデータが設定されています。
        ASSERT_EQ(1U, networkInfo.ldn.advertiseDataSize);
        ASSERT_EQ(0xA5, networkInfo.ldn.advertiseData[0]);

        // ネットワークに接続します。
        NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
            networkInfo, security, user, 0, ConnectionTimeout));
        ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        ASSERT_EQ(1U, networkInfo.ldn.advertiseDataSize);
        ASSERT_EQ(0xA5, networkInfo.ldn.advertiseData[0]);
        NNT_LDN_SYNC("Advertise.UserDataUpdate.Connected");

        // アドバータイズデータの更新まで待機します。
        NNT_LDN_SYNC("Advertise.UserDataUpdate.AdvertiseUpdated");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        ASSERT_EQ(nn::ldn::AdvertiseDataSizeMax, networkInfo.ldn.advertiseDataSize);
        for (size_t i = 0; i < nn::ldn::AdvertiseDataSizeMax; ++i)
        {
            ASSERT_EQ((i + 1) & 0xFF, networkInfo.ldn.advertiseData[i]);
        }

        // ネットワークから切断されるまで待機します。
        NNT_LDN_SYNC("Advertise.UserDataUpdate.AdvertiseReceived");
        while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
        {
            NNT_LDN_ASSERT_SIGNALED_WEAK(
                stateChangeEvent, ConnectionTimeout + AdvertisingTime);
        }
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    }
}

//
// Advertise で前回のユーザデータを引き継ぐことを確認します。
//
TEST(Advertise, UserDataKeep)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.UserDataKeep.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
                                           nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // ネットワーク接続に使用するパラメータを生成します。
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

        // 前回のユーザデータを引き継ぐことを確認します。
        {
            // ネットワークを探索します。
            nn::ldn::NetworkInfo networkInfo;
            NNT_LDN_SYNC("Advertise.UserDataKeep.NetworkCreated");
            NNT_ASSERT_RESULT_SUCCESS(TimedScan(
                &networkInfo, g_TestConfig.channel, ScanTimeout));

            // 最大サイズのユーザデータが設定されています。
            ASSERT_EQ(nn::ldn::AdvertiseDataSizeMax, networkInfo.ldn.advertiseDataSize);
            for (size_t i = 0; i < nn::ldn::AdvertiseDataSizeMax; ++i)
            {
                ASSERT_EQ((i + 1) & 0xFF, networkInfo.ldn.advertiseData[i]);
            }

            // ネットワークに接続します。
            NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
                networkInfo, security, user, 0, ConnectionTimeout));
            ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

            // 他のステーションの接続まで待機します。
            while (networkInfo.ldn.nodeCount < nodeCountMax)
            {
                NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
                NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
            }
            ASSERT_EQ(nn::ldn::AdvertiseDataSizeMax, networkInfo.ldn.advertiseDataSize);
            for (size_t i = 0; i < nn::ldn::AdvertiseDataSizeMax; ++i)
            {
                ASSERT_EQ((i + 1) & 0xFF, networkInfo.ldn.advertiseData[i]);
            }
            NNT_LDN_SYNC("Advertise.UserDataKeep.Connected");

            // ネットワークから切断されるまで待機します。
            while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
            {
                NNT_LDN_ASSERT_SIGNALED_WEAK(
                    stateChangeEvent, ConnectionTimeout + AdvertisingTime);
            }
            ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        }
    }
}

//
// Advertise でノード情報が正しく配信されることを検証します。
//
TEST(Advertise, NodeInfo)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.NodeInfo.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
                                           nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // ネットワーク接続に使用するパラメータを生成します。
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));

        // 適当な名前でネットワークに接続します。
        {
            // ネットワークを探索します。
            NNT_LDN_SYNC("Advertise.NodeInfo.NetworkCreated");
            nn::ldn::NetworkInfo networkInfo;
            NNT_ASSERT_RESULT_SUCCESS(TimedScan(
                &networkInfo, g_TestConfig.channel, ScanTimeout));

            // AP のユーザ名を確認します。
            ASSERT_STREQ(ApUserName, networkInfo.ldn.nodes[0].userName);

            // ネットワークに接続します。
            nn::ldn::UserConfig user = { };
            std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);
            NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
                networkInfo, security, user, 0, ConnectionTimeout));
            ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());

            // 自身の IPv4 アドレスを取得します。
            nn::ldn::Ipv4Address address;
            nn::ldn::SubnetMask  mask;
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&address, &mask));
            nn::ldn::Ipv4Address netAddress = nn::ldn::MakeNetworkAddress(address, mask);
            NNT_LDN_SYNC("Advertise.NodeInfo.Connected");

            // 最新のアドバータイズを受信するため、一時待機します。
            nn::os::SleepThread(AdvertisingTime);
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
            NNT_LDN_SYNC("Advertise.NodeInfo.AdvertiseReceived");

            // 全員のユーザ名を確認します。
            const auto& ap = networkInfo.ldn.nodes[0];
            ASSERT_TRUE(ap.isConnected);
            ASSERT_EQ(0, ap.nodeId);
            ASSERT_STREQ(ApUserName, networkInfo.ldn.nodes[0].userName);
            ASSERT_NE(address, ap.ipv4Address);
            ASSERT_EQ(netAddress, nn::ldn::MakeNetworkAddress(ap.ipv4Address, mask));
            ASSERT_NE(nn::ldn::ZeroMacAddress, ap.macAddress);
            ASSERT_NE(nn::ldn::BroadcastMacAddress, ap.macAddress);
            for (int i = 1; i < nodeCountMax; ++i)
            {
                const auto& node = networkInfo.ldn.nodes[i];
                ASSERT_TRUE(node.isConnected);
                ASSERT_EQ(i, node.nodeId);
                ASSERT_STREQ(StaUserName, networkInfo.ldn.nodes[i].userName);
                ASSERT_NE(ap.ipv4Address, node.ipv4Address);
                ASSERT_EQ(netAddress, nn::ldn::MakeNetworkAddress(node.ipv4Address, mask));
                ASSERT_NE(nn::ldn::ZeroMacAddress, node.macAddress);
                ASSERT_NE(nn::ldn::BroadcastMacAddress, node.macAddress);
            }
            for (int i = nodeCountMax; i < nn::ldn::NodeCountMax; ++i)
            {
                ASSERT_FALSE(networkInfo.ldn.nodes[i].isConnected);
                ASSERT_EQ(i, networkInfo.ldn.nodes[i].nodeId);
            }

            // ネットワークから切断されるまで待機します。
            while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
            {
                NNT_LDN_ASSERT_SIGNALED_WEAK(
                    stateChangeEvent, ConnectionTimeout + AdvertisingTime);
            }
            ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
        }
    }
}

//
// Advertise 配信される情報を検証します。
//
TEST(Advertise, Others)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Advertise.Others.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // 各セキュリティ・モードについてテストを実施します。
    nn::ldn::SecurityMode securities[] = { nn::ldn::SecurityMode_Debug,
                                           nn::ldn::SecurityMode_Product };
    for (auto securityMode : securities)
    {
        // ネットワーク接続に使用するパラメータを生成します。
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(securityMode);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

        // ネットワークを探索します。
        NNT_LDN_SYNC("Advertise.Others.NetworkCreated");
        nn::ldn::NetworkInfo networkInfo;
        NNT_ASSERT_RESULT_SUCCESS(TimedScan(
            &networkInfo, g_TestConfig.channel, ScanTimeout));

        // ネットワークの情報を検証します。
        ASSERT_EQ(securityMode, networkInfo.ldn.securityMode);
        ASSERT_EQ(nn::ldn::AcceptPolicy_AlwaysAccept, networkInfo.ldn.stationAcceptPolicy);
        ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
        ASSERT_GE(networkInfo.ldn.nodeCount, 1);
        ASSERT_EQ(nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId),
            networkInfo.networkId.intentId);
        ASSERT_EQ(networkInfo.ldn.nodes[0].macAddress, networkInfo.common.bssid);
        ASSERT_EQ(nn::ldn::SsidLengthMax, networkInfo.common.ssid.length);
        ASSERT_NE(0, networkInfo.common.channel);
        ASSERT_TRUE(nn::ldn::LinkLevelMin <= networkInfo.common.linkLevel &&
                    networkInfo.common.linkLevel <= nn::ldn::LinkLevelMax);
        ASSERT_EQ(nn::ldn::NetworkType_Ldn, networkInfo.common.networkType);

        // ネットワークに接続します。
        NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
            networkInfo, security, user, 0, ConnectionTimeout));
        ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
        NNT_LDN_SYNC("Advertise.Others.Connected");

        // ネットワークの情報を検証します。
        ASSERT_EQ(securityMode, networkInfo.ldn.securityMode);
        ASSERT_EQ(nn::ldn::AcceptPolicy_AlwaysAccept, networkInfo.ldn.stationAcceptPolicy);
        ASSERT_EQ(nodeCountMax, networkInfo.ldn.nodeCountMax);
        ASSERT_GE(networkInfo.ldn.nodeCount, 1);
        ASSERT_EQ(nn::ldn::MakeIntentId(LocalId, g_TestConfig.sceneId),
            networkInfo.networkId.intentId);
        ASSERT_EQ(networkInfo.ldn.nodes[0].macAddress, networkInfo.common.bssid);
        ASSERT_EQ(nn::ldn::SsidLengthMax, networkInfo.common.ssid.length);
        ASSERT_NE(0, networkInfo.common.channel);
        ASSERT_TRUE(nn::ldn::LinkLevelMin <= networkInfo.common.linkLevel &&
                    networkInfo.common.linkLevel <= nn::ldn::LinkLevelMax);
        ASSERT_EQ(nn::ldn::NetworkType_Ldn, networkInfo.common.networkType);

        // ネットワークから切断されるまで待機します。
        while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
        {
            NNT_LDN_ASSERT_SIGNALED_WEAK(
                stateChangeEvent, ConnectionTimeout + AdvertisingTime);
        }
        ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    }
}

//
// ノード接続時の NodeLatestUpdate の検証です。
//
TEST(NodeLatestUpdate, Connect)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("NodeLatestUpdate.Connect.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("NodeLatestUpdate.Connect.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

    // ステーションの接続を待ち受けます。
    int count = 0;
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
    while (count < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
        count += static_cast<int>(std::count_if(std::begin(updates), std::end(updates),
        [](const nn::ldn::NodeLatestUpdate& update)
        {
            return update.stateChange == nn::ldn::NodeStateChange_Connect;
        }));
    }
    ASSERT_EQ(nodeCountMax, count);

    // これ以上、接続状態に変化はありません。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
        &networkInfo, updates, nn::ldn::NodeCountMax));
    ASSERT_TRUE(std::all_of(std::begin(updates), std::end(updates),
                            [](const nn::ldn::NodeLatestUpdate& update)
    {
        return update.stateChange == nn::ldn::NodeStateChange_None;
    }));
    NNT_LDN_SYNC("NodeLatestUpdate.Connect.Connected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// ノード切断時の NodeLatestUpdate の検証です。
//
TEST(NodeLatestUpdate, Disconnect)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("NodeLatestUpdate.Disconnect.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("NodeLatestUpdate.Disconnect.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

    // ステーションの接続を待ち受けます。
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
    }

    // アクセスポイントに接続状態を伝えるため、一定時間待機します。
    NNT_LDN_SYNC("NodeLatestUpdate.Disconnect.Connected");

    // 他のステーションが切断した際にイベントが発生することを確認します。
    // 切断はインデックスの大きいステーションから順に実行します。
    int count = nodeCountMax;
    while (g_TestConfig.nodeIndex + 1 < count)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
        count -= static_cast<int>(std::count_if(std::begin(updates), std::end(updates),
        [](const nn::ldn::NodeLatestUpdate& update)
        {
            return update.stateChange == nn::ldn::NodeStateChange_Disconnect;
        }));
    }
    ASSERT_EQ(g_TestConfig.nodeIndex + 1, count);
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());

    // 自発的にアクセスポイントから切断し、イベントが発生することを確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Disconnect());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// ノード切断＋接続時の NodeLatestUpdate の検証です。
//
TEST(NodeLatestUpdate, DisconnectAndConnect)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));

    // ステーションの接続を待ち受けます。
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];
    while (networkInfo.ldn.nodeCount < nodeCountMax)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(stateChangeEvent, ConnectionTimeout);
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(
            &networkInfo, updates, nn::ldn::NodeCountMax));
    }

    // アクセスポイントに接続状態を伝えるため、一定時間待機します。
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.Connected");

    // 自発的にアクセスポイントから切断します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Disconnect());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // アクセスポイントに切断状態を伝えるため、一定時間待機します。
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.Disconnected");

    // すぐに再接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    NNT_LDN_SYNC("NodeLatestUpdate.DisconnectAndConnect.Reconnected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// ネットワーク再構築時に IPv4 アドレスを持ち越せていることの検証です。
//
TEST(Reconnection, ReuseIpv4Address)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.Connected");

    // IPv4 アドレスを取得します。
    nn::ldn::Ipv4Address ipv4Address;
    nn::ldn::SubnetMask mask;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&ipv4Address, &mask));

    // 他の端末の接続がすぐに反映されない可能性があるため、一定時間待機します。
    nn::os::SleepThread(AdvertisingTime);

    // ネットワークへの再接続に必要な情報を取得します。
    nn::ldn::SecurityParameter securityParam;
    nn::ldn::NetworkConfig networkConfig;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetSecurityParameter(&securityParam));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkConfig(&networkConfig));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.GetNetworkInfo");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // ネットワークへの接続を再試行します。
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.NetworkReconstructed");
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnectPrivate(
        networkConfig, security, securityParam, user, 0, ConnectionTimeout));

    // IPv4 アドレスが再利用されていることを確認します。
    nn::ldn::Ipv4Address newIpv4Address;
    nn::ldn::SubnetMask newMask;
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&newIpv4Address, &newMask));
    ASSERT_EQ(ipv4Address, newIpv4Address);
    ASSERT_EQ(mask.raw, newMask.raw);
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.Reconnected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseStation());

    // IPv4 アドレスを再利用するためのエントリを作成します。
    nn::ldn::AddressEntry entries[nn::ldn::NodeCountMax];
    int nodeCount = 0;
    for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
    {
        const auto& node = networkInfo.ldn.nodes[i];
        if (node.isConnected)
        {
            auto& entry = entries[nodeCount];
            entry.ipv4Address = node.ipv4Address;
            entry.macAddress = node.macAddress;
            ++nodeCount;
        }
    }
    ASSERT_EQ(nodeCountMax, nodeCount);

    // ステーション側でネットワークを構築し、IPv4 アドレスが再利用されることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetworkPrivate(
        networkConfig, security, securityParam, user, nodeCount, entries));
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&newIpv4Address, &newMask));
    ASSERT_EQ(ipv4Address, newIpv4Address);
    ASSERT_EQ(mask.raw, newMask.raw);
    NNT_LDN_SYNC("Reconnection.ReuseIpv4Address.Finished");
}

//
// アプリ起因のネットワーク切断時に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, DisconnectedByUser)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Connected");

    // 自身の要求によってネットワークから切断します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Disconnect());
    ASSERT_EQ(nn::ldn::DisconnectReason_DisconnectedByUser, nn::ldn::GetDisconnectReason());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Disconnected");

    // もう一度ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&networkInfo));
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Reconnected");

    // 自身の要求によってネットワークから切断します。
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseStation());
    ASSERT_EQ(nn::ldn::DisconnectReason_DisconnectedByUser, nn::ldn::GetDisconnectReason());
    ASSERT_EQ(nn::ldn::State_Initialized, nn::ldn::GetState());
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByUser.Disconnected");
}

//
// システム起因のネットワーク破棄時に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, DisconnectedByWifiOff)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByWifiOff.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::NifmInitializer nifmInitializer;
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByWifiOff.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByWifiOff.Connected");

    // 無線オフによってネットワークから切断します。
    nnt::ldn::ScopedWirelessCommunicationSwitchSetter setter(false);
    ASSERT_EQ(nn::ldn::DisconnectReason_DisconnectedBySystem, nn::ldn::GetDisconnectReason());
    ASSERT_EQ(nn::ldn::State_Error, nn::ldn::GetState());
    NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultWifiOff, nn::ldn::OpenStation());
    NNT_LDN_SYNC("DisconnectReason.DisconnectedByWifiOff.Disconnected");
}

//
// アプリ起因のネットワーク破棄時に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, DestroyedByUser)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.Connected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::DisconnectReason_DestroyedByUser, nn::ldn::GetDisconnectReason());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // もう一度ネットワークに接続します。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.NetworkReconstructed");
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_LDN_SYNC("DisconnectReason.DestroyedByUser.Reconnected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::DisconnectReason_DestroyedByUser, nn::ldn::GetDisconnectReason());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// システム起因のネットワーク破棄時に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, DestroyedByWifiOff)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByWifiOff.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("DisconnectReason.DestroyedByWifiOff.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_LDN_SYNC("DisconnectReason.DestroyedByWifiOff.Connected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::DisconnectReason_DestroyedBySystem, nn::ldn::GetDisconnectReason());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// AP 側のアプリから STA を切断する場合に切断理由を正しく取得できることを確認します。
//
TEST(DisconnectReason, Rejected)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("DisconnectReason.Rejected.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("DisconnectReason.Rejected.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));

    // ネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_LDN_SYNC("DisconnectReason.Rejected.Connected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::DisconnectReason_Rejected, nn::ldn::GetDisconnectReason());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// ローカル通信バージョンの比較テストです。。
//
TEST(LocalCommunicationVersion, Compare)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("LocalCommunicationVersion.Compare.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = { };
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = { };
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("LocalCommunicationVersion.Compare.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, ScanTimeout));
    int version = networkInfo.ldn.nodes[0].localCommunicationVersion;
    ASSERT_EQ(1024, version);

    // アクセスポイントと異なるローカル通信バージョンでネットワークに接続します。
    NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultLowerVersion, nnt::ldn::TimedConnect(
        networkInfo, security, user, version - 1, ConnectionTimeout));
    NNT_ASSERT_RESULT_FAILURE(nn::ldn::ResultHigherVersion, nnt::ldn::TimedConnect(
        networkInfo, security, user, version + 1, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    // アクセスポイントと一致するローカル通信バージョンでネットワークに接続します。
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 1024, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_LDN_SYNC("LocalCommunicationVersion.Compare.Connected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// デフォルトのローカル通信識別子に関するテストです。
//
TEST(LocalCommunicationId, Default)
{
    // テストケース開始前の同期です。
    NNT_LDN_SYNC("LocalCommunicationId.Default.Start");

    // 最大台数のステーションを接続させます。最大接続台数を超えている場合には接続を自粛します。
    int nodeCountMax = std::min<int>(nn::ldn::NodeCountMax, g_TestConfig.nodeCount);
    if (nodeCountMax <= g_TestConfig.nodeIndex)
    {
        return;
    }

    // STA としての動作を開始します。
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());

    // 状態変化を監視するイベントを取得します。
    nn::os::SystemEvent stateChangeEvent;
    nn::ldn::AttachStateChangeEvent(stateChangeEvent.GetBase());

    // ネットワーク接続に使用するパラメータを生成します。
    nn::ldn::SecurityConfig security = {};
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    nn::ldn::UserConfig user = {};
    std::strncpy(user.userName, StaUserName, nn::ldn::UserNameBytesMax);

    // ネットワークを探索します。
    NNT_LDN_SYNC("LocalCommunicationId.Default.NetworkCreated");
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::ScanFilter filter;
    const auto defaultIntentId = nn::ldn::MakeIntentId(
        nn::ldn::DefaultLocalCommunicationId, g_TestConfig.sceneId);
    filter.flag = nn::ldn::ScanFilterFlag_IntentId;
    filter.networkId.intentId = defaultIntentId;
    NNT_ASSERT_RESULT_SUCCESS(TimedScan(
        &networkInfo, g_TestConfig.channel, filter, ScanTimeout));
    ASSERT_EQ(LocalId, networkInfo.networkId.intentId.localCommunicationId);

    // ネットワークに接続します。
    networkInfo.networkId.intentId = defaultIntentId;
    NNT_ASSERT_RESULT_SUCCESS(nnt::ldn::TimedConnect(
        networkInfo, security, user, 0, ConnectionTimeout));
    ASSERT_EQ(nn::ldn::State_StationConnected, nn::ldn::GetState());
    ASSERT_EQ(nn::ldn::DisconnectReason_None, nn::ldn::GetDisconnectReason());
    NNT_LDN_SYNC("LocalCommunicationId.Default.Connected");

    // ネットワークから切断されるまで待機します。
    while (nn::ldn::GetState() == nn::ldn::State_StationConnected)
    {
        NNT_LDN_ASSERT_SIGNALED_WEAK(
            stateChangeEvent, ConnectionTimeout + AdvertisingTime);
    }
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::ldn::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::ldn::CommandLineOptionFlag_NodeCount |
        nnt::ldn::CommandLineOptionFlag_NodeIndex |
        nnt::ldn::CommandLineOptionFlag_SceneId |
        nnt::ldn::CommandLineOptionFlag_Channel |
        nnt::ldn::CommandLineOptionFlag_OperationMode |
        nnt::ldn::CommandLineOptionFlag_WirelessControllerRestriction);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax;
    setting.defaultSceneId = 0x1101;

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::ldn::Parse(&g_TestConfig, setting, argc, argv);

    // 無線スイッチをオンにします。
    {
        nnt::ldn::NifmInitializer nifmInitializer;
        nnt::ldn::SetWirelessCommunicationSwitch(true);
    }

    // 他のインスタンスと同期するためにテストサーバに接続します。
    nnt::ldn::HtcsSynchronizationClient client(
        g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
    int result = client.Connect("nn::ldn::IntegrationNetwork");

    // テストを実行します。
    int exitCode;
    if (result == nnt::ldn::SynchronizationResult_Success)
    {
        g_pClient = &client;
        if (g_TestConfig.nodeIndex == 0)
        {
            g_TestConfig.nodeIndex = static_cast<int8_t>(client.GetClientIndex());
        }
        if (g_TestConfig.nodeCount == 0)
        {
            g_TestConfig.nodeCount = static_cast<int8_t>(client.GetClientCount() + 1);
        }
        exitCode = RUN_ALL_TESTS();
        client.Disconnect();
    }
    else
    {
        NNT_LDN_LOG_ERROR("failed to sync: %d\n", result);
        exitCode = result;
    }
    nnt::Exit(exitCode);
}
