﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ldn.h>
#include <nn/ldn/ldn_MonitorApi.h>
#include <nn/ldn/ldn_PrivateApi.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/crypto.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nn/util/util_TinyMt.h>
#include <nnt/nntest.h>
#include <nnt/ldn/testLdn_Config.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>

namespace nnt { namespace ldn { namespace
{
    void InitializeImpl() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Initialize());
    }

    void InitializeSystemImpl() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
    }

    void OpenAccessPointImpl() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
    }

    void OpenStationImpl() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
    }

    void CloseAccessPointImpl() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseAccessPoint());
    }

    void CloseStationImpl() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CloseStation());
    }

}}} // namespace nnt::ldn::<unnamed>

namespace nnt { namespace ldn
{
    Initializer::Initializer() NN_NOEXCEPT
    {
        InitializeImpl();
    }

    Initializer::~Initializer() NN_NOEXCEPT
    {
        nn::ldn::Finalize();
    }

    SystemInitializer::SystemInitializer() NN_NOEXCEPT
    {
        InitializeSystemImpl();
    }

    SystemInitializer::~SystemInitializer() NN_NOEXCEPT
    {
        nn::ldn::FinalizeSystem();
    }

    MonitorInitializer::MonitorInitializer() NN_NOEXCEPT
    {
        nn::ldn::InitializeMonitor();
    }

    MonitorInitializer::~MonitorInitializer() NN_NOEXCEPT
    {
        nn::ldn::FinalizeMonitor();
    }

    AccessPointStarter::AccessPointStarter() NN_NOEXCEPT
    {
        OpenAccessPointImpl();
    }

    AccessPointStarter::~AccessPointStarter() NN_NOEXCEPT
    {
        CloseAccessPointImpl();
    }

    StationStarter::StationStarter() NN_NOEXCEPT
    {
        OpenStationImpl();
    }

    StationStarter::~StationStarter() NN_NOEXCEPT
    {
        CloseStationImpl();
    }

    void CreatePassphrase(nn::ldn::SecurityConfig* pOut, const char* keyword) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);
        NN_SDK_ASSERT_NOT_NULL(keyword);
        nn::crypto::GenerateSha256Hash(
            pOut->passphrase, nn::ldn::PassphraseSizeMax, keyword, std::strlen(keyword));
        pOut->passphraseSize = 32U;
    }

    bool IsValidAccessPointIpv4Address(nn::ldn::Ipv4Address ipv4Address) NN_NOEXCEPT
    {
        // 169.254.x.y s.t. 1 <= x <= 127 && 1 <= y <= 127
        auto address = nn::socket::InetHtonl(ipv4Address.raw);
        auto p = reinterpret_cast<nn::Bit8*>(&address);
        return (p[0] == 169) &&
               (p[1] == 254) &&
               (1 <= p[2] && p[2] <= 255) &&
               (1 <= p[3] && p[3] <= 255);
    }

    bool IsValidStationIpv4Address(
        nn::ldn::Ipv4Address apAddress, nn::ldn::Ipv4Address staAddress) NN_NOEXCEPT
    {
        auto ap = nn::socket::InetHtonl(apAddress.raw);
        auto sta = nn::socket::InetHtonl(staAddress.raw);
        auto a = reinterpret_cast<nn::Bit8*>(&ap);
        auto s = reinterpret_cast<nn::Bit8*>(&sta);
        return IsValidAccessPointIpv4Address(apAddress) &&
               (a[0] == s[0]) &&
               (a[1] == s[1]) &&
               (a[2] == s[2]) &&
               (a[3] != s[3] && 1 <= s[3] && s[3] <= 255);
    }

    nn::Result TimedScan(
        nn::ldn::NetworkInfo* pOutNetwork, int* pOutCount, int bufferCount,
        nn::ldn::ScanFilter filter, int channel, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // スキャン開始時刻を記憶します。
        nn::os::Tick start = nn::os::GetSystemTick();
        nn::os::Tick now = nn::os::GetSystemTick();
        nn::os::Tick diff = now - start;

        // 目的のネットワークを発見するか、タイムアウトするまで検索します。
        *pOutCount = 0;
        while (*pOutCount == 0 && diff.ToTimeSpan() < timeout)
        {
            // ネットワークをスキャンします。
            NN_RESULT_DO(nn::ldn::Scan(pOutNetwork, pOutCount, bufferCount, filter, channel));
            now = nn::os::GetSystemTick();
            diff = now - start;
        }

        // ネットワークが見つからなかった場合にはタイムアウトです。
        if (*pOutCount == 0)
        {
            return nn::ldn::ResultNetworkNotFound();
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result TimedConnect(
        const nn::ldn::NetworkInfo& network,
        const nn::ldn::SecurityConfig& security,
        const nn::ldn::UserConfig& user,
        int version,
        nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // 接続開始時刻を記憶します。
        nn::os::Tick start = nn::os::GetSystemTick();
        nn::os::Tick now = nn::os::GetSystemTick();
        nn::os::Tick diff = now - start;

        // 乱数生成の準備です。
        nn::Bit32 seed;
        nn::os::GenerateRandomBytes(&seed, sizeof(seed));
        nn::util::TinyMt random;
        random.Initialize(seed);

        // 目的のネットワークへの接続に成功するか、タイムアウトするまで接続を試行します。
        nn::Result result;
        do
        {
            // 全員が一斉に接続することを防止するため適当なスリープを入れます。
            const int ms = random.GenerateRandomN(400);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ms));

            // ネットワークへの接続を試行します。
            result = nn::ldn::Connect(
                network, security, user, version, nn::ldn::ConnectOption_None);
            now = nn::os::GetSystemTick();
            diff = now - start;
        } while ((nn::ldn::ResultConnectionTimeout::Includes(result) ||
                  nn::ldn::ResultNetworkNotFound::Includes(result)) &&
                 diff.ToTimeSpan() < timeout);
        return result;
    }


    nn::Result TimedConnectPrivate(
        const nn::ldn::NetworkConfig& network,
        const nn::ldn::SecurityConfig& security,
        const nn::ldn::SecurityParameter& securityParam,
        const nn::ldn::UserConfig& user,
        int version,
        nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // 接続開始時刻を記憶します。
        nn::os::Tick start = nn::os::GetSystemTick();
        nn::os::Tick now = nn::os::GetSystemTick();
        nn::os::Tick diff = now - start;

        // 乱数生成の準備です。
        nn::Bit32 seed;
        nn::os::GenerateRandomBytes(&seed, sizeof(seed));
        nn::util::TinyMt random;
        random.Initialize(seed);

        // 目的のネットワークへの接続に成功するか、タイムアウトするまで接続を試行します。
        nn::Result result = nn::ldn::ResultNetworkNotFound();
        do
        {
            // 全員が一斉に接続することを防止するため適当なスリープを入れます。
            const int ms = random.GenerateRandomN(400);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ms));

            // ネットワークへの接続を試行します。
            result = nn::ldn::ConnectPrivate(
                network, security, securityParam, user, version, nn::ldn::ConnectOption_None);
            now = nn::os::GetSystemTick();
            diff = now - start;
        } while ((nn::ldn::ResultConnectionTimeout::Includes(result) ||
                  nn::ldn::ResultNetworkNotFound::Includes(result)) &&
                 diff.ToTimeSpan() < timeout);
        return result;
    }

}} // namespace nnt::ldn
