﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/lcs/lcs_PrivateDebugApi.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nnt.h>
#include <nnt/lcs/testLcs_Utility.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>
#include "Precondition.h"

namespace
{
    // テストを実行するヘルパです。
    nnt::lcs::TestHelper* g_pHelper;

    // Socket の初期化用バッファです。
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

} // namespace <unnamed>

//
// パッチの Required System Version に基づく LUP の検証です。
// ホストから全クライアントに対する LUP です。
//
TEST(LupPatchRequiredSystemVersion, HostToAllClients)
{
    const auto cond = GetPrecondition(
        "LupPatchRequiredSystemVersion", "HostToAllClients",
        nnt::lcs::Application::Lup01);
    g_pHelper->RunTest(cond, "LupPatchRequiredSystemVersion.HostToAllClients", 2);
}

//
// パッチの Required System Version に基づく LUP の検証です。
// ホストから一部のクライアントに対する LUP です。
//
TEST(LupPatchRequiredSystemVersion, HostToSomeClients)
{
    const auto cond = GetPrecondition(
        "LupPatchRequiredSystemVersion", "HostToSomeClients",
        nnt::lcs::Application::Lup02);
    g_pHelper->RunTest(cond, "LupPatchRequiredSystemVersion.HostToSomeClients", 3);
}

//
// パッチの Required System Version に基づく LUP の検証です。
// クライアントからホストと他の全クライアントに対する LUP です。
//
TEST(LupPatchRequiredSystemVersion, ClientToHostAndAllOtherClients)
{
    const auto cond = GetPrecondition(
        "LupPatchRequiredSystemVersion", "ClientToHostAndAllOtherClients",
        nnt::lcs::Application::Lup01);
    g_pHelper->RunTest(cond, "LupPatchRequiredSystemVersion.ClientToHostAndAllOtherClients", 2);
}

//
// パッチの Required System Version に基づく LUP の検証です。
// クライアントからホストに対する LUP です。
//
TEST(LupPatchRequiredSystemVersion, ClientToHost)
{
    const auto cond = GetPrecondition(
        "LupPatchRequiredSystemVersion", "ClientToHost",
        nnt::lcs::Application::Lup02);
    g_pHelper->RunTest(cond, "LupPatchRequiredSystemVersion.ClientToHost", 2);
}

//
// パッチの Required System Version に基づく LUP の検証です。
// クライアントからホストと 1 台のクライアントに対する LUP です。
//
TEST(LupPatchRequiredSystemVersion, ClientToHostAndOneClient)
{
    const auto cond = GetPrecondition(
        "LupPatchRequiredSystemVersion", "ClientToHostAndOneClient",
        nnt::lcs::Application::Lup01);
    g_pHelper->RunTest(cond, "LupPatchRequiredSystemVersion.ClientToHostAndOneClient", 3);
}

//
// パッチの Required System Version に基づく LUP の検証です。
// クライアントから 1 台のクライアントに対する LUP です。
//
TEST(LupPatchRequiredSystemVersion, ClientToOneClient)
{
    const auto cond = GetPrecondition(
        "LupPatchRequiredSystemVersion", "ClientToOneClient",
        nnt::lcs::Application::Lup02);
    g_pHelper->RunTest(cond, "LupPatchRequiredSystemVersion.ClientToOneClient", 3);
}

//
// パッチの Required System Version に基づく LUP の検証です。
// このテストでは LUP が走りません。
//
TEST(LupPatchRequiredSystemVersion, NotRequired)
{
    const auto cond = GetPrecondition(
        "LupPatchRequiredSystemVersion", "NotRequired",
        nnt::lcs::Application::Lup02);
    g_pHelper->RunTest(cond, "LupPatchRequiredSystemVersion.NotRequired", 2);
}

//
// LCS の Required System Version に基づく LUP の検証です。
// ホストから全クライアントに対する LUP です。
//
TEST(LupLcsRequiredSystemVersion, HostToAllClients)
{
    const auto cond = GetPrecondition(
        "LupLcsRequiredSystemVersion", "HostToAllClients",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupLcsRequiredSystemVersion.HostToAllClients", 2);
}

//
// LCS の Required System Version に基づく LUP の検証です。
// ホストから一部のクライアントに対する LUP です。
//
TEST(LupLcsRequiredSystemVersion, HostToSomeClients)
{
    const auto cond = GetPrecondition(
        "LupLcsRequiredSystemVersion", "HostToSomeClients",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupLcsRequiredSystemVersion.HostToSomeClients", 3);
}

//
// LCS の Required System Version に基づく LUP の検証です。
// クライアントからホストと他の全クライアントに対する LUP です。
//
TEST(LupLcsRequiredSystemVersion, ClientToHostAndAllOtherClients)
{
    const auto cond = GetPrecondition(
        "LupLcsRequiredSystemVersion", "ClientToHostAndAllOtherClients",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupLcsRequiredSystemVersion.ClientToHostAndAllOtherClients", 2);
}

//
// LCS の Required System Version に基づく LUP の検証です。
// クライアントからホストに対する LUP です。
//
TEST(LupLcsRequiredSystemVersion, ClientToHost)
{
    const auto cond = GetPrecondition(
        "LupLcsRequiredSystemVersion", "ClientToHost",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupLcsRequiredSystemVersion.ClientToHost", 2);
}

//
// LCS の Required System Version に基づく LUP の検証です。
// クライアントからホストと 1 台のクライアントに対する LUP です。
//
TEST(LupLcsRequiredSystemVersion, ClientToHostAndOneClient)
{
    const auto cond = GetPrecondition(
        "LupLcsRequiredSystemVersion", "ClientToHostAndOneClient",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupLcsRequiredSystemVersion.ClientToHostAndOneClient", 3);
}

//
// LCS の Required System Version に基づく LUP の検証です。
// クライアントから 1 台のクライアントに対する LUP です。
//
TEST(LupLcsRequiredSystemVersion, ClientToOneClient)
{
    const auto cond = GetPrecondition(
        "LupLcsRequiredSystemVersion", "ClientToOneClient",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupLcsRequiredSystemVersion.ClientToOneClient", 3);
}

//
// LCS の Required System Version に基づく LUP の検証です。
// このテストでは LUP が走りません。
//
TEST(LupLcsRequiredSystemVersion, NotRequired)
{
    const auto cond = GetPrecondition(
        "LupLcsRequiredSystemVersion", "NotRequired",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupLcsRequiredSystemVersion.NotRequired", 3);
}

//
// Application Delivery Protocol に基づく LUP の検証です。
// ホストから全クライアントに対する LUP です。
//
TEST(LupApplicationDeliveryProtocol, HostToAllClients)
{
    const auto cond = GetPrecondition(
        "LupApplicationDeliveryProtocol", "HostToAllClients",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupApplicationDeliveryProtocol.HostToAllClients", 2);
}

//
// Application Delivery Protocol に基づく LUP の検証です。
// ホストから一部のクライアントに対する LUP です。
//
TEST(LupApplicationDeliveryProtocol, HostToSomeClients)
{
    const auto cond = GetPrecondition(
        "LupApplicationDeliveryProtocol", "HostToSomeClients",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupApplicationDeliveryProtocol.HostToSomeClients", 3);
}

//
// Application Delivery Protocol に基づく LUP の検証です。
// クライアントからホストと他の全クライアントに対する LUP です。
//
TEST(LupApplicationDeliveryProtocol, ClientToHostAndAllOtherClients)
{
    const auto cond = GetPrecondition(
        "LupApplicationDeliveryProtocol", "ClientToHostAndAllOtherClients",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupApplicationDeliveryProtocol.ClientToHostAndAllOtherClients", 3);
}

//
// Application Delivery Protocol に基づく LUP の検証です。
// クライアントからホストに対する LUP です。
//
TEST(LupApplicationDeliveryProtocol, ClientToHost)
{
    const auto cond = GetPrecondition(
        "LupApplicationDeliveryProtocol", "ClientToHost",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupApplicationDeliveryProtocol.ClientToHost", 2);
}

//
// Application Delivery Protocol に基づく LUP の検証です。
// クライアントからホストと 1 台のクライアントに対する LUP です。
//
TEST(LupApplicationDeliveryProtocol, ClientToHostAndOneClient)
{
    const auto cond = GetPrecondition(
        "LupApplicationDeliveryProtocol", "ClientToHostAndOneClient",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupApplicationDeliveryProtocol.ClientToHostAndOneClient", 3);
}

//
// Application Delivery Protocol に基づく LUP の検証です。
// クライアントから 1 台のクライアントに対する LUP です。
//
TEST(LupApplicationDeliveryProtocol, ClientToOneClient)
{
    const auto cond = GetPrecondition(
        "LupApplicationDeliveryProtocol", "ClientToOneClient",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupApplicationDeliveryProtocol.ClientToOneClient", 3);
}

//
// Application Delivery Protocol に基づく LUP の検証です。
// このテストでは LUP が走りません。
//
TEST(LupApplicationDeliveryProtocol, NotRequired)
{
    const auto cond = GetPrecondition(
        "LupApplicationDeliveryProtocol", "NotRequired",
        nnt::lcs::Application::Lup03);
    g_pHelper->RunTest(cond, "LupApplicationDeliveryProtocol.NotRequired", 3);
}

//
// LUP の途中でコンテンツ配信を終了する挙動の検証です。
// ホストから全クライアントに LUP が行われ、クライアントはシステム受信後に一時離脱ではなく離脱をします。
//
TEST(LupDiscontinue, LeaveSuspendedWithRebootRequired)
{
    const auto cond = GetPrecondition(
        "LupDiscontinue", "LeaveSuspendedWithRebootRequired",
        nnt::lcs::Application::Lup01);
    g_pHelper->RunTest(cond, "LupDiscontinue.LeaveSuspendedWithRebootRequired", 3);
}

//
// LUP の途中でコンテンツ配信を終了する挙動の検証です。
// ホストから全クライアントに LUP が行われ、クライアントは一時離脱後に再起動せずに再合流します。
//
TEST(LupDiscontinue, ResumeWithoutReboot)
{
    const auto cond = GetPrecondition(
        "LupDiscontinue", "ResumeWithoutReboot",
        nnt::lcs::Application::Lup01);
    g_pHelper->RunTest(cond, "LupDiscontinue.ResumeWithoutReboot", 3);
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::lcs::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::lcs::CommandLineOptionFlag_NodeCount |
        nnt::lcs::CommandLineOptionFlag_NodeIndex |
        nnt::lcs::CommandLineOptionFlag_Resume |
        nnt::lcs::CommandLineOptionFlag_SceneId);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax;

    // コマンドライン引数を解析します。
    nnt::lcs::TestConfig config;
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::lcs::Parse(&config, setting, argc, argv);

    // socket ライブラリと ns ライブラリを初期化しておきます。
    nn::socket::Initialize(g_SocketConfig);
    nnt::lcs::NsInitializer nsInitiailzer;

    // テストの準備です。
    nnt::lcs::Setup(config.sceneId);

    // テストを実行します。
    g_pHelper = new nnt::lcs::TestHelper(config, "nnt::lcs::IntegrationFork");
    int exitCode = RUN_ALL_TESTS();
    delete g_pHelper;

    // 終了処理です。
    nn::socket::Finalize();
    nnt::Exit(exitCode);
}
