﻿# 指定された 1 つのアプリケーションをインストールするタスクです。
Function InstallApplicationTask($Target, $Application, $Patch)
{
    # アプリケーションのインストール状態を確認します。
    $InstalledApplication = $Target.InstalledApplications |
        Where-Object Id -eq $Application.Id |
        Select-Object -First 1

    # 指定されたバージョンがインストールされるように分岐します。
    if ($InstalledApplication -eq $NULL)
    {
        # アプリケーションがインストールされていないため、インストールします。
        InstallApplication $Target.Name (ResolvePath $Application.Path) $TRUE $TRUE
        if ($Patch -ne $NULL)
        {
            ApplyPatch $Target.Name (ResolvePath $Patch.Path) $TRUE $TRUE
        }
    }
    elseif ($Patch -eq $NULL)
    {
        if ($InstalledApplication.Version -ne $Application.Version.DisplayVersion)
        {
            # 不要なパッチが適用されているため削除します。
            UninstallApplication $Target.Name $Application.Id $TRUE
            InstallApplication $Target.Name (ResolvePath $Application.Path) $TRUE $TRUE
        }
    }
    elseif ($InstalledApplication.Version -ne $Patch.Version.DisplayVersion)
    {
        # バージョン違いのパッチが適用されているため、上書きします。
        ApplyPatch $Target.Name (ResolvePath $Patch.Path) $TRUE $TRUE
    }

    # アプリケーションのインストール状態を更新します。
    if ($InstalledApplication -ne $NULL)
    {
        $Target.InstalledApplications.Remove($InstalledApplication) > $null
    }
    $Target.InstalledApplications.Add(@{
        Id      = $Application.Id;
        Version = $Application.Version.DisplayVersion;
        Name    = $Application.Name;
    }) > $null
}

# 指定されたシステムプログラムをインストールするタスクです。
Function InstallSystemProgramTask($Target, $SystemProgram)
{
    # システムプログラムのインストール状態を確認します。
    $InstalledSystemProgram = $Target.InstalledSystemPrograms |
        Where-Object Id -eq $Application.Id |
        Select-Object -First 1

    # 必要に応じてシステムプログラムをインストールします。
    if ($InstalledSystemProgram -eq $NULL)
    {
        InstallSystemProgram $Target.Name (ResolvePath $SystemProgram.Path) $TRUE $TRUE
    }
    elseif ($InstalledSystemProgram.Version -ne $SystemProgram.Version.Value)
    {
        # 元々インストールされていたシステムプログラムをアンインストールしなければ、
        # 再起動後にバージョンが新しい方のシステムプログラムだけが残ってしまいます。
        UninstallSystemProgram $Target.Name $SystemProgram.Id $TRUE
        InstallSystemProgram $Target.Name (ResolvePath $SystemProgram.Path) $TRUE $TRUE
    }

    # システムプログラムのインストール状態を更新します。
    if ($InstalledSystemProgram -ne $NULL)
    {
        $Target.InstalledSystemPrograms.Remove($InstalledSystemProgram) > $null
    }
    $Target.InstalledSystemPrograms.Add(@{
        Id      = $SystemProgram.Id;
        Version = $SystemProgram.Version.DisplayVersion;
        Type    = $NULL;
        Name    = $SystemProgram.Name;
    }) > $null
}

# 指定されたシステムアップデートメタをインストールするタスクです。
Function InstallSystemUpdateMetaTask($Target, $SystemUpdateMeta)
{
    # システムアップデートメタをインストールします。
    InstallSystemProgramTask $Target $SystemUpdateMeta

    # システムアップデートメタに含まれるコンテンツをまとめてインストールします。
    foreach ($SystemData in $SystemUpdateMeta.SystemData)
    {
        InstallSystemProgramTask $Target $SystemData
    }
}

# 指定された 1 つのコンテンツをインストールするタスクです。
Function InstallSubTask($Target, $Value, $Contents)
{
    # コンテンツの種類に応じて処理を分岐します。
    if ($Value.application)
    {
        # アプリケーションを検索します。
        $Application = $Contents.Applications[$Value.application]
        if ($Application -eq $NULL)
        {
            throw "Install Failed: Unknown Application = $($Value.application)"
        }

        # パッチを検索します。
        $Patch = $NULL
        if ($Value.patch)
        {
            $Patch = $Application.Patches[$Value.patch]
            if ($Patch -eq $NULL)
            {
                throw "Install Failed: Unknown Patch = $($Value.application) $($Value.patch)"
            }
        }
        InstallApplicationTask $Target $Application $patch
    }
    elseif ($Value.system_data)
    {
        $SystemData = $Contents.SystemData[$Value.system_data]
        if ($SystemData -eq $NULL)
        {
            throw "Install Failed: Unknown System Data = $($Value.system_data)"
        }
        InstallSystemProgramTask $Target $SystemData
    }
    elseif ($Value.system_update_meta)
    {
        # システムアップデートメタを検索します。
        $SystemUpdateMeta = $Contents.SystemUpdateMeta[$Value.system_update_meta]
        if ($SystemUpdateMeta -eq $NULL)
        {
            throw "Install Failed: Unknown System Update Meta = $($Value.system_update_meta)"
        }

        # バージョンに一致するシステムアップデートメタを検索します。
        $Variation = $SystemUpdateMeta.Variations[(SystemVersion $Value.version).Value]
        if ($Variation -eq $NULL)
        {
            throw "Install Failed: Unknown System Update Meta Version = $($Value.version)"
        }
        InstallSystemUpdateMetaTask $Target $Variation
    }
    else {
        throw "Install Failed: Unspecified Content"
    }
}

# 指定されたコンテンツを全てインストールするタスクです。
Function InstallTask($Target, $Values, $Contents)
{
    foreach ($Value in $Values)
    {
        InstallSubTask $Target $Value $Contents
    }
}
