﻿# 定数を定義します。
Set-Variable TEMP_DIR "$Env:TEMP\Siglo\Lcs" -Option Constant -Scope Script
Set-Variable APPLICATION_DIR "$TEMP_DIR\Applications" -Option Constant -Scope Script
Set-Variable PATCH_DIR "$TEMP_DIR\Patches" -Option Constant -Scope Script
Set-Variable SYSTEM_DATA_DIR "$TEMP_DIR\SystemData" -Option Constant -Scope Script
Set-Variable SYSTEM_UPDATE_META_DIR "$TEMP_DIR\SystemUpdateMeta" -Option Constant -Scope Script

# 一時ファイルを置くディレクトリを生成します。
Function CreateTempDirectory()
{
    if (!(Test-Path $TEMP_DIR))
    {
        New-Item $TEMP_DIR -ItemType Directory > $NULL
    }

    if (!(Test-Path $APPLICATION_DIR))
    {
        New-Item $APPLICATION_DIR -ItemType Directory > $NULL
    }

    if (!(Test-Path $PATCH_DIR))
    {
        New-Item $PATCH_DIR -ItemType Directory > $NULL
    }
    
    if (!(Test-Path $SYSTEM_DATA_DIR))
    {
        New-Item $SYSTEM_DATA_DIR -ItemType Directory > $NULL
    }
    
    if (!(Test-Path $SYSTEM_UPDATE_META_DIR))
    {
        New-Item $SYSTEM_UPDATE_META_DIR -ItemType Directory > $NULL
    }
}

# アプリケーションのバージョンを生成します。
Function PatchVersion($Major, $Minor)
{
    if ($Major -match "[0-9]+\.[0-9]+")
    {
        $Versions = $Major -split "\."
        $Major = [int]$Versions[0]
        $Minor = [int]$Versions[1]
    }
    return @{
        Major = $Major;
        Minor = $Minor;
        DisplayVersion = "$($Major).$($Minor)";
        Value = $Major;
    }
}

# システムバージョンを生成します。
Function SystemVersion($Major, $Minor, $Micro)
{
    if ($Major -match "(?<Major>(?<MajorSign>[+-]?)\d+)\.(?<Minor>(?<MinorSign>[+-]?)\d+)\.(?<Micro>(?<MicroSign>[+-]?)\d+)")
    {
        $FirmwareVersion = GetFirmwareSystemVersion

        $Version = New-Object PSObject -Property $Matches
        $Major = if ($Version.MajorSign -eq "") { [int]$Version.Major } else { [int]$FirmwareVersion.Major + [int]$Version.Major }
        $Minor = if ($Version.MinorSign -eq "") { [int]$Version.Minor } else { [int]$FirmwareVersion.Minor + [int]$Version.Minor }
        $Micro = if ($Version.MicroSign -eq "") { [int]$Version.Micro } else { [int]$FirmwareVersion.Micro + [int]$Version.Micro }
    }
    return @{
        Major = $Major;
        Minor = $Minor;
        Micro = $Micro;
        DisplayVersion = "$($Major).$($Minor).$($Micro)";
        Value = $major * 67108864 + $minor * 1048576 + $micro * 65536;
    }
}

# JSON データからテストアプリケーションに対するパッチ定義を取得します。
Function ReadTestPatchFromJson($Application, $Json)
{
    # 特に名前の指定が無かった場合はベースアプリケーションと同じ名前を使用します。
    $Name = if ($Json.name) { $Json.name } else { $Application.Name }

    # バージョン定義を解析します。
    $PatchVersion = if ($Json.version) { $Json.version } else { "1.0" }
    $ReqSysVersion = SystemVersion $(if ($Json.required) { $Json.required } else { "1.0.0" })

    # データサイズを取得します。デフォルトは 4 Byte です。
    $Size = if ($Json.size) { $Json.size } else { "4B" }

    # NSP のパスを取得します。指定がなければ自動生成します。
    $FileName = "$($Application.Name)_v$($PatchVersion)_for_system_v$($ReqSysVersion.DisplayVersion)"
    $Path = if ($Json.path) { $Json.path } else { "$PATCH_DIR\$FileName.nsp" }

    # パッチを参照する際に使用する名前はデフォルトではバージョン番号です。
    $Alias = if ($Json.alias) { $Json.alias } else { $PatchVersion }
        
    # パッチ定義を生成します。
    $Patch = @{
        Name = $Name;
        Version = (PatchVersion $PatchVersion);
        RequiredSystemVersion = $ReqSysVersion;
        Size = (DataSize $Size)
        Path = $Path;
        Alias = $Alias;
    }
    return $Patch
}

# JSON データからテストアプリケーションに対するアプリケーション定義を取得します。
Function ReadTestApplicationFromJson($Json)
{
    # バージョン定義を解析します。
    $PatchVersion = if ($Json.version) { $Json.version } else { "1.0" }
    $ReqSysVersion = SystemVersion $(if ($Json.required) { $Json.required } else { "1.0.0" })

    # データサイズを取得します。
    $Size = if ($Json.size) { $Json.size } else { "4B" }

    # NSP のパスを取得します。指定がなければ自動生成します。
    $FileName = "$($Json.name)_v$($PatchVersion)_for_system_v$($ReqSysVersion.DisplayVersion)"
    $Path = if ($Json.path) { $Json.path } else { "$APPLICATION_DIR\$FileName.nsp" }
    
    # 鍵世代を取得します。
    $KeyGeneration = if ($Json.key_generation -ne $NULL) { $Json.key_generation }

    # アプリケーションを参照する際に使用する名前はデフォルトでは名前です。
    $Alias = if ($Json.alias) { $Json.alias } else { $Json.name }

    # アプリケーション定義を生成します。
    $Application = @{
        Id = $Json.id;
        Name = $Json.name;
        Version = (PatchVersion $PatchVersion);
        RequiredSystemVersion = $ReqSysVersion;
        Size = (DataSize $Size)
        Path = $Path;
        Patches = @{};
        KeyGeneration = $KeyGeneration;
        Alias = $Alias;
    }

    # パッチを生成します。
    if ($Json.patches)
    {
        $Json.patches | foreach {
            $Patch = (ReadTestPatchFromJson $Application $_)
            $Application.Patches[$Patch.Alias] = $Patch
        }
    }
    return $Application
}

# JSON データからテスト用のシステムデータの定義を取得します。
Function ReadTestSystemDataFromJson($Json)
{
    # システムデータを参照する際に使用する名前はデフォルトでは名前です。
    $Alias = if ($Json.alias) { $Json.alias } else { $Json.name }

    # システムデータの ID と名前を取得します。
    $SystemData = @{
        Id = $Json.id;
        Name = $Json.name;
        Alias = $Alias;
        Variations = @{};
    }

    # 各バージョン毎にシステムデータを生成します。
    foreach ($Def in $Json.variations)
    {
        $Version = SystemVersion $(if ($Def.version) { $Def.version } else { "1.0.0" })
        $Size = if ($Def.size) { $Def.size } else { "256KB" }
        $FileName = "$($SystemData.Name)_v$($Version.DisplayVersion).nsp"
        $Path = if ($Def.path) { $Def.path } else { "$SYSTEM_DATA_DIR\$FileName"}
        $SystemDataItem = @{
            Id = $($SystemData.Id);
            Name = $($SystemData.Name);
            Version = $Version;
            Size = (DataSize $Size);
            Path = $Path;
        }
        $SystemData.Variations[$Version.Value] = $SystemDataItem
    }
    return $SystemData
}

# JSON データからテスト用のシステムアップデートメタの定義を取得します。
Function ReadTestSystemUpdateMetaFromJson($Json, $Contents)
{
    # システムデータを参照する際に使用する名前はデフォルトでは名前です。
    $Alias = if ($Json.alias) { $Json.alias } else { $Json.name }

    # システムアップデートメタの ID と名前を取得します。
    $SystemUpdateMeta = @{
        Id = $Json.id;
        Name = $Json.name;
        Alias = $Alias;
        Variations = @{};
    }

    # 各バージョン毎にシステムアップデートメタを生成します。
    foreach ($Def in $Json.variations)
    {
        $Version = SystemVersion $(if ($Def.version) { $Def.version } else { "1.0.0" })
        $FileName = "$($SystemUpdateMeta.Name)_v$($Version.DisplayVersion).nsp"
        $Path = if ($Def.path) { $Def.path } else { "$SYSTEM_UPDATE_META_DIR\$FileName"}
        $SystemUpdateMetaItem = @{
            Id = $($SystemUpdateMeta.Id);
            Name = $($SystemUpdateMeta.Name);
            Version = $Version;
            Path = $Path;
            SystemData = @();
        }
        $SystemUpdateMeta.Variations[$Version.Value] = $SystemUpdateMetaItem

        foreach ($ContentDef in $Def.contents)
        {
            if ($ContentDef.system_data)
            {
                $SysVer = SystemVersion $(if ($ContentDef.version) { $ContentDef.version } else { "1.0.0" })
                $SystemData = $Contents.SystemData[$ContentDef.system_data].Variations[$SysVer.Value]
                $SystemUpdateMetaItem.SystemData += $SystemData
            }
        }
    }
    return $SystemUpdateMeta
}

# JSON データからシステムデータの定義を取得します。
Function ReadSystemDataFromJson($Json)
{
    # システムデータを参照する際に使用する名前はデフォルトでは名前です。
    $Alias = if ($Json.alias) { $Json.alias } else { $Json.name }

    # システムデータの ID と名前を取得します。
    $SystemData = @{
        Id = $Json.id;
        Name = $Json.name;
        Path = $Json.path;
        Alias = $Alias;
    }

    return $SystemData
}

# コンテンツ定義を JSON ファイルから読み込みます。
Function ReadContentsFromJson($Path)
{
    $Contents = @{
        TestApplications = @{};
        TestSystemData = @{};
        TestSystemUpdateMeta = @{};
        Applications = @{};
        SystemData = @{};
        SystemUpdateMeta = @{};
    }

    (ReadJson "$Path\TestApplications") | foreach {
        $Application = (ReadTestApplicationFromJson $_)
        $Contents.TestApplications[$Application.Alias] = $Application
        $Contents.Applications[$Application.Alias] = $Application
    }

    (ReadJson "$Path\TestSystemData") | foreach {
        $SystemData = (ReadTestSystemDataFromJson $_)
        $Contents.TestSystemData[$SystemData.Alias] = $SystemData
        $Contents.SystemData[$SystemData.Alias] = $SystemData
    }

    (ReadJson "$Path\TestSystemUpdateMeta") | foreach {
        $SystemUpdateMeta = (ReadTestSystemUpdateMetaFromJson $_ $Contents)
        $Contents.TestSystemUpdateMeta[$SystemUpdateMeta.Alias] = $SystemUpdateMeta
        $Contents.SystemUpdateMeta[$SystemUpdateMeta.Alias] = $SystemUpdateMeta
    }

    (ReadJson "$Path\SystemData") | foreach {
        $SystemData = (ReadSystemDataFromJson $_)
        $Contents.SystemData[$SystemData.Alias] = $SystemData
    }

    return $Contents
}

# テストアプリケーションに対するパッチを生成します。
Function CreateTestPatch($Application, $Patch)
{
    if (!(Test-Path $Patch.Path))
    {
        # 出力パスをツールの仕様にあった形式に変換します。
        $Dir = (Split-Path -Parent $Patch.Path)
        $BaseName = [System.IO.Path]::GetFileNameWithoutExtension($Patch.Path)

        # 鍵世代を設定します。
        $ExtraList = @()
        if ($Application.KeyGeneration -ne $NULL)
        {
            $ExtraList += '--keygeneration'
            $ExtraList += $Application.KeyGeneration
        }

        MakeTestApplication `
            --id=$($Application.Id) `
            --ver=$($Patch.Version.Value) `
            --required-version=$($Patch.RequiredSystemVersion.Value) `
            --name=$($Patch.Name) `
            --type Patch `
            --original-application $($Application.Path) `
            -o $($Dir) `
            --output-file-name $($BaseName) `
            --size $($Patch.Size) `
            --small-code `
            --direct=$($ExtraList -join ' ') > $null 
    }
}

# テストアプリケーションを生成します。
Function CreateTestApplication($Application)
{
    if (!(Test-Path $Application.Path))
    {
        # 出力パスをツールの仕様にあった形式に変換します。
        $Dir = (Split-Path -Parent $Application.Path)
        $BaseName = [System.IO.Path]::GetFileNameWithoutExtension($Application.Path)

        # 鍵世代を設定します。
        $ExtraList = @()
        if ($Application.KeyGeneration -ne $NULL)
        {
            $ExtraList += '--keygeneration'
            $ExtraList += $Application.KeyGeneration
        }

        MakeTestApplication `
            --id=$($Application.Id) `
            --ver=$($Application.Version.Value) `
            --required-version=$($Application.RequiredSystemVersion.Value) `
            --name=$($Application.Name) `
            -o $($Dir) `
            --output-file-name $($BaseName) `
            --size $($Application.Size) `
            --small-code `
            --direct=$($ExtraList -join ' ') > $null 
        }
}

# 与えられた定義に従ってシステムデータを生成します。
Function CreateSystemData($SystemData)
{
    foreach ($Key in $SystemData.Variations.Keys)
    {
        $Variation = $SystemData.Variations[$Key]

        # 既にシステムデータが存在する場合はスキップします。
        if (Test-Path $Variation.Path)
        {
            continue
        }

        # システムデータ用のデータディレクトリを生成します。
        $DataDir = "$($SYSTEM_DATA_DIR)\Data"
        if (Test-Path $DataDir)
        {
            Remove-Item -path $DataDir -Recurse -Force
        }
        New-Item $DataDir -ItemType Directory -Force > $NULL

        # 適当なダミーファイルを生成します。
        $File = [System.IO.File]::Create("$DataDir\dummy.txt")
        $File.SetLength($Variation.Size)
        $File.Close()

        # メタファイルを生成します。
        $MetaPath = "$($SYSTEM_DATA_DIR)\$($Variation.Name)_v$($Variation.Version.DisplayVersion).nmeta";
        $Meta  = "<?xml version=`"1.0`"?>`n"
        $Meta += "<NintendoSdkMeta>`n"
        $Meta += "  <SystemData>`n"
        $Meta += "    <Id>$($Variation.Id)</Id>`n"
        $Meta += "    <Version>$($Variation.Version.Value)</Version>`n"
        $Meta += "  </SystemData>`n"
        $Meta += "</NintendoSdkMeta>`n"
        New-Item $MetaPath -ItemType File -Value $Meta -Force > $NULL

        # システムデータを生成します。
        MakeSystemData --meta $($MetaPath) `
            --data $($DataDir) `
            -o $($Variation.Path)

        # 作業ディレクトリとメタファイルを削除します。
        Remove-Item -path $DataDir -Recurse -Force
        Remove-Item -path $MetaPath -Force
    }
}

# 与えられた定義に従ってシステムアップデートメタを生成します。
Function CreateSystemUpdateMeta($SystemUpdateMeta)
{
    # 各バージョン毎にシステムアップデートメタを生成します。
    foreach ($Key in $SystemUpdateMeta.Variations.Keys)
    {
        $Variation = $SystemUpdateMeta.Variations[$Key]

        # 既にシステムアップデートメタが存在する場合はスキップします。
        if (Test-Path $Variation.Path)
        {
            continue
        }
        
        # メタファイルを生成します。
        $MetaFile = "$($Variation.Name)_v$($Variation.Version.DisplayVersion).nmeta"
        $MetaPath = "$($SYSTEM_UPDATE_META_DIR)\$MetaFile"
        $Meta  = "<?xml version=`"1.0`"?>`n"
        $Meta += "<NintendoSdkMeta>`n"
        $Meta += "  <SystemUpdate>`n"
        $Meta += "    <ContentMeta>`n"
        $Meta += "      <Type>SystemUpdate</Type>`n"
        $Meta += "      <Id>$($Variation.Id)</Id>`n"
        $Meta += "      <Version>$($Variation.Version.Value)</Version>`n"
        foreach ($SystemData in $Variation.SystemData)
        {
            $Meta += "      <ContentMeta>`n"
            $Meta += "        <Type>SystemData</Type>`n"
            $Meta += "        <Id>$($SystemData.Id)</Id>`n"
            $Meta += "        <Version>$($SystemData.Version.Value)</Version>`n"
            $Meta += "      </ContentMeta>`n"
        }
        $Meta += "    </ContentMeta>`n"
        $Meta += "  </SystemUpdate>`n"
        $Meta += "</NintendoSdkMeta>`n"
        New-Item $MetaPath -ItemType File -Value $Meta -Force > $NULL

        # システムアップデートを生成します。
        MakeSystemUpdate --meta $($MetaPath) `
            -o $($Variation.Path)

        # メタファイルを削除します。
        Remove-Item -path $MetaPath -Force
    }
}

# テストで使用するコンテンツを生成します。
Function CreateTestContents($Contents)
{
    foreach ($AppKey in $Contents.TestApplications.Keys)
    {
        $Application = $Contents.TestApplications[$AppKey]
        CreateTestApplication $Application
        foreach ($PatchKey in $Application.Patches.Keys)
        {
            $Patch = $Application.Patches[$PatchKey]
            CreateTestPatch $Application $Patch
        }
    }

    foreach ($SystemDataKey in $Contents.TestSystemData.Keys)
    {
        $SystemData = $Contents.SystemData[$SystemDataKey]
        CreateSystemData $SystemData
    }

    foreach ($SystemUpdateMetaKey in $Contents.TestSystemUpdateMeta.Keys)
    {
        $SystemUpdateMeta = $Contents.SystemUpdateMeta[$SystemUpdateMetaKey]
        CreateSystemUpdateMeta $SystemUpdateMeta
    }
}
