﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/lcs.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace lcs { namespace
{
    struct SaveData
    {
        nn::lcs::SessionContext context;
        char _reserved[3 * 1024];
    };

    const char MountName[] = "save";
    const char Path[] = "save:/session_context.bin";

}}} // namespace nnt::lcs::<unnamed>

namespace nnt { namespace lcs
{
    nn::Result SaveSessionContext(const nn::lcs::SessionContext& context) NN_NOEXCEPT
    {
        // セーブデータをマウントします。
        NN_RESULT_DO(nn::fs::MountSaveDataForDebug(MountName));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        {
            // ファイルをオープンします。
            nn::fs::FileHandle handle;
            NN_RESULT_TRY(nn::fs::OpenFile(&handle, Path, nn::fs::OpenMode_Write));
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(nn::fs::CreateFile(Path, sizeof(SaveData)));
                NN_RESULT_DO(nn::fs::OpenFile(&handle, Path, nn::fs::OpenMode_Write));
            }
            NN_RESULT_END_TRY;
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            // ファイルにデータを書き込みます。
            SaveData data = {};
            data.context = context;
            NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &data, sizeof(data), nn::fs::WriteOption()));
            NN_RESULT_DO(nn::fs::FlushFile(handle));
        }
        NN_RESULT_DO(nn::fs::CommitSaveData(MountName));
        NN_RESULT_SUCCESS;
    }

    nn::Result LoadSessionContext(nn::lcs::SessionContext* pOutContext) NN_NOEXCEPT
    {
        // セーブデータをマウントします。
        NN_RESULT_DO(nn::fs::MountSaveDataForDebug(MountName));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        // ファイルをオープンします。
        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, Path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        // ファイルからデータを読み込みます。
        SaveData data;
        size_t readSize;
        NN_RESULT_DO(nn::fs::ReadFile(&readSize, handle, 0, &data, sizeof(data)));
        *pOutContext = data.context;
        NN_RESULT_SUCCESS;
    }

}} // namespace nnt::lcs
