﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <limits>
#include <cmath>
#include "testLbl_AutoControlEmulation.h"

namespace nnt { namespace lbl {

namespace {

const double Epsilon = 0.001;               // 微分値を計算するための刻み幅
const double SensorAdjuster = 0.00122527;   // 蛍光灯下でのセンサー値 816.147339 の逆数
const double SensorMultiplierMin = 0.4;     // 輝度設定値に従って適用する係数の最小値 (New3DS の自動輝度調整機能での輝度1設定と同じ)
const double SensorMultiplierMax = 1.53;    // 輝度設定値に従って適用する係数の最大値 (New3DS の自動輝度調整機能での輝度5設定と同じ)
const double SensorMultiplierRange =
    SensorMultiplierMax - SensorMultiplierMin;  // 係数の最大値と最小値の差

float s_AppliedBrightness;                  // バックライト出力に適用されている輝度設定値
float SensorCoefficients[] = {0.0f, 1.0f, 0.0f};
const float DelayDarkToBright = 1.0f;
const float DelayBrightToDark = 1.0f;

/**
 * @brief   2次関数の微分値を導出する。
 */
float CalculateQuadricDifferential(double x)
{
    // 飽和演算のために double で値を扱う。
    double xDx = (x - Epsilon);
    auto result = (x * x - xDx * xDx) / Epsilon;
    if (std::numeric_limits<float>::lowest() > result)
    {
        return std::numeric_limits<float>::lowest();
    }
    else if (std::numeric_limits<float>::max() < result)
    {
        return std::numeric_limits<float>::max();
    }
    return static_cast<float>(result);
}

/**
 * @brief   照度値を輝度設定値に変換する写像の係数に積算する係数を計算。
 */
double CalculateSensorMultiplier(double initialMultiplier)
{
    // initialMultiplier の最大値は1.0, 最小値は0.0と仮定。
    // ただし、最大値は1.0より大きくても構わない(エラーとして扱わない)。
    double multiplier = SensorMultiplierMin + SensorMultiplierRange * initialMultiplier;
    if (multiplier <= SensorMultiplierMin)
    {
        return SensorMultiplierMin;
    }
    else if (multiplier >= SensorMultiplierMax)
    {
        return SensorMultiplierMax;
    }
    return multiplier;
}

/**
 * @brief   輝度設定値を遷移させる。
 *          2次関数の微分値に基いた速さで遷移。
 */
float TransitBrightness(float postBrightness, float currentBrightness)
{
    auto brightnessDifference = postBrightness - currentBrightness;
    auto delay = (0 < brightnessDifference) ? DelayDarkToBright : DelayBrightToDark;
    NN_ASSERT(0.0f <= delay, "The negative reflection delay:[%f] is invalid.", delay);
    if (0 < delay)
    {
        auto stepLevel = 1.0f / (delay * 60.0f);
        auto transition = CalculateQuadricDifferential(brightnessDifference);
        auto transitedBrightness = currentBrightness + stepLevel * transition;
        if ( 0 < brightnessDifference * (postBrightness - transitedBrightness) )
        {
            // 目標の輝度に達していない場合は、そのときの輝度を返す。
            return transitedBrightness;
        }
        // 目標の輝度に達している場合は目標の輝度をそのまま返す。
    }
    return postBrightness;
}

/**
 * @brief   照度値を輝度設定値に変換する写像。
 *          New3DSの自動輝度調整で使われた式を流用。
 */
float TransformAmbientLightSensorToBrightness(double sensor, double initialBrightness)
{
    // 飽和演算のために double で値を扱う。
    double normalizedSensor = sensor*SensorAdjuster;
    double brightness = CalculateSensorMultiplier(initialBrightness) * (
        SensorCoefficients[0] * std::log10(normalizedSensor + std::numeric_limits<float>::min()) +
        SensorCoefficients[1] * normalizedSensor +
        SensorCoefficients[2]);
    if (0.0 > brightness)
    {
        return 0.0;
    }
    else if (std::numeric_limits<float>::max() < brightness)
    {
        return std::numeric_limits<float>::max();
    }
    return static_cast<float>(brightness);
}

}

float CalculateAutoControlledBrightness(float sensorValue, float brightnessLevel) NN_NOEXCEPT
{
    auto postBrightness = TransformAmbientLightSensorToBrightness(sensorValue, brightnessLevel);
    s_AppliedBrightness = TransitBrightness(postBrightness, s_AppliedBrightness);
    if (s_AppliedBrightness < 0.0f)
    {
        if (-0.0001f < s_AppliedBrightness)
        {
            s_AppliedBrightness = 0.0f;
        }
        else
        {
            NN_LOG("Warning: s_AppliedBrightness is too negative to be used as an argument of SetCurrentBrightnessSetting:[%f]\n", s_AppliedBrightness);
        }
    }
    return s_AppliedBrightness;
}

}}  // namespace nnt::lbl
