﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testLMem_ThreadSafe.h"

namespace {

NN_ALIGNAS(4096) char g_ThreadStack[MaxThreadCount][ThreadStackSize];   // 各スレッドのスタック
nn::os::ThreadType g_Thread[MaxThreadCount];                            // スレッド構造体

}   // unnamed namespace

void LMemThreadSafeTest::InitializeCommmon()
{
    // グローバルなヒープからのメモリ領域確保
    m_Heap = std::malloc(m_HeapSize);

    // 利用できるコア番号とコア数の取得
    // 最大 MaxThreadCount 個取得する
    nn::Bit64 coreMask = nn::os::GetThreadAvailableCoreMask();
    m_MaxCoreCount = 0;
    for(int i = 0; i < 64; ++i)
    {
        if(coreMask & (1ULL << i))
        {
            m_CoreNumber[m_MaxCoreCount] = static_cast<int>(i);
            m_MaxCoreCount++;
            if(m_MaxCoreCount >= MaxThreadCount)
            {
                break;
            }
        }
    }
}

void LMemThreadSafeTest::StartThread(const int FunctionCount)
{
    for(int i = 0; i < FunctionCount && i < m_MaxCoreCount; ++i)
    {
        nn::os::StartThread(&g_Thread[i]);
    }

    for(int i = 0; i < FunctionCount && i < m_MaxCoreCount; ++i)
    {
        nn::os::WaitThread(&g_Thread[i]);
    }

    for(int i = 0; i < FunctionCount && i < m_MaxCoreCount; ++i)
    {
        nn::os::DestroyThread(&g_Thread[i]);
    }
}

void LMemThreadSafeTest::DoMultiThreadTest(nn::os::ThreadFunction* functions, const int FunctionCount)
{
    ASSERT_TRUE(FunctionCount <= MaxThreadCount);

    nn::Result result;

    // functions の関数を動作させる
    for(int i = 0; i < FunctionCount && i < m_MaxCoreCount; ++i)
    {
        result = nn::os::CreateThread(&g_Thread[i], functions[i], m_HeapHandle, &g_ThreadStack[i], ThreadStackSize, nn::os::DefaultThreadPriority, m_CoreNumber[i]);
        ASSERT_TRUE(result.IsSuccess());
    }

    StartThread(FunctionCount);
}

void LMemThreadSafeTest::DoMultiThreadTestWithCheckStruct(nn::os::ThreadFunction* functions, const int FunctionCount)
{
    ASSERT_TRUE(FunctionCount <= MaxThreadCount);

    nn::Result result;

    for(int i = 0; i < FunctionCount && i < m_MaxCoreCount; ++i)
    {
        CheckStruct check;
        check.handle = m_HeapHandle;
        check.fill = static_cast<nn::Bit64>(i);
        result = nn::os::CreateThread(&g_Thread[i], functions[i], &check, &g_ThreadStack[i], ThreadStackSize, nn::os::DefaultThreadPriority, m_CoreNumber[i]);
        ASSERT_TRUE(result.IsSuccess());
    }

    StartThread(FunctionCount);
}
