﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lmem/lmem_ExpHeap.h>

#include "testLMem_ThreadSafe.h"

namespace {

/**
 * @brief   拡張ヒープのテストフィクスチャです。
 */
class ExpHeapThreadSafeTest : public LMemThreadSafeTest
{
protected:
    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        InitializeCommmon();

        // デバッグフィルすることで、スレッドセーフでない場合ヘッダ部分がフィルされてしまい
        // ヒープチェックで引っかかるようになる
        m_HeapHandle = nn::lmem::CreateExpHeap(m_Heap, m_HeapSize, nn::lmem::CreationOption_ThreadSafe | nn::lmem::CreationOption_DebugFill);
        ASSERT_NOT_NULL(m_HeapHandle);
    }

    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        nn::lmem::DestroyExpHeap(m_HeapHandle);
        std::free(m_Heap);
    }
};

// =============
// スレッド関数
// =============

/**
 * @brief   メモリブロックの確保を行います。
 * @details また、既に確保済みのメモリブロックが破壊されていないか確認します。
 */
void AllocateExpHeap(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);

    const int BlockCount = 512;         // 確保するメモリブロックの数
    const size_t AllocateSize = 16;     // 確保するメモリブロックのサイズ
    void* memoryBlocks[BlockCount];     // 確保したメモリブロックのアドレス

    for(int i = 0; i < BlockCount; ++i)
    {
        memoryBlocks[i] = nn::lmem::AllocateFromExpHeap(handle, AllocateSize);
        ASSERT_NOT_NULL(memoryBlocks[i]);
        for(int j = 0; j <= i; ++j)
        {
            ASSERT_TRUE(nn::lmem::CheckExpHeapBlock(handle, memoryBlocks[j], nn::lmem::ErrorOption_NoOption));
        }
    }
}

/**
 * @brief   メモリブロックの確保と解放を行います。
 * @details また、確保済みのメモリブロックが破壊されていないか確認します。
 */
void AllocateFreeExpHeap(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);

    const int BlockCount = 512;         // 確保するメモリブロックの数
    const size_t AllocateSize = 16;     // 確保するメモリブロックのサイズ
    void* memoryBlocks[BlockCount];     // 確保したメモリブロックのアドレス

    for(int i = 0; i < BlockCount; ++i)
    {
        memoryBlocks[i] = nn::lmem::AllocateFromExpHeap(handle, AllocateSize);
        ASSERT_NOT_NULL(memoryBlocks[i]);
        for(int j = 0; j <= i; ++j)
        {
            ASSERT_TRUE(nn::lmem::CheckExpHeapBlock(handle, memoryBlocks[j], nn::lmem::ErrorOption_NoOption));
        }
    }

    for(int i = BlockCount - 1; i >= 0; --i)
    {
        nn::lmem::FreeToExpHeap(handle, memoryBlocks[i]);
        for(int j = 0; j < i; ++j)
        {
            ASSERT_TRUE(nn::lmem::CheckExpHeapBlock(handle, memoryBlocks[j], nn::lmem::ErrorOption_NoOption));
        }
    }
}

/**
 * @brief   メモリブロックの確保・解放とリサイズを行います。
 * @details また、確保済みのメモリブロックが破壊されていないか確認します。
 */
void ResizeExpHeap(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);

    const int BlockCount = 512;         // 確保するメモリブロックの数
    const size_t AllocateSize = 16;     // 確保するメモリブロックのサイズ
    void* memoryBlocks[BlockCount];     // 確保したメモリブロックのアドレス

    for(int i = 0; i < BlockCount; ++i)
    {
        memoryBlocks[i] = nn::lmem::AllocateFromExpHeap(handle, AllocateSize);
        ASSERT_NOT_NULL(memoryBlocks[i]);
        for(int j = 0; j <= i; ++j)
        {
            ASSERT_TRUE(nn::lmem::CheckExpHeapBlock(handle, memoryBlocks[j], nn::lmem::ErrorOption_NoOption));
        }
        if(i % 2)
        {
            nn::lmem::ResizeExpHeapBlock(handle, memoryBlocks[i], AllocateSize * 2);
        }
        else
        {
            nn::lmem::ResizeExpHeapBlock(handle, memoryBlocks[i], AllocateSize / 2);
        }
    }
}

/**
 * @brief   後方からのメモリ確保と解放を繰り返しメモリアクセスする領域を重複させます。
 */
void AllocateExpHeapFromTail(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);

    const int RepeatCount = 1024;       // 確保と解放を繰り返す回数
    const size_t AllocateSize = 8;      // 確保するメモリブロックのサイズ
    void* pMemoryBlock;                 // 確保したメモリブロックのアドレス

    for(int i = 0; i < RepeatCount; ++i)
    {
        pMemoryBlock = nn::lmem::AllocateFromExpHeap(handle, AllocateSize, -nn::lmem::DefaultAlignment);
        if(pMemoryBlock != nullptr)
        {
            ASSERT_TRUE(nn::lmem::CheckExpHeapBlock(handle, pMemoryBlock, nn::lmem::ErrorOption_NoOption));
            nn::lmem::FreeToExpHeap(handle, pMemoryBlock);
        }
    }
}

/**
 * @brief   ヒープ領域を少しずつ縮小させていきます。
 */
void AdjustExpHeap(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);

    const int RepeatCount = 512;                // Adjust を繰り返す回数
    const int FirstAllocateSize = 1024 * 1024;  // 最初に確保するサイズ
    const int DecreaseSize = 128;               // 1 回の試行で減らしていくヒープ領域のサイズ
    size_t allocateSize;                        // 確保するメモリブロックのサイズ
    void* pMemoryBlock;                         // 確保したメモリブロックのアドレス
    nn::lmem::MemoryRange range;

    allocateSize = FirstAllocateSize - DecreaseSize;
    for(int i = 0; i < RepeatCount; ++i)
    {
        // まず Adjust でヒープ領域が全て消えてしまわないように前方からメモリを確保する
        pMemoryBlock = nn::lmem::AllocateFromExpHeap(handle, allocateSize);
        ASSERT_NOT_NULL(pMemoryBlock);
        ASSERT_TRUE(nn::lmem::CheckExpHeapBlock(handle, pMemoryBlock, nn::lmem::ErrorOption_NoOption));

        // Adjust してヒープ領域を少し縮める
        range = nn::lmem::AdjustExpHeap(handle);

        nn::lmem::FreeToExpHeap(handle, pMemoryBlock);
        if(range.size != 0)
        {
            allocateSize -= DecreaseSize;
        }
    }
}

/**
 * @brief   空きメモリサイズの取得を繰り返します。
 */
void GetTotalFreeSizeExpHeap(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);
    const int RepeatCount = 1024;        // 試行を繰り返す回数
    for(int i = 0; i < RepeatCount; ++i)
    {
        nn::lmem::GetExpHeapAllocatableSize(handle, nn::lmem::DefaultAlignment);
    }
}

/**
 * @brief   確保可能サイズの取得を繰り返します。
 */
void GetAllocatableSizeExpHeap(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);
    const int RepeatCount = 1024;        // 試行を繰り返す回数
    for(int i = 0; i < RepeatCount; ++i)
    {
        nn::lmem::GetExpHeapAllocatableSize(handle, nn::lmem::DefaultAlignment);
    }
}

/**
 * @brief   全てのメモリブロックで実行するコールバック関数です。
 * @details 今回のテスト内容ではこの関数の中身は関係しないため、処理は適当です。
 */
void WalkFunction(void* pBlock, nn::lmem::HeapHandle heapHandle, uintptr_t userParam)
{
    NN_UNUSED(heapHandle);
    NN_UNUSED(userParam);
    char* str = reinterpret_cast<char*>(pBlock);
    str[0] = 1;
}

/**
 * @brief   Visit 関数を繰り返し呼びます。
 */
void VisitExpHeap(void* arg)
{
    nn::lmem::HeapHandle handle = reinterpret_cast<nn::lmem::HeapHandle>(arg);
    const int RepeatCount = 1024;        // 試行を繰り返す回数
    for(int i = 0; i < RepeatCount; ++i)
    {
        nn::lmem::VisitExpHeapAllBlocks(handle, WalkFunction, 0);
    }
}

}   // unnamed namespace

// =======
// テスト
// =======

/**
 * @brief   複数スレッドを作成し、確保を繰り返します。
 * @details スレッドセーフなら、複数スレッドで同じ領域を確保してしまうことはありません。@n
 *          アンスレッドセーフなら、同じ領域を確保してしまう可能性があります。
 */
TEST_F(ExpHeapThreadSafeTest, AllocateExpHeap)
{
    const int FunctionCount = 8;    // 並列実行する関数の数 ( < MaxThreadCount)

    // 並列実行する関数
    // ただし、環境によっていくつのコアがあるかはわからないため、どの関数までが実行されるかは環境依存
    // 例えば、コア数が 4 なら、 functions[0] ～ functions[3] が並列実行される
    nn::os::ThreadFunction functions[FunctionCount] = { AllocateExpHeap,    // コア0で実行したい関数
                                                        AllocateExpHeap,    // コア1で実行したい関数
                                                        AllocateExpHeap,    // コア2で実行したい関数
                                                        AllocateExpHeap,    // コア3で実行したい関数
                                                        AllocateExpHeap,    // コア4で実行したい関数
                                                        AllocateExpHeap,    // コア5で実行したい関数
                                                        AllocateExpHeap,    // コア6で実行したい関数
                                                        AllocateExpHeap };  // コア7で実行したい関数
    DoMultiThreadTest(functions, FunctionCount);
}

/**
 * @brief   Allocate テストを解放も含めて行ったものです。
 */
TEST_F(ExpHeapThreadSafeTest, AllocateFreeExpHeap)
{
    const int FunctionCount = 8;
    nn::os::ThreadFunction functions[FunctionCount] = { AllocateFreeExpHeap,
                                                        AllocateFreeExpHeap,
                                                        AllocateFreeExpHeap,
                                                        AllocateFreeExpHeap,
                                                        AllocateFreeExpHeap,
                                                        AllocateFreeExpHeap,
                                                        AllocateFreeExpHeap,
                                                        AllocateFreeExpHeap };
    DoMultiThreadTest(functions, FunctionCount);
}

/**
 * @brief   Allocate テストをリサイズも含めて行ったものです。
 */
TEST_F(ExpHeapThreadSafeTest, ResizeExpHeap)
{
     const int FunctionCount = 8;
     nn::os::ThreadFunction functions[FunctionCount] = { ResizeExpHeap,
                                                         ResizeExpHeap,
                                                         ResizeExpHeap,
                                                         ResizeExpHeap,
                                                         ResizeExpHeap,
                                                         ResizeExpHeap,
                                                         ResizeExpHeap,
                                                         ResizeExpHeap };
     DoMultiThreadTest(functions, FunctionCount);
}

/**
 * @brief   複数スレッドで後方からの確保とヒープの縮小を繰り返します。
 * @details スレッドセーフでない場合、以下のタイミングでヒープが縮小された場合、ヒープがメモリブロックをたどれない状況になってしまいます。
 *              スレッドA: 後方から確保されていないかチェック
 *              スレッドB: 後方からのメモリブロック確保
 *              スレッドA: ヒープ領域の縮小
 */
TEST_F(ExpHeapThreadSafeTest, AdjustExpHeap)
{
     const int FunctionCount = 2;
     nn::os::ThreadFunction functions[FunctionCount] = { AllocateExpHeapFromTail,
                                                         AdjustExpHeap };
     DoMultiThreadTest(functions, FunctionCount);
}

/**
 * @brief   複数スレッドでサイズ取得関数を利用します。
 * @details スレッドセーフでない場合、ヒープの空き容量計算のためメモリブロックをたどっている途中に別スレッドがメモリブロックを解放してしまうと、メモリブロックをたどれなくなります。
 *
 */
TEST_F(ExpHeapThreadSafeTest, GetSizeExpHeap)
{
     const int FunctionCount = 8;
     nn::os::ThreadFunction functions[FunctionCount] = { GetAllocatableSizeExpHeap,
                                                         GetTotalFreeSizeExpHeap,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail };
     DoMultiThreadTest(functions, FunctionCount);
}

/**
 * @brief   複数スレッドで確保済みのメモリブロックに特定の処理を行います。
 * @details スレッドセーフでない場合、すべてのメモリブロックをたどっている途中にに別スレッドがメモリブロックを解放してしまうと、メモリブロックをたどれなくなります。
 */
TEST_F(ExpHeapThreadSafeTest, VisitExpHeap)
{
     const int FunctionCount = 8;
     nn::os::ThreadFunction functions[FunctionCount] = { VisitExpHeap,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail,
                                                         AllocateExpHeapFromTail };
     DoMultiThreadTest(functions, FunctionCount);
}
