﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TestOnOtherProcessJitRunner.h"

#include "../Common/IProtoJitLoader.sfdl.h"
#include "../Common/TestVmJitAarch64.h"
#include <utility>
#include <memory>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ProxyObjectAllocator.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/init/init_Malloc.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/svc/svc_Base.h>

namespace nn { namespace jit { namespace testvm {

sf::ProxyObjectAllocator<10> g_ProxyObjectAllocator = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;

Result CreateJitEnvironment(sf::SharedPointer<IJitEnvironment>* pOut, uintptr_t bufferAddress, size_t bufferSize) NN_NOEXCEPT
{
    auto jitLoader2 = sf::CreateHipcProxyByName<IProtoJitLoader2>(g_ProxyObjectAllocator.GetMemoryResource(), "jit:p2");
    NN_RESULT_DO(jitLoader2);
    sf::SharedPointer<IJitEnvironment> pJitEnvironment;
    sf::NativeHandle processHandle(svc::PSEUDO_HANDLE_CURRENT_PROCESS.value, false);
    nn::Bit64 processId;
    NN_RESULT_DO(svc::GetProcessId(&processId, svc::PSEUDO_HANDLE_CURRENT_PROCESS));
    NN_RESULT_DO(jitLoader2.Ref()->CreateJitEnvironment(&pJitEnvironment, std::move(processHandle), processId, bufferAddress, bufferSize));
    *pOut = std::move(pJitEnvironment);
    NN_RESULT_SUCCESS;
}

class OtherProcessJitRunner
    : public JitRunner
{
private:

    sf::SharedPointer<ITestVmJitCompileEnvironment> m_pCompileEnvironment;

public:

    OtherProcessJitRunner(sf::SharedPointer<ITestVmJitCompileEnvironment> pCompileEnvironment, MachineContext* pContext) NN_NOEXCEPT
        : JitRunner(pContext)
        , m_pCompileEnvironment(std::move(pCompileEnvironment))
    {
    }

    virtual void* GenerateCode(size_t* pByteSize, uint32_t* pCodeCount, int32_t pc) NN_NOEXCEPT NN_OVERRIDE
    {
        uint64_t entry;
        uint64_t byteSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pCompileEnvironment->GenerateCode(&entry, &byteSize, pCodeCount, pc));
        *pByteSize = static_cast<size_t>(byteSize);
        return reinterpret_cast<void*>(entry);
    }

};

Result RunOnOtherProcessJitImpl(nn::jit::testvm::MachineContext* pContext) NN_NOEXCEPT
{
    const auto CodeBufferSize = 1024 * 1024;
    auto codeBuffer = nn::init::GetAllocator()->Allocate(CodeBufferSize, 4096);
    NN_UTIL_SCOPE_EXIT
    {
        nn::init::GetAllocator()->Free(codeBuffer);
    };

    sf::SharedPointer<IJitEnvironment> pJitEnvironment;
    NN_RESULT_DO(CreateJitEnvironment(&pJitEnvironment, reinterpret_cast<uintptr_t>(codeBuffer), CodeBufferSize));
    sf::SharedPointer<ITestVmJitCompileEnvironment> pCompileEnvironment;
    NN_RESULT_DO(pJitEnvironment->CreateTestVmJitCompileEnvironment(&pCompileEnvironment));
    pJitEnvironment = nullptr;

    auto programBufferSize = util::align_up(pContext->program.instructionCount * 4, 4096);
    auto programBuffer = nn::init::GetAllocator()->Allocate(programBufferSize, 4096);
    std::memcpy(programBuffer, pContext->program.instructions, pContext->program.instructionCount * 4);
    NN_UTIL_SCOPE_EXIT
    {
        nn::init::GetAllocator()->Free(programBuffer);
    };

    {
        os::TransferMemory rom(programBuffer, programBufferSize, os::MemoryPermission_ReadOnly);
        NN_RESULT_DO(pCompileEnvironment->ShareRom(0, sf::NativeHandle(rom.Detach(), true), programBufferSize));
    }

    OtherProcessJitRunner runner(pCompileEnvironment, pContext);
    runner.Run();

    NN_RESULT_SUCCESS;
}

void RunOnOtherProcessJit(nn::jit::testvm::MachineContext* pContext) NN_NOEXCEPT
{
    g_ProxyObjectAllocator.Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(RunOnOtherProcessJitImpl(pContext));
    g_ProxyObjectAllocator.Finalize();
}

}}}
