﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#if defined( NN_BUILD_CONFIG_OS_WIN )
//#include <GL/glew.h>
#endif
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nnt/graphics/testGraphics_Png.h>
#include <nnt/graphics/testGraphics_InitializeAllocatorFunctionForStandardAllocator.h>

#include "FrameCapture.h"


namespace nnt{ namespace eft{

    FrameCapture::FrameCapture()
        : m_IsInitialized(false),
          m_ScreenWidth(0),
          m_ScreenHeight(0),
          m_pDevice(NULL)
    {
    }

    FrameCapture::~FrameCapture()
    {
        this->Finalize();
    }

    //-------------------------------------------

    void FrameCapture::Initialize(
        nn::gfx::Device* pDevice,
        nn::gfx::MemoryPool* pMemoryPool,
        ptrdiff_t* pMemoryPoolOffset,
        nn::gfx::MemoryPool* pMemoryPoolInvisible,
        ptrdiff_t* pMemoryPoolInvisibleOffset,
        int screenWidth,
        int screenHeight)
    {
        static const size_t HeapSize = 32 * 1024 * 1024;
        m_TestGraphicsStandardAllocator.Initialize( malloc(HeapSize), HeapSize );
        nnt::graphics::InitializeAllocatorFunctionForStandardAllocator::Initialize(
            &m_TestGraphicsAllocatorFunction,
            &m_TestGraphicsStandardAllocator);

        m_ScreenWidth  = screenWidth;
        m_ScreenHeight = screenHeight;
        m_pDevice      = pDevice;

        // バッファを作成
        size_t imageBufferSize = screenWidth * screenHeight * 16;
        {
            nn::gfx::Buffer::InfoType info;
            info.SetDefault();
            info.SetSize(imageBufferSize);
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);

            if (NN_STATIC_CONDITION(nn::gfx::Texture::IsMemoryPoolRequired))
            {
                *pMemoryPoolOffset = nn::util::align_up(*pMemoryPoolOffset, nn::gfx::Buffer::GetBufferAlignment(m_pDevice, info));
                m_CaptureBuffer.Initialize(m_pDevice, info, pMemoryPool, *pMemoryPoolOffset, info.GetSize());
                *pMemoryPoolOffset += info.GetSize();
            }
            else
            {
                m_CaptureBuffer.Initialize(pDevice, info, NULL, 0, 0);
            }
        }
        m_CapturedImageBuffer = static_cast<uint8_t*>(m_TestGraphicsStandardAllocator.Allocate(imageBufferSize, NN_ALIGNOF(std::max_align_t)));
        m_CapturedImageBufferSize = imageBufferSize;

        // テクスチャを作成
        {
            nn::gfx::Texture::InfoType info;
            info.SetDefault();
            info.SetWidth(screenWidth);
            info.SetHeight(screenHeight);
            info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
            info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
            info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
            info.SetMipCount(1);
            info.SetDepth(1);

            *pMemoryPoolInvisibleOffset = nn::util::align_up(*pMemoryPoolInvisibleOffset, nn::gfx::Texture::CalculateMipDataAlignment(m_pDevice, info));
            size_t size = nn::gfx::Texture::CalculateMipDataSize(m_pDevice, info);
            m_CaptureTexture.Initialize(m_pDevice, info, pMemoryPoolInvisible, *pMemoryPoolInvisibleOffset, size);
            *pMemoryPoolInvisibleOffset += size;
        }

        m_IsInitialized = true;
    }

    void FrameCapture::Finalize()
    {
        if(!m_IsInitialized)
        {
            return;
        }
        m_CaptureTexture.Finalize(m_pDevice);
        m_CaptureBuffer.Finalize(m_pDevice);

        m_TestGraphicsStandardAllocator.Free(m_CapturedImageBuffer);
        m_CapturedImageBuffer = NULL;
        m_CapturedImageBufferSize = 0;

        m_IsInitialized = false;
    }

    //-------------------------------------------
    void FrameCapture::PushCaptureCommand(nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::ColorTargetView* pSrcView, nn::gfx::ImageFormat srcTextureFormat)
    {
        pCommandBuffer->Resolve(&m_CaptureTexture, 0, 0, pSrcView, nullptr);
        PushCaptureCommand(pCommandBuffer, &m_CaptureTexture, srcTextureFormat);
    }

    //-------------------------------------------

    void FrameCapture::PushCaptureCommand(nn::gfx::CommandBuffer* pCommandBuffer, nn::gfx::Texture* pSrcTexture, nn::gfx::ImageFormat srcTextureFormat)
    {
        NN_ASSERT(
            srcTextureFormat == nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm ||
            srcTextureFormat == nn::gfx::ImageFormat_R32_G32_B32_A32_Float);

        nn::gfx::TextureCopyRegion region;
        region.SetDefault();
        region.SetWidth(m_ScreenWidth);
        region.SetHeight(m_ScreenHeight);
        m_CaptureFormat = srcTextureFormat;
        pCommandBuffer->CopyImageToBuffer(&m_CaptureBuffer, 0, pSrcTexture, region);
    }

    //-------------------------------------------

    void FrameCapture::FetchCaptureResult()
    {
        if(m_CaptureFormat == nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm)
        {
            size_t stride = 4 * m_ScreenWidth;
            const uint8_t* pBuffer = static_cast<uint8_t*>(m_CaptureBuffer.Map());
            m_CaptureBuffer.InvalidateMappedRange(0, 4 * m_ScreenWidth * m_ScreenHeight);
            int ySrc = 0;
            int yDst = m_ScreenHeight - 1;
            for(; ySrc < m_ScreenHeight; ySrc++, yDst--)
            {
                const uint8_t* pSrc = pBuffer + stride * ySrc;
                uint8_t* pDst = m_CapturedImageBuffer + stride * yDst;
                for(int x = 0; x < m_ScreenWidth; x++)
                {
                    *(pDst + x * 4 + 0) = *(pSrc + x * 4 + 0);
                    *(pDst + x * 4 + 1) = *(pSrc + x * 4 + 1);
                    *(pDst + x * 4 + 2) = *(pSrc + x * 4 + 2);
                    *(pDst + x * 4 + 3) = 255;                  // α
                }
            }
        }
        else if(m_CaptureFormat == nn::gfx::ImageFormat_R32_G32_B32_A32_Float)
        {
            size_t srcStride = 16 * m_ScreenWidth;
            size_t dstStride = 4 * m_ScreenWidth;
            const uint8_t* pBuffer = static_cast<uint8_t*>(m_CaptureBuffer.Map());
            m_CaptureBuffer.InvalidateMappedRange(0, 16 * m_ScreenWidth * m_ScreenHeight);
            int ySrc = 0;
            int yDst = m_ScreenHeight - 1;
            for(; ySrc < m_ScreenHeight; ySrc++, yDst--)
            {
                const float* pSrc = reinterpret_cast<const float*>(pBuffer + srcStride * ySrc);
                uint8_t* pDst = m_CapturedImageBuffer + dstStride * yDst;
                for(int x = 0; x < m_ScreenWidth * 4; x++)
                {
                    if ((x % 4) == 3)
                    {
                        pDst[x] = 255;      // α
                    }
                    else
                    {
                        if(pSrc[x] < 0)
                        {
                            pDst[x] = 0;
                        }
                        else if(pSrc[x] > 1)
                        {
                            pDst[x] = 255;
                        }
                        else
                        {
                            pDst[x] = static_cast<uint8_t>(pSrc[x] * 255);
                        }
                    }
                }
            }
        }
        else
        {
            NN_ABORT("Unknown capture format\n");
        }

        m_CaptureBuffer.Unmap();
    }

    //-------------------------------------------

    void FrameCapture::SaveToPng(const char* destPath)
    {
        nnt::graphics::PngIhdr idr;
        idr.width = m_ScreenWidth;
        idr.height = m_ScreenHeight;
        idr.bitDepth = 8;
        idr.colorType = nnt::graphics::PngColorType_RgbAlpha;
        idr.compressionType = 0;
        idr.filterType = 0;
        idr.interlaceType = 0;
        idr.channels = 4;

        size_t size = static_cast<size_t>(idr.width * idr.height * idr.channels);

        nnt::graphics::PngIO::Write
            ( destPath
            , m_CapturedImageBuffer
            , size
            , &idr
            , &m_TestGraphicsAllocatorFunction);
    }
}}

