﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/irsensor.h>
#include <nnt.h>
#include "../Common/testIrsensor_Util.h"

using namespace ::nn::irsensor;

namespace nnt { namespace irsensor {
    IrCameraHandle IrSensorTest::s_Handles[NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}}

namespace
{
const int NotReadyWaitLoopCountMax = 100;
const nn::TimeSpan PollingInterval = nn::TimeSpan::FromMilliSeconds(15);

class ClusteringProcessorTest : public ::nnt::irsensor::IrSensorTest
{
protected:
    void DoTestWoiBoundary(int x, int y, int width, int height) NN_NOEXCEPT;

    void DoTestObjectIntensityMinBoundary(int threshold) NN_NOEXCEPT;

    void DoTestObjectPixelCountBoundary(int min, int max) NN_NOEXCEPT;

    void DoTestClusteringData(const ClusteringData& object, const ClusteringProcessorConfig& config) NN_NOEXCEPT;
};

TEST_F(ClusteringProcessorTest, DefaultClusteringProcessorConfig)
{
    ClusteringProcessorConfig config;
    GetClusteringProcessorDefaultConfig(&config);

    EXPECT_LE(ClusteringProcessorExposureTimeMin, config.irCameraConfig.exposureTime);
    EXPECT_LE(config.irCameraConfig.exposureTime, ClusteringProcessorExposureTimeMax);
    EXPECT_LE(IrCameraGainMin, config.irCameraConfig.gain);
    EXPECT_LE(config.irCameraConfig.gain, IrCameraGainMax);
    EXPECT_EQ(false, config.irCameraConfig.isNegativeImageUsed);
    EXPECT_EQ(IrCameraLightTarget_AllObjects, config.irCameraConfig.lightTarget);

    EXPECT_EQ(0, config.windowOfInterest.x);
    EXPECT_EQ(0, config.windowOfInterest.y);
    EXPECT_EQ(IrCameraImageWidth, config.windowOfInterest.width);
    EXPECT_EQ(IrCameraImageHeight, config.windowOfInterest.height);
    EXPECT_LE(0, config.objectIntensityMin);
    // objectPixelCountMin と objectPixelCountMax の間に順序関係の制約はない
    EXPECT_LE(0, config.objectPixelCountMin);
    EXPECT_LE(config.objectPixelCountMin, ClusteringProcessorObjectPixelCountMax);
    EXPECT_LE(0, config.objectPixelCountMax);
    EXPECT_LE(config.objectPixelCountMax, ClusteringProcessorObjectPixelCountMax);
}

void ClusteringProcessorTest::DoTestWoiBoundary(int x, int y, int width, int height) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        ClusteringProcessorConfig config;
        GetClusteringProcessorDefaultConfig(&config);

        config.windowOfInterest = MakeRect(x, y, width, height);
        RunClusteringProcessor(handle, config);
        StopImageProcessor(handle);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TEST_F(ClusteringProcessorTest, WoiXBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, 0, IrCameraImageWidth, IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(1, 0, IrCameraImageWidth - 1, IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(IrCameraImageWidth - 1, 0, 1, IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, IrCameraImageWidth, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(-1, 0, IrCameraImageWidth, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(IrCameraImageWidth, 0, 0, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(IrCameraImageWidth, 0, 1, IrCameraImageHeight), "");
}

TEST_F(ClusteringProcessorTest, WoiYBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, 1, IrCameraImageWidth, IrCameraImageHeight - 1), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, IrCameraImageHeight - 1, IrCameraImageWidth, 1), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 1, IrCameraImageWidth, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, -1, IrCameraImageWidth, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, IrCameraImageHeight, IrCameraImageWidth, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, IrCameraImageHeight, IrCameraImageWidth, 1), "");
}

TEST_F(ClusteringProcessorTest, WoiWidthBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, 0, 1, IrCameraImageHeight), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 0, 0, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, 0, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 0, IrCameraImageWidth + 1, IrCameraImageHeight), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, IrCameraImageWidth + 1, IrCameraImageHeight), "");
}

TEST_F(ClusteringProcessorTest, WoiHeightBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestWoiBoundary(0, 0, IrCameraImageWidth, 1), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 0, IrCameraImageWidth, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, IrCameraImageWidth, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(0, 0, IrCameraImageWidth, IrCameraImageHeight + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestWoiBoundary(1, 0, IrCameraImageWidth, IrCameraImageHeight + 1), "");
}

void ClusteringProcessorTest::DoTestObjectIntensityMinBoundary(int threshold) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        ClusteringProcessorConfig config;
        GetClusteringProcessorDefaultConfig(&config);

        config.objectIntensityMin = threshold;
        RunClusteringProcessor(handle, config);
        StopImageProcessor(handle);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TEST_F(ClusteringProcessorTest, ObjectIntensityMinBoundary)
{
    NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectIntensityMinBoundary(0), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectIntensityMinBoundary(IrCameraIntensityMax), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestObjectIntensityMinBoundary(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestObjectIntensityMinBoundary(IrCameraIntensityMax + 1), "");
}

void ClusteringProcessorTest::DoTestObjectPixelCountBoundary(int min, int max) NN_NOEXCEPT
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        ClusteringProcessorConfig config;
        GetClusteringProcessorDefaultConfig(&config);

        config.objectPixelCountMin = min;
        config.objectPixelCountMax = max;
        RunClusteringProcessor(handle, config);
        StopImageProcessor(handle);
    }

    NNT_IRSENSOR_EXIT_SUCCESS;
}

TEST_F(ClusteringProcessorTest, ObjectPixelCountBoundary)
{
    auto ObjectPixelCountMax = ClusteringProcessorObjectPixelCountMax;

    // 最小値と最大値との間に制限はない
    NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(0, 0), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(0, ObjectPixelCountMax), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(ObjectPixelCountMax, 0), NNT_IRSENSOR_EXIT_0, "");
    NNT_IRSENSOR_EXPECT_EXIT(DoTestObjectPixelCountBoundary(ObjectPixelCountMax, ObjectPixelCountMax), NNT_IRSENSOR_EXIT_0, "");

    EXPECT_DEATH_IF_SUPPORTED(DoTestObjectPixelCountBoundary(-1, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestObjectPixelCountBoundary(-1, ObjectPixelCountMax), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestObjectPixelCountBoundary(0, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestObjectPixelCountBoundary(0, ObjectPixelCountMax + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestObjectPixelCountBoundary(ObjectPixelCountMax, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(DoTestObjectPixelCountBoundary(ObjectPixelCountMax + 1, 0), "");
}

void ClusteringProcessorTest::DoTestClusteringData(const ClusteringData& object, const ClusteringProcessorConfig& config) NN_NOEXCEPT
{
    const auto& woi = config.windowOfInterest;

    EXPECT_LE(0, object.averageIntensity);
    EXPECT_LE(object.averageIntensity, IrCameraIntensityMax);

    EXPECT_LE(0, object.pixelCount);
    EXPECT_LE(object.pixelCount, woi.width * woi.height);

    EXPECT_LE(woi.x, object.bound.x);
    EXPECT_LT(object.bound.x, woi.x + woi.width);
    EXPECT_LE(woi.y, object.bound.y);
    EXPECT_LT(object.bound.y, woi.y + woi.height);

    EXPECT_LE(0, object.bound.width);
    EXPECT_LE(object.bound.x + object.bound.width, woi.x + woi.width);
    EXPECT_LE(0, object.bound.height);
    EXPECT_LE(object.bound.y + object.bound.height, woi.y + woi.height);

    EXPECT_LE(object.bound.x, object.centroid.x);
    EXPECT_LE(object.centroid.x, object.bound.x + object.bound.width - 1);
    EXPECT_LE(object.bound.y, object.centroid.y);
    EXPECT_LE(object.centroid.y, object.bound.y + object.bound.height - 1);
}

TEST_F(ClusteringProcessorTest, GetClusteringProcessorState)
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        ClusteringProcessorConfig config;
        GetClusteringProcessorDefaultConfig(&config);

        // ASSERT_EQ(IrCameraStatus_Available, GetIrCameraStatus(handle));
        RunClusteringProcessor(handle, config);

        ClusteringProcessorState state;
        // NNT_ASSERT_RESULT_SUCCESS(GetClusteringProcessorState(&state, handle));
        nn::Result result = nn::ResultSuccess();
        int counter = 0;
        while (NN_STATIC_CONDITION(true))
        {
            result = GetClusteringProcessorState(&state, handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
            {
                counter++;
                nn::os::SleepThread(PollingInterval);
                ASSERT_LT(counter, NotReadyWaitLoopCountMax);
            }
            else
            {
                break;
            }
        }
        EXPECT_LE(0, state.samplingNumber);

        for (auto j = 0; j < state.objectCount; ++j)
        {
            DoTestClusteringData(state.objects[j], config);
        }

        StopImageProcessor(handle);
    }
}

TEST_F(ClusteringProcessorTest, GetClusteringProcessorStates)
{
    EXPECT_LT(0, s_HandleCount);

    for (auto i = 0; i < s_HandleCount; ++i)
    {
        const auto& handle = s_Handles[i];

        ClusteringProcessorConfig config;
        GetClusteringProcessorDefaultConfig(&config);

        // ASSERT_EQ(IrCameraStatus_Available, GetIrCameraStatus(handle));
        RunClusteringProcessor(handle, config);

        ClusteringProcessorState states[ClusteringProcessorStateCountMax];
        int count;
        // NNT_ASSERT_RESULT_SUCCESS(GetClusteringProcessorStates(states, &count, nnt::irsensor::GetArrayLength(states), handle));
        nn::Result result = nn::ResultSuccess();
        int counter = 0;
        while (NN_STATIC_CONDITION(true))
        {
            result = GetClusteringProcessorStates(states, &count, nnt::irsensor::GetArrayLength(states), handle);
            if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
            {
                counter++;
                nn::os::SleepThread(PollingInterval);
                ASSERT_LT(counter, NotReadyWaitLoopCountMax);
            }
            else
            {
                break;
            }
        }
        EXPECT_LE(0, count);

        for (auto j = 0; j < count; ++j)
        {
            auto state = states[j];

            EXPECT_LE(0, state.samplingNumber);
            if (j > 0)
            {
                EXPECT_LT(states[j - 1].samplingNumber, state.samplingNumber);
            }

            for (auto k = 0; k < state.objectCount; ++k)
            {
                DoTestClusteringData(state.objects[k], config);
            }
        }

        StopImageProcessor(handle);
    }
}

} // namespace
