﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Thread.h>

#include <memory>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace sfutil {

class TestThread
{
    NN_DISALLOW_COPY(TestThread);
    NN_DISALLOW_MOVE(TestThread);
private:

    bool m_Initialized;
    nn::os::ThreadType m_Base;

    static void NoArgumentThraedFunction(void (*f)())
    {
        f();
    }

public:

    TestThread() NN_NOEXCEPT
        : m_Initialized(false)
    {
    }

    void Initialize(void (*f)(), void* stack, size_t stackSize, int priority) NN_NOEXCEPT
    {
        Initialize(NoArgumentThraedFunction, f, stack, stackSize, priority);
    }

    template <typename T>
    void Initialize(void (*f)(T*), T* p, void* stack, size_t stackSize, int priority) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!m_Initialized);
        auto vf = reinterpret_cast<nn::os::ThreadFunction>(f);
        auto vp = const_cast<void*>(reinterpret_cast<const volatile void*>(p));
        auto result = nn::os::CreateThread(&m_Base, vf, vp, stack, stackSize, priority);
        NN_ABORT_UNLESS(result.IsSuccess());
        this->m_Initialized = true;
    }

    template <typename T>
    TestThread(void (*f)(T*), T* p, void* stack, size_t stackSize, int priority) NN_NOEXCEPT
        : m_Initialized(false)
    {
        Initialize(f, p, stack, stackSize, priority);
    }

    ~TestThread() NN_NOEXCEPT
    {
        Finalize();
    }

    void Finalize() NN_NOEXCEPT
    {
        if (m_Initialized)
        {
            Wait();
            nn::os::DestroyThread(&m_Base);
            this->m_Initialized = false;
        }
    }

    void Start() NN_NOEXCEPT
    {
        nn::os::StartThread(&m_Base);
    }

    void Wait() NN_NOEXCEPT
    {
        nn::os::WaitThread(&m_Base);
    }

    nn::os::ThreadType* GetBase() NN_NOEXCEPT
    {
        return &m_Base;
    }

};

}}
