﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{DebugTextRenderer.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a singleton class that
 *  renders text to the screen. This class sets
 *  up the font textures, provides functionality
 *  to control size and placement of text, and sets
 *  up a command buffer to draw the text to a given
 *  render target.
 */

#pragma once

#include <cstdint>
#include <string>
#include <nvn\nvn.h>
#include <nn\nn_SdkAssert.h>
#include "MemoryPool.h"

const uint32_t g_MaxNumLines = 100;
const uint32_t g_MaxNumChars = 100;

    /* Max Buffer Size for vsnprintf_s. */
const size_t g_FontMaxBufferSize = 512;

class TextureIDManager;
class AssetFileLoadingHelper;
class AssetFileDataHolder;


class DebugTextRenderer
{
    NN_DISALLOW_COPY(DebugTextRenderer);

    public:
        ~DebugTextRenderer() ;

        void Init(NVNdevice* pDevice, NVNqueue* pQueue, TextureIDManager* pTextureIDManager);

        void CleanUp();

        void SetSpacing(bool proportional) ;
        void SetGridSize(float xGrid, float yGrid) ;
        void SetColor(float r, float g, float b, float a) ;
        void SetZValue(float zValue) ;
        void SetDrawEnable(bool enable) ;

        bool IsEnabled() ;

        void GetCharSize(float* pCharWidth, float* pCharHeight) ;

        void Puts(float column, float line, const char* pStr) ;
        void Printf(float column, float line, const char* pFmt, ...) ;

        void Draw(NVNtexture* pRenderTarget, int width, int height) ;

        static DebugTextRenderer* GetInstance()
        {
            static DebugTextRenderer instance;

            return &instance;
        }

    private:
        struct CharContainer
        {
            uint32_t m_Id;
            float    m_MinS;
            float    m_MinT;
            float    m_MaxS;
            float    m_MaxT;
            float    m_MinX;
            float    m_MinY;
            float    m_MaxX;
            float    m_MaxY;
            float    m_XAdvance;
        };

        struct FontBufferData
        {
            uint64_t   m_Address;
            NVNbuffer* m_pObject;
            void*      m_pBufferMap;
        };

        struct FontTextureData
        {
            NVNtextureHandle m_TextureHandle;
            NVNtexture*      m_pTexture;
            NVNsampler*      m_pSampler;
            FontBufferData   m_TextureBufferData;
        };

        struct FontData
        {
            CharContainer* m_pCharDataBuffer;
            float m_FontTextureWidth;
            float m_FontTextureHeight;
            uint32_t m_Channel;
            uint32_t m_NumCharData;
            float m_GridOffsetY;
            float m_GridOffsetX;
            uint8_t* m_pFontImageData;
            FontTextureData m_FontTextureData;
        };

        struct FontRenderData
        {
            float* m_pCharMagScale;
            float m_CharMagScaleF[2];
            float m_CharMagScaleP[2];
            float m_Depth;
            float m_Color[4];

        };

        DebugTextRenderer() ;

        void GetFontCharData(FontData* pData, const uint32_t* pFontHeader, const uint32_t* pCharData, const uint8_t* pFontImageData) ;
        void CreateBuffer(FontBufferData* pData, NVNmemoryPool* pPool, size_t size, ptrdiff_t offset);
        void InitFontTexture(FontData* pData, ptrdiff_t poolOffset);
        void UpdateScale(FontRenderData* pData, float scaleX, float scaleY) ;
        void UpdateTextureBuffer(FontTextureData* pTexture) ;
        void UpdateUniformBuffers(float* pMagScale, float* pColor, NVNtextureHandle* pTextureHandle) ;
        uint32_t BSearchIndex(FontData* pFont, uint32_t id) ;
        FontData* GetCurrentFont() ;
        void UpdateVertexBuffers(const char* pStr, uint32_t strLength, float x, float y) ;
        void CleanBufferData(FontBufferData& bufferData);

        NVNdevice*              m_pDevice;
        NVNqueue*               m_pQueue;

        MemoryPool*             m_pCommandMemoryPool;

        ptrdiff_t               m_CommandPoolOffset;
        void*                   m_pControlPool;
        NVNcommandBuffer        m_CommandBuffer;
        NVNcommandHandle        m_CommandHandle;
        NVNsync                 m_CommandBufferSync;

        NVNblendState           m_BlendState;
        NVNchannelMaskState     m_ChannelMaskState;
        NVNcolorState           m_ColorState;
        NVNdepthStencilState    m_DepthStencilState;
        NVNmultisampleState     m_MultisampleState;
        NVNpolygonState         m_PolygonState;

        NVNvertexAttribState    m_VertexAttributeStates[2];
        NVNvertexStreamState    m_VertexStreamStates[2];

        AssetFileLoadingHelper* m_pAssetLoader;
        AssetFileDataHolder*    m_pDataHolder;

        NVNtextureBuilder       m_TextureBuilder;
        NVNsamplerBuilder       m_SamplerBuilder;
        NVNbufferBuilder        m_BufferBuilder;

        MemoryPool*             m_pVertexDataPool;
        MemoryPool*             m_pUniformBufferPool;

        MemoryPool*             m_pFontTexturePool;
        MemoryPool*             m_pFontTextureBufferPool;

        TextureIDManager*       m_pTextureIDManager;


            /* P: Proportional, F: Fixed mono_space. */
        FontData m_FontP, m_FontF;

        uint32_t m_FontNumLines;

        uint32_t         m_StringSize[g_MaxNumLines];
        FontBufferData   m_FontPositionData[g_MaxNumLines];
        FontBufferData   m_FontTextureCoordinateData[g_MaxNumLines];
        float            m_FontMagScale[g_MaxNumLines][2];
        float            m_FontColor[g_MaxNumLines][4];
        FontBufferData   m_FontUniformDataVS[g_MaxNumLines];
        FontBufferData   m_FontUniformDataPSColor[g_MaxNumLines];
        FontBufferData   m_FontUniformDataPSSampler[g_MaxNumLines];
        FontTextureData* m_FontTexture[g_MaxNumLines];
        FontBufferData   m_FontIndexData;

        FontRenderData m_Font;
        bool m_Proportional;
        bool m_Enabled;

        CharContainer m_CharDataBufferP[g_MaxNumChars];
        CharContainer m_CharDataBufferF[g_MaxNumChars];

        int m_PositionLocation;
        int m_TextureCoordinateLocation;
};
